package ru.yandex.webmaster3.core.semantic.review_business.util;

import com.google.common.collect.Lists;
import org.json.*;
import org.slf4j.LoggerFactory;

import java.util.*;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;

/**
 * @author willzyx
 * @author dimas
 */
public class JsonAccessUtils {
    /**
     * Converts JSONArray that contains only String elements
     * to Collection of Strings
     *
     * @param array to be converted
     * @return Collection of Strings from given array.
     *         If array is null an empty collection will be returned.
     * @throws JSONException escalated from JSONArray access
     */
    public static List<String> arrayToStrings(final JSONArray array)
            throws JSONException {
        if (array == null) {
            return emptyList();
        }
        final List<String> result = Lists.newArrayList();
        for (int i = 0; i < array.length(); i++) {
            result.add(array.getString(i));
        }
        return result;
    }

    /**
     * Converts JSONArray that contains only {@link JSONObject}s
     * to Collection of {@link JSONObject}
     *
     * @param array to be converted
     * @return Collection of JSONObjects from given array.
     *         If array is null an empty collection will be returned.
     * @throws JSONException escalated from JSONArray access
     */
    public static List<JSONObject> arrayToJsons(final JSONArray array)
            throws JSONException {
        if (array == null) {
            return emptyList();
        }
        final List<JSONObject> result = Lists.newArrayList();
        for (int i = 0; i < array.length(); i++) {
            result.add(array.getJSONObject(i));
        }
        return result;
    }

    /**
     * Converts JSONArray that contains only {@link Long}s
     * to Collection of {@link Long}
     *
     * @param array to be converted
     * @return Collection of Long from given array.
     *         If array is null an empty collection will be returned.
     */
    public static Collection<Long> arrayToLongs(final JSONArray array) {
        if (array == null) {
            return emptySet();
        } else {
            try {
                Set<Long> out = new LinkedHashSet<Long>();
                for (int i = 0; i < array.length(); i++) {
                    out.add(array.getLong(i));
                }
                return out;
            } catch (JSONException e) {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * Obtain optional Float from JSONObject
     *
     * @param json JSONObject that contains desirable float value
     * @param key  of pair that contains desirable float value
     */
    public static Float optFloat(final JSONObject json, final String key) {
        final String floatAsStr = json.optString(key, null);
        if (floatAsStr == null)
            return null;
        try {
            return Float.parseFloat(floatAsStr);
        } catch (NumberFormatException e) {
            LoggerFactory.getLogger(JsonAccessUtils.class).error(
                    "Unable to parse float from {" + floatAsStr + "} at key {" + key + "}");
            return null;
        }
    }

    /**
     * Obtain optional Double from JSONObject
     *
     * @param json JSONObject that contains desirable double value
     * @param key  of pair that contains desirable double value
     */
    public static Double optDouble(final JSONObject json, final String key) {
        final String doubleAsStr = json.optString(key, null);
        if (doubleAsStr == null)
            return null;
        try {
            return Double.parseDouble(doubleAsStr);
        } catch (NumberFormatException e) {
            LoggerFactory.getLogger(JsonAccessUtils.class).error(
                    "Unable to parse double from {" + doubleAsStr + "} at key {" + key + "}");
            return null;
        }
    }

    /**
     * Obtain optional Integer from JSONObject
     *
     * @param json JSONObject that contains desirable integer value
     * @param key  of pair that contains desirable integer value
     */
    public static Integer optInt(final JSONObject json, final String key) {
        return json.has(key) ? json.optInt(key) : null;
    }

    /**
     * Obtain optional Long from JSONObject
     *
     * @param json JSONObject that contains desirable long value
     * @param key  of pair that contains desirable long value
     */
    public static Long optLong(final JSONObject json, final String key) {
        return json.has(key) ? json.optLong(key) : null;
    }

    /**
     * Obtain optional Boolean from JSONObject
     *
     * @param json JSONObject that contains desirable boolean value
     * @param key  of pair that contains desirable boolean value
     */
    public static Boolean optBoolean(final JSONObject json, final String key) {
        return json.has(key) ? json.optBoolean(key) : null;
    }
}
