package ru.yandex.webmaster3.core.semantic.semantic_document_parser.exceptions;

import org.jetbrains.annotations.NotNull;
import org.json.JSONException;
import org.json.JSONObject;
import ru.yandex.common.util.StringEscapeUtils;
import ru.yandex.common.util.StringUtils;
import ru.yandex.common.util.collections.Cf;
import ru.yandex.common.util.xml.Xmler;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.exceptions.MFException;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.exceptions.MFExceptions;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.tanker.LanguageContext;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static ru.yandex.common.util.xml.Xmler.*;

/**
 * Created by IntelliJ IDEA.
 * User: rasifiel
 * Date: 1/20/12
 * Time: 2:54 AM
 */
public class TankerVerifierExceptionSerializer {

    private final LanguageContext context;
    private static final String ALT_WORD = "alt_word";
    private static final String ERROR = "error";
    private static final String WARNING = "warning";
    private final String alt_word;
    private final String error;
    private final String warning;

    static final public String startBoldTag = "<b>";
    static final public String endBoldTag = "</b>";
    static final public String separator = ".";
    static final public String altSeparator = "\\|";


    public TankerVerifierExceptionSerializer(final LanguageContext context) {
        this.context = context;
        alt_word = context.getValue(ALT_WORD).getString();
        error = context.getValue(ERROR).getString();
        warning = context.getValue(WARNING).getString();
    }

    private String getMessage(final String key) {
        return context.getValue(key).getString();
    }

    private String getMessage(final VerifierException ex) {
        return getMessage(ex.getKey());
    }


    public List<Xmler.Tag> toXML(@NotNull final VerifierException ex) {
        if (ex instanceof MFExceptions) {
            return toXML((MFExceptions) ex);
        }
        final StringBuilder message = new StringBuilder();
        message.append((ex.getSeverity()) ? error : warning).append(" ");
        String resultMessage = getMessage(ex);
        final String fullMsg = ex.getMessage();

        final ArrayList<String> parts = new ArrayList<>();
        Pattern delimeter = Pattern.compile("(\\$\\$)[^\\$]");
        Matcher matcher = delimeter.matcher(fullMsg);
        int begin = 0;
        int end;

        int times = 0;
        while (matcher.find()) {
            end = matcher.start(1);
            parts.add(fullMsg.substring(begin, end));
            begin = matcher.end(1);
            times++;
        }

        if (times == 0) {
            parts.add(fullMsg);
        } else {
            parts.add(fullMsg.substring(begin, fullMsg.length()));
        }
        if (ex instanceof PrefixException) {
            message.append(getMessage(((PrefixException) ex).getPrefix())).append(" ");
        }
        if (ex.getRecurrentMessage() != null) {
            resultMessage =
                    substituteArgument("<rec>", resultMessage, getMessage(ex.getRecurrentMessage()), true, true);
        }
        {
            resultMessage = substituteArgument("%s", resultMessage, parts.get(0), ex.isEscapedMessage, false);
        }
        int k = parts.size();
        if (!resultMessage.contains("%" + parts.size())) {
            for (int i = parts.size() - 1; i >= 0; i--) {
                if (resultMessage.contains("%" + i)) {
                    k = i;
                    break;
                }
            }
        }
        int move = parts.size() - k;
        resultMessage = substituteArgument("%1", resultMessage, StringUtils.join(parts.subList(0, move + 1), "$$"), ex.isEscapedMessage, false);
        for (int i = 1; i < k; i++) {
            final String pattern = "%" + (i + 1);
            resultMessage = substituteArgument(pattern, resultMessage, parts.get(i + move), ex.isEscapedMessage, false);
        }
        message.append(resultMessage);
        return Cf.list(
                unescapedTag("error", attribute("card_hash", ex.getHash()).and(attribute("error_code", ex.getKey())),
                        message.toString()));

    }

    private List<Xmler.Tag> toXML(final MFExceptions exceptions) {
        final List<Xmler.Tag> result = new LinkedList<Xmler.Tag>();
        for (final MFException exception : exceptions.getExceptionsByTreeView()) {
            result.addAll(toXML(exception));
        }
        return result;
    }

    private String substituteArgument(final String pattern, String resultMessage, final String part, final boolean paramEscaped, final boolean noBold) {
        final String procMessage = part.replaceAll("\\|", alt_word);
        final StringBuilder msgTagBuilder = new StringBuilder();
        String name = noBold ? "span" : "b";
        if (paramEscaped) {
            msgTagBuilder.append("<").append(name).append(">")
                    .append(procMessage.isEmpty() ? "" : procMessage)
                    .append("</").append(name).append(">\n");
        } else {
            msgTagBuilder.append("<").append(name).append(">")
                    .append(StringEscapeUtils.escapeXml(procMessage.isEmpty() ? "" : procMessage))
                    .append("</").append(name).append(">\n");
        }
        if (msgTagBuilder.charAt(msgTagBuilder.length() - 1) == '\n') {
            msgTagBuilder.deleteCharAt(msgTagBuilder.length() - 1); //remove last \n
        }
        resultMessage = resultMessage.replaceAll(pattern,
                msgTagBuilder.toString().replaceAll("\\\\", "\\\\\\\\").replaceAll("\\$", "\\\\\\$"));
        return resultMessage;
    }


    public Collection<JSONObject> toJson(final VerifierException ex) throws JSONException {
        if (ex instanceof MFExceptions) {
            return toJson((MFExceptions) ex);
        }
        return exception2JSON(ex);
    }


    protected Collection<JSONObject> toJson(final MFExceptions exceptions) {
        final List<JSONObject> result = new LinkedList<JSONObject>();
        for (final MFException exception : exceptions.getExceptionsByTreeView()) {
            try {
                result.addAll(toJson(exception));
            } catch (JSONException e) {
                new RuntimeException(e);
            }
        }
        return result;
    }

    private final String YNDX_TYPE_ERROR = "yandex";
    private final String ERROR_TYPE_ERROR = "error";
    private final String WARNING_TYPE_ERROR = "warning";


    private Collection<JSONObject> exception2JSON(final VerifierException ex) throws JSONException {
        final StringBuilder message = new StringBuilder();
        message.append((ex.getSeverity()) ? error : warning).append(" ");
        String resultMessage = getMessage(ex);
        final String fullMsg = ex.getMessage();
        final ArrayList<String> parts = new ArrayList<>();
        Pattern delimeter = Pattern.compile("(\\$\\$)[^\\$]");
        Matcher matcher = delimeter.matcher(fullMsg);
        String error_type = ex.getSeverity() ? ERROR_TYPE_ERROR : WARNING_TYPE_ERROR;


        int begin = 0;
        int end;

        int times = 0;

        while (matcher.find()) {
            end = matcher.start(1);
            parts.add(fullMsg.substring(begin, end));
            begin = matcher.end(1);
            times++;
        }

        if (times == 0) {
            parts.add(fullMsg);
        } else {
            parts.add(fullMsg.substring(begin, fullMsg.length()));
        }
        if (ex instanceof PrefixException) {
            message.append(getMessage(((PrefixException) ex).getPrefix())).append(" ");
            error_type = YNDX_TYPE_ERROR;
        }

        if (ex.getRecurrentMessage() != null) {
            resultMessage =
                    substituteArgument("<rec>", resultMessage, getMessage(ex.getRecurrentMessage()), true, true);
        }
        {
            resultMessage = substituteArgument("%s", resultMessage, parts.get(0), ex.isEscapedMessage, false);
        }
        int k = parts.size();
        if (!resultMessage.contains("%" + parts.size())) {
            for (int i = parts.size() - 1; i >= 0; i--) {
                if (resultMessage.contains("%" + i)) {
                    k = i;
                    break;
                }
            }
        }
        int move = parts.size() - k;
        resultMessage = substituteArgument("%1", resultMessage, StringUtils.join(parts.subList(0, move + 1), "$$"), ex.isEscapedMessage, false);
        for (int i = 1; i < k; i++) {
            final String pattern = "%" + (i + 1);
            resultMessage = substituteArgument(pattern, resultMessage, parts.get(i + move), ex.isEscapedMessage, false);
        }
        message.append(resultMessage);
        JSONObject result = new JSONObject();
        result.put("#message", message.toString()).put("#error_code", ex.getKey()).put("#location", ex.getLocation()).put("#type", error_type);
        return Cf.list(result);
    }
}
