package ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.validators;

import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.util.Su;
import ru.yandex.common.util.URLUtils;
import ru.yandex.common.util.collections.Cf;
import ru.yandex.webmaster3.core.semantic.schema_org_information_extractor.SchemaClass;
import ru.yandex.webmaster3.core.semantic.schema_org_information_extractor.SchemaField;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.MicrodataUtils;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.data.ComplexMicrodata;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.data.Microdata;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.data.TextMicrodata;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.exceptions.*;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.validators.ISO8601Validator;

import java.util.*;
import java.util.regex.Pattern;

/**
 * Created by aleksart on 07.03.14.
 */

public class GoodRelationsValidator implements MicrodataValidator {

    private String vocab;

    @Required
    public void setVocab(final String vocab) {
        this.vocab = vocab;
    }

    private final Map<String, SchemaClass> classMaps;

    public GoodRelationsValidator(final Map<String, SchemaClass> classMap) {
        this.classMaps = classMap;
    }

    @Override
    public List<MicrodataValidatorException> validate(final Microdata microdata) {
        if (microdata instanceof ComplexMicrodata) {
            final ComplexMicrodata complexMicrodata = (ComplexMicrodata) microdata;
            final String hrefType = complexMicrodata.getType();
            if(hrefType != null){
                final String vocabHost = MicrodataUtils.getVocabHost(hrefType);
                String httpType = MicrodataUtils.addPrefixIfEmpty(hrefType);
                String vocabType = MicrodataUtils.addPrefixIfEmpty(vocab);
                List<String> vocabTypes = MicrodataUtils.protocolAliases(vocabType);
                boolean rightVocab = false;
                for(String vocab : vocabTypes){
                    if(httpType.startsWith(vocab)){
                        rightVocab = true;
                    }
                }
                if (rightVocab) {
                    final Map<String, SchemaClass> classMap = classMaps;
                    final List<MicrodataValidatorException> exceptions = new LinkedList<MicrodataValidatorException>();
                    final SchemaClass cls = classMap.get(hrefType);
                    final List<String> additionalTypes =
                            MicrodataUtils.extractStringList(complexMicrodata, "additionalType");
                    final List<SchemaClass> additionalClasses = new LinkedList<SchemaClass>();
                    for (final String addType : additionalTypes) {
                        if (classMaps.containsKey(addType)) {
                            final SchemaClass addCls = classMap.get(addType);
                            if (addCls != null) {
                                additionalClasses.add(addCls);
                            }
                        }
                    }
                    if (cls == null) {
                        final String fixedType = MicrodataUtils.addPrefixIfEmpty(hrefType);
                        boolean found = false;
                        for (final String klass : classMap.keySet()) {
                            if (fixedType.toLowerCase().equals(klass.toLowerCase())) {
                                exceptions.add(
                                        new SchemaMisspellMicrodataValidatorException("type", true, hrefType + "$$" + klass + "$$" + vocab,
                                                microdata));
                                found = true;
                            }
                        }
                        if (!found) {
                            exceptions.add(new NoSuchSchemaOrgClassMicrodataValidatorException(false, hrefType+"$$"+vocabHost, microdata));
                        }
                    } else {
                        additionalClasses.add(cls);
                    }
                    if (additionalClasses.isEmpty()) {
                        return exceptions;
                    }
                    validate(complexMicrodata, hrefType, classMap, exceptions, additionalClasses);
                    return exceptions;
                } else {
                    return Collections.emptyList();
                }
            }  else {
                return Collections.emptyList();
            }

        }else {
            return Collections.emptyList();
        }
    }

    private void validate(ComplexMicrodata complexMicrodata, String hrefType, Map<String, SchemaClass> classMap, List<MicrodataValidatorException> exceptions, List<SchemaClass> types) {
        for (final String fieldName : complexMicrodata.getPropList()) {
            final List<Microdata> values = complexMicrodata.getPropAsList(fieldName);
            final String vocabHost = MicrodataUtils.getVocabHost(hrefType);
            if ("itemId".equals(fieldName)) {
                continue;
            }
            boolean foundField = false;
            if (URLUtils.isValidHttpURL(fieldName)) {
                String fieldHost = MicrodataUtils.getVocabHost(fieldName);
                if(fieldHost.equals("schema.org") || fieldHost.equals("purl.org")){
                    SchemaClass curCls = classMap.get("AllFieldsClass");
                    if(curCls.fields.containsKey(fieldName)){
                        continue;
                    }
                    else{
                        exceptions.add(new NoSuchSchemaOrgFieldMicrodataValidatorException(false,
                                fieldName + "$$" + fieldHost, complexMicrodata));
                        continue;
                    }
                }
                else{
                    continue;
                }
            }

            final List<SchemaField> fields = new LinkedList<SchemaField>();
            for (final SchemaClass curCls : types) {
                //final String longFieldName = curCls.name.substring(0, curCls.name.length() - curCls.label.length()) + fieldName;

                String longFieldName = "http://purl.org/goodrelations/v1#" + fieldName;

                for(String longFieldNameAlias : MicrodataUtils.protocolAliases(longFieldName)) {

                    final SchemaField fld = curCls.fields.get(longFieldNameAlias);
                    if (fld == null) {
                        continue;
                    }
                    foundField = true;
                    fields.add(fld);
                }
            }
            if (!foundField) {
                SchemaClass foundScheme = null;
                String foundFieldName = null;
                for (final SchemaClass curCls : types) {
                    // field name to long
                    for (final String field : curCls.fields.keySet()) {
                        if (field.toLowerCase().equals(fieldName.toLowerCase())) {
                            foundScheme = curCls;
                            foundFieldName = field;
                            foundField = true;
                            break;
                        }
                    }
                    if (foundField) {
                        break;
                    }
                }
                if (foundField) {
                    exceptions.add(new SchemaMisspellMicrodataValidatorException("field", true,
                            fieldName + "$$" + foundScheme.name + "$$" + foundFieldName, complexMicrodata));
                } else {
                    exceptions.add(new NoSuchSchemaOrgFieldMicrodataValidatorException(false,
                            fieldName + "$$" + hrefType, complexMicrodata));
                }
            } else {
                for (final Microdata value : values) {
                    if (value instanceof TextMicrodata) {
                        if (onlyType(fields, "http://www.w3.org/2001/XMLSchema#dateTime")) {
                            if (!ISO8601Validator.validate(((TextMicrodata) value).data)) {
                                exceptions.add(new DatetimeFormatMicrodataValidatorException(false,
                                        ((TextMicrodata) value).data + "$$" + fieldName, complexMicrodata));
                            }
                        } else if (onlyType(fields, "http://www.w3.org/2001/XMLSchema#time")) {
                            if (!ISO8601Validator.validate(((TextMicrodata) value).data)) {
                                exceptions.add(new DatetimeFormatMicrodataValidatorException(false,
                                        ((TextMicrodata) value).data + "$$" + fieldName, complexMicrodata));
                            }
                        } else if (onlyType(fields, "http://www.w3.org/2001/XMLSchema#float")) {
                            if (!FLOAT_REGEXP.matcher(((TextMicrodata) value).data).matches()) {
                                exceptions.add(new SchemaTypeMicrodataValidatorException("float", true,
                                        ((TextMicrodata) value).data + "$$" + fieldName, complexMicrodata));
                            }
                        } else if (onlyType(fields, "http://www.w3.org/2001/XMLSchema#int")) {
                            if (!INTEGER_REGEXP.matcher(((TextMicrodata) value).data).matches()) {
                                exceptions.add(new SchemaTypeMicrodataValidatorException("integer", true,
                                        ((TextMicrodata) value).data + "$$" + fieldName, complexMicrodata));
                            }
                        } else if (onlyType(fields, "http://www.w3.org/2001/XMLSchema#boolean")) {
                            if (!"True".equals(((TextMicrodata) value).data) &&
                                    !"False".equals(((TextMicrodata) value).data)) {
                                exceptions.add(new SchemaTypeMicrodataValidatorException("boolean", true,
                                        ((TextMicrodata) value).data + "$$" + fieldName, complexMicrodata));
                            }
                        }
                        // text value ok for all fields
                    } else if (value instanceof ComplexMicrodata) {
                        boolean foundType = false;
                        ArrayList<String> validTypes = new ArrayList<String>();
                        for (final SchemaField fld : fields) {
                            if ((fld.ranges.size() == 1) && (fld.ranges.get(0).endsWith("#"))) {

                                String range = fld.ranges.get(0);

                                final SchemaClass schemaClass = classMap.get(range);
                                validate((ComplexMicrodata) value, "", classMap, exceptions, Cf.list(schemaClass));

                            }
                            final SchemaClass chkCls = classMap.get(((ComplexMicrodata) value).getType());
                            if (chkCls != null && checkClass(chkCls, fld)) {
                                foundType = true;
                            } else {
                                for (String rng : fld.ranges) {
                                    validTypes.add(rng);
                                }
                            }

                        }
                        if (!foundType) {
                            exceptions.add(new WrongClassForFieldMicrodataValidatorException(false,
                                    fieldName + "$$" + ((ComplexMicrodata) value).getType() + "$$" + Su.join(MicrodataUtils.httpPrefixOnly(validTypes), ", "), complexMicrodata));
                        }
                    }
                }
            }
        }
    }



    private final static Pattern FLOAT_REGEXP = Pattern.compile("^[-+]?[0-9]*\\.?[0-9]+([eE][-+]?[0-9]+)?$");
    private final static Pattern INTEGER_REGEXP = Pattern.compile("^[-+]?[0-9]+$");

    private boolean onlyType(final List<SchemaField> fields, final String type) {
        final Set<String> classNames = new HashSet<String>();
        for (final SchemaField field : fields) {
            classNames.addAll(field.ranges);
        }
        for(String addType: MicrodataUtils.protocolAliases(type)) {
            if (classNames.contains(addType)){
                return true;
            }
        }
        return false;
    }
    private boolean checkClass(final SchemaClass chkCls, final SchemaField fld) {
        for (final String range : fld.ranges) {
            if (range.equals(chkCls.name)) {
                return true;
            }
        }
        if (chkCls != null) {
            for (final SchemaClass parent : chkCls.subClassOf) {
                if (checkClass(parent, fld)) {
                    return true;
                }
            }
        }
        return false;
    }
}
