package ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.spec.instances;

import org.jetbrains.annotations.NotNull;
import org.w3c.dom.Node;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.data.MicroformatData;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.exceptions.EmptyMFException;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.exceptions.InvalidActionException;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.exceptions.MFExceptions;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.exceptions.PropertyNotDefinedMFException;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.spec.MFPropertyPlural;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.spec.MFPropertySingular;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.spec.Microformat;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.spec.property_types.DatetimeProperty;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.spec.property_types.TextProperty;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.spec.property_types.URIProperty;

public class HRecipe extends Microformat {
    final private static HRecipe instance = new HRecipe("hrecipe", true);

    /**
     * the name should be equal to the header of microformat (e. g. "vcard")
     *
     * @param name
     */
    protected HRecipe(@NotNull final String name, final boolean root) {
        super(name, root);
    }

    /**
     * Указываем список полей данного микроформата. Есть несколько видов свойств: MFPropertySingular - не более одного значения у поля,
     * MFPropertyPlural - множественные значения, MFPropertyConcatenated - множественные значения объединяемые в одно
     * У каждого поля задается его имя и далее список типов данных, которые могут встретиться в данном поле
     */
    static {
        try {
            instance.addProperty(new MFPropertySingular("fn", TextProperty.getInstance()));
            instance.addProperty(
                    new MFPropertyPlural("ingredient", Ingredient.getInstance(), TextProperty.getInstance()));
            instance.addProperty(new MFPropertySingular("yield", TextProperty.getInstance()));
            instance.addProperty(
                    new MFPropertySingular("instructions", Instructions.getInstance(), TextProperty.getInstance()));
            instance.addProperty(new MFPropertyPlural("duration", DatetimeProperty.getInstance()));
            instance.addProperty(new MFPropertyPlural("photo", URIProperty.getInstance()));
            instance.addProperty(new MFPropertySingular("result-photo", URIProperty.getInstance()));
            instance.addProperty(new MFPropertySingular("summary", TextProperty.getInstance()));
            instance.addProperty(
                    new MFPropertyPlural("nutrition", GoogleNutrition.getInstance(), Nutrition.getInstance()));
            instance.addProperty(new MFPropertySingular("cuisine-type", TextProperty.getInstance()));
            instance.addProperty(new MFPropertySingular("category", TextProperty.getInstance()));
            instance.addProperty(new MFPropertySingular("sub-category", TextProperty.getInstance()));
            instance.addProperty(new MFPropertySingular("weight", TextProperty.getInstance()));
            instance.addProperty(new MFPropertyPlural("author", HCard.getInstance(), TextProperty.getInstance()));
            instance.addProperty(new MFPropertySingular("published", DatetimeProperty.getInstance()));
            instance.addProperty(new MFPropertyPlural("review", HReviewAggregate.getInstance()));
            instance.addProperty(new MFPropertyPlural("tag", TextProperty.getInstance()));
        } catch (InvalidActionException e) {
            throw new RuntimeException(e);
        }
        instance.freeze();
    }

    public static HRecipe getInstance() {
        return instance;
    }

    /**
     * Постобработка и валидация данных микроформатного объекта
     *
     * @param data    - данные
     * @param content - текстовое содержимое разбираемой области
     * @param node    - фрагмент DOM дерева
     * @return true в случае, если данные были изменены, иначе false
     * @throws InvalidActionException
     * @throws EmptyMFException       - микроформатный объект пуст
     * @throws MFExceptions           - агрегатор других сообщений об ошибках
     */
    @Override
    public boolean postProcess(@NotNull final MicroformatData data, @NotNull final String content, final Node node) throws InvalidActionException, EmptyMFException, MFExceptions {
        if (data.isEmpty()) {
            throw new EmptyMFException(getName());
        }
        final MFExceptions resultEx = new MFExceptions(getName());
        if (data.isEmpty("fn")) {
            resultEx.put(new PropertyNotDefinedMFException("fn", true, data.getLocation()),"hRecipe_condition");
        }
        if (data.isEmpty("ingredient")) {
            resultEx.put(new PropertyNotDefinedMFException("ingredients", true, data.getLocation()),"hRecipe_condition");
        }
//        if (data.isEmpty("instructions")) {
//            resultEx.put(new PropertyNotDefinedMFException("instructions", true));
//        }
        if (!resultEx.isEmpty()) {
            throw resultEx;
        }
        return false;

    }
}
