package ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.transformer;

import org.htmlcleaner.*;
import org.w3c.dom.*;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * Created by aleksart on 28.07.15.
 * <p>
 * Copy of Dom Serilizer but with location in User Data
 */
public class LocatedDomSerializer {
    /**
     * The HTML Cleaner properties set by the user to control the HTML cleaning.
     */
    protected CleanerProperties props;
    public UserDataHandler LOCATION_DATA_HANDLER = null;

    /**
     * Whether XML entities should be escaped or not.
     */
    protected boolean escapeXml = true;

    /**
     * @param props     the HTML Cleaner properties set by the user to control the HTML cleaning.
     * @param escapeXml if true then escape XML entities
     */
    public LocatedDomSerializer(CleanerProperties props, boolean escapeXml) {
        this.props = props;
        this.escapeXml = escapeXml;
    }

    /**
     * @param props the HTML Cleaner properties set by the user to control the HTML cleaning.
     */
    public LocatedDomSerializer(CleanerProperties props) {
        this(props, true);
    }

    /**
     * @param rootNode the HTML Cleaner root node to serialize
     * @return the W3C Document object
     * @throws ParserConfigurationException if there's an error during serialization
     */
    public Document createDOM(TagNode rootNode) throws ParserConfigurationException {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        DOMImplementation impl = builder.getDOMImplementation();


        Document document;

        //
        // Where a DOCTYPE is supplied in the input, ensure that this is in the output DOM. See issue #27
        //
        if (rootNode.getDocType() != null) {
            String qualifiedName = rootNode.getDocType().getPart1();
            String publicId = rootNode.getDocType().getPublicId();
            String systemId = rootNode.getDocType().getSystemId();
            DocumentType documentType = impl.createDocumentType(qualifiedName, publicId, systemId);
            //
            // While the qualfiied name is "HTML" for some DocTypes, we want the actual document root name to be "html". See bug #116
            //
            if (qualifiedName.equals("HTML")) qualifiedName = "html";
            document = impl.createDocument(getNamespaceURIOnPath(rootNode,""), qualifiedName, documentType);
            document.setUserData("location", getForatedLocation(rootNode), null);
        } else {
            document = builder.newDocument();
            Element rootElement = document.createElement(rootNode.getName());
            rootElement.setUserData("location", getForatedLocation(rootNode), null);
            document.appendChild(rootElement);
        }
        Map<String, String> attributes = rootNode.getAttributes();
        Iterator<Map.Entry<String, String>> entryIterator = attributes.entrySet().iterator();
        while (entryIterator.hasNext()) {
            Map.Entry<String, String> entry = entryIterator.next();
            String attrName = entry.getKey();
            String attrValue = entry.getValue();
            if (escapeXml) {
                attrValue = Utils.escapeXml(attrValue, props, true);
            }

            document.getDocumentElement().setAttribute(attrName, attrValue);

            //
            // Flag the attribute as an ID attribute if appropriate. Thanks to Chris173
            //
            if (attrName.equalsIgnoreCase("id")) {
                document.getDocumentElement().setIdAttribute(attrName, true);
            }

        }
        document.setUserData("location", getForatedLocation(rootNode), LOCATION_DATA_HANDLER);

        createSubnodes(document, document.getDocumentElement(), rootNode.getAllChildren());

        return document;
    }

    String getNamespaceURIOnPath(TagNode root, String nsPrefix) {
        var nsDeclarations = root.getNamespaceDeclarations();
        if (nsDeclarations != null) {
            for (Map.Entry<String, String> nsEntry : nsDeclarations.entrySet()) {
                String currName = nsEntry.getKey();
                if (currName.equals(nsPrefix) || ("".equals(currName) && nsPrefix == null)) {
                    return nsEntry.getValue();
                }
            }
        }
        if (root.getParent() != null) {
            return getNamespaceURIOnPath(root.getParent(),nsPrefix);
        }

        return null;
    }

    /**
     * @param element the element to check
     * @return true if the passed element is a script or style element
     */
    protected boolean isScriptOrStyle(Element element) {
        String tagName = element.getNodeName();
        return "script".equalsIgnoreCase(tagName) || "style".equalsIgnoreCase(tagName);
    }

    /**
     * encapsulate content with <[CDATA[ ]]> for things like script and style elements
     *
     * @param element
     * @return true if <[CDATA[ ]]> should be used.
     */
    protected boolean dontEscape(Element element) {
        // make sure <script src=..></script> doesn't get turned into <script src=..><[CDATA[]]></script>
        // TODO check for blank content as well.
        return props.isUseCdataForScriptAndStyle() && isScriptOrStyle(element) && !element.hasChildNodes();
    }

    protected String outputCData(CData cdata) {
        return cdata.getContentWithoutStartAndEndTokens();
    }

    /**
     * Serialize a given HTML Cleaner node.
     *
     * @param document    the W3C Document to use for creating new DOM elements
     * @param element     the W3C element to which we'll add the subnodes to
     * @param tagChildren the HTML Cleaner nodes to serialize for that node
     */
    private void createSubnodes(Document document, Element element, List<? extends BaseToken> tagChildren) {

        if (tagChildren != null) {
            for (Object item : tagChildren) {

                if (item instanceof CommentNode) {
                    CommentNode commentNode = (CommentNode) item;
                    Comment comment = document.createComment(commentNode.getContent());
                    element.appendChild(comment);
                } else if (item instanceof CData) {
                    //
                    // Only include CData inside Script and Style tags
                    //
                    if (isScriptOrStyle(element)) {
                        element.appendChild(document.createCDATASection(outputCData((CData) item)));
                    }
                } else if (item instanceof ContentNode) {
                    ContentNode contentNode = (ContentNode) item;
                    String content = contentNode.getContent();

                    boolean specialCase = dontEscape(element);
                    if (escapeXml && !specialCase) {
                        content = Utils.escapeXml(content, props, true);
                    }
                    element.appendChild(specialCase ? document.createCDATASection(content) : document.createTextNode(content));
                } else if (item instanceof TagNode) {
                    TagNode subTagNode = (TagNode) item;
                    Element subelement = document.createElement(subTagNode.getName());
                    subelement.setUserData("location", getForatedLocation(subTagNode), LOCATION_DATA_HANDLER);
                    Map<String, String> attributes = subTagNode.getAttributes();
                    Iterator<Map.Entry<String, String>> entryIterator = attributes.entrySet().iterator();
                    while (entryIterator.hasNext()) {
                        Map.Entry<String, String> entry = entryIterator.next();
                        String attrName = entry.getKey();
                        String attrValue = entry.getValue();
                        if (escapeXml) {
                            attrValue = Utils.escapeXml(attrValue, props, true);
                        }

                        subelement.setAttribute(attrName, attrValue);

                        //
                        // Flag the attribute as an ID attribute if appropriate. Thanks to Chris173
                        //
                        if (attrName.equalsIgnoreCase("id")) {
                            subelement.setIdAttribute(attrName, true);
                        }

                    }

                    // recursively create subnodes
                    createSubnodes(document, subelement, subTagNode.getAllChildren());

                    element.appendChild(subelement);
                } else if (item instanceof List) {
                    List<? extends BaseToken> sublist = (List<? extends BaseToken>) item;
                    createSubnodes(document, element, sublist);
                }
            }
        }
    }

    private <T extends BaseToken> String getForatedLocation(T token) {
        return token.getRow() + ":" + token.getCol();
    }
}

