package ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.transformer;

import org.apache.log4j.Logger;
import org.htmlcleaner.*;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.Text;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSSerializer;
import ru.yandex.common.util.Su;
import ru.yandex.common.util.collections.Cf;
import ru.yandex.common.util.collections.CollectionFactory;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.exceptions.InvalidActionException;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.exceptions.MaxDepthMFException;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.spec.MFProperty;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.spec.MFType;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.spec.Microformat;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.spec.instances.MicroformatsManager;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.*;
import java.util.regex.Pattern;

import static ru.yandex.common.util.StringUtils.isEmpty;


public class TransofmerUtils {
    private static final Pattern MULTI_SPACE = Pattern.compile("\\s+");

    public static String[] splitBySpaces(final String s) {
        return MULTI_SPACE.split(s);
    }

    public static Element findById(final Node root, final String id) {
        if (root instanceof Element) {
            final Element rootAsElement = (Element) root;
            if (id.equals(rootAsElement.getAttribute("id"))) {
                return rootAsElement;
            }
            Node nd = root.getFirstChild();
            while (nd != null) {
                final Element result = findById(nd, id);
                if (result != null) {
                    return result;
                }
                nd = nd.getNextSibling();
            }
        }
        return null;
    }

    public static TagNode findById(final TagNode root, final String id) {

        final TagNode rootAsElement = root;
        if (id.equals(rootAsElement.getAttributeByName("id"))) {
            return rootAsElement;
        }
        TagNode[] nds = root.getChildTags();
        for(TagNode nd: nds){
            final TagNode result = findById(nd, id);
            if (result != null) {
                return result;
            }
        }
        return null;
    }

    public static List<Node> findByTagName(final Node root, final String tagName) {
        final LinkedList<Node> result = new LinkedList<Node>();
        if (root instanceof Element && tagName.equalsIgnoreCase(((Element) root).getTagName())) {
            result.add(root);
        }

        Node nd = root.getFirstChild();
        while (nd != null) {
            result.addAll(findByTagName(nd, tagName));
            nd = nd.getNextSibling();
        }
        return result;
    }

    public static List<TagNode> findByTagName(final TagNode root, final String tagName) {
        final LinkedList<TagNode> result = new LinkedList<>();
        if (tagName.equalsIgnoreCase(root.getName())) {
            result.add(root);
        }
        TagNode[] nds = root.getChildTags();
        for(TagNode nd : nds ) {
            result.addAll(findByTagName(nd, tagName));
        }
        return result;
    }


    public static List<Element> findByClass(final Node root, final String className, final boolean includeRoot) {
        final LinkedList<Element> result = new LinkedList<Element>();
        if (includeRoot) {
            if (root instanceof Element) {
                final String classAtt = ((Element) root).getAttribute("class");
                if (classAtt != null) {
                    for (final String classPart : splitBySpaces(classAtt)) {
                        if (className.equalsIgnoreCase(classPart)) {
                            result.add((Element) root);
                            break;
                        }
                    }
                }
            }
        }
        Node nd = root.getFirstChild();
        while (nd != null) {
            if (nd instanceof Element) {
                final String classAtt = ((Element) nd).getAttribute("class");
                if (classAtt != null) {
                    for (final String classPart : splitBySpaces(classAtt)) {
                        if (className.equalsIgnoreCase(classPart)) {
                            result.add((Element) nd);
                            break;
                        }
                    }
                }
                result.addAll(findByClass(nd, className, false));
            }
            nd = nd.getNextSibling();
        }
        return result;
    }

    public static List<TagNode> findByClass(final TagNode root, final String className, final boolean includeRoot) {
        final LinkedList<TagNode> result = new LinkedList<TagNode>();
        if (includeRoot) {
            final String classAtt = root.getAttributeByName("class");
            if (classAtt != null) {
                for (final String classPart : splitBySpaces(classAtt)) {
                    if (className.equalsIgnoreCase(classPart)) {
                        result.add(root);
                        break;
                    }
                }
            }
        }
       for(TagNode nd: root.getChildTags()){
            final String classAtt = nd.getAttributeByName("class");
            if (classAtt != null) {
                for (final String classPart : splitBySpaces(classAtt)) {
                    if (className.equalsIgnoreCase(classPart)) {
                        result.add(nd);
                        break;
                    }
                }
            }
            result.addAll(findByClass(nd, className, false));
        }
        return result;
    }

    public static List<TagNode> findByClassNotAnyIncluded(final TagNode root, final String className, final boolean includeRoot) {
        final LinkedList<TagNode> result = new LinkedList<>();
        if (includeRoot) {

            final String classAtt = root.getAttributeByName("class");
            if (classAtt != null) {
                for (final String classPart : splitBySpaces(classAtt)) {
                    if (className.equalsIgnoreCase(classPart)) {
                        result.add( root);
                        return result;
                    }
                }
            }

        }
        for(TagNode nd : root.getChildTags()){
            if (nd instanceof Element) {
                final String classAtt = nd.getAttributeByName("class");
                boolean found = false;
                if (classAtt != null) {
                    for (final String classPart : splitBySpaces(classAtt)) {
                        if (className.equalsIgnoreCase(classPart)) {
                            result.add(nd);
                            found = true;
                            break;
                        }
                    }
                }
                if (!found) {
                    result.addAll(findByClassNotAnyIncluded(nd, className, false));
                }
            }

        }
        return result;
    }


    public static List<Element> findByClassNotAnyIncluded(final Node root, final String className, final boolean includeRoot) {
        final LinkedList<Element> result = new LinkedList<Element>();
        if (includeRoot) {
            if (root instanceof Element) {
                final String classAtt = ((Element) root).getAttribute("class");
                if (classAtt != null) {
                    for (final String classPart : splitBySpaces(classAtt)) {
                        if (className.equalsIgnoreCase(classPart)) {
                            result.add((Element) root);
                            return result;
                        }
                    }
                }
            }
        }
        Node nd = root.getFirstChild();
        while (nd != null) {
            if (nd instanceof Element) {
                final String classAtt = ((Element) nd).getAttribute("class");
                boolean found = false;
                if (classAtt != null) {
                    for (final String classPart : splitBySpaces(classAtt)) {
                        if (className.equalsIgnoreCase(classPart)) {
                            result.add((Element) nd);
                            found = true;
                            break;
                        }
                    }
                }
                if (!found) {
                    result.addAll(findByClassNotAnyIncluded(nd, className, false));
                }
            }
            nd = nd.getNextSibling();
        }
        return result;
    }

    private static final int MAX_DEPTH = 10;

    public static List<TagNode> filterParent(final List<TagNode> input, final TagNode root, final Microformat parent, final MFProperty child, final DocumentProperties documentProperties) throws InvalidActionException, MaxDepthMFException {
        final List<TagNode> propFields = new LinkedList<>();
        propFields.addAll(input);
        final Iterator<TagNode> propFieldsIt = propFields.iterator();
        while (propFieldsIt.hasNext()) {
            final TagNode cur = propFieldsIt.next();
            if (cur == root) {
                continue;
            }
            TagNode chk = cur;
            boolean ok = true;
            final List<Microformat> oneLevelFormats = new ArrayList<Microformat>();
            final List<MFProperty> oneLevelProperties = new ArrayList<MFProperty>();
            int propertyDepth = 0;
            while (chk != null && !chk.equals(root)) {

                    final TagNode current = chk;
                    final List<Microformat> formats =
                            Tu.getRootMFsForClasses(current.getAttributeByName("class"), documentProperties.getManager());
                    final Set<Microformat> microformats = new HashSet<Microformat>();
                    microformats.add(parent);
                    TagNode p = chk.getParent();
                    int microformatsDepth = 0;
                    while (p != null && !p.equals(root)) {

                        final TagNode cp =  p;
                        final List<Microformat> found =
                                Tu.getRootMFsForClasses(cp.getAttributeByName("class"), documentProperties.getManager());
                        if (!found.isEmpty()) {
                            ++microformatsDepth;
                            microformats.addAll(found);
                            if (microformatsDepth > MAX_DEPTH) {
                                throw new MaxDepthMFException(child.getName());
                            }
                        }

                        p = p.getParent();
                    }
                    if (p != null) {
                        final TagNode cp =  p;
                        microformats.addAll(
                                Tu.getRootMFsForClasses(cp.getAttributeByName("class"), documentProperties.getManager()));
                    }
                    final List<MFProperty> props = Tu.getPropsForClasses(current.getAttributeByName("class"), microformats);
                    if (!props.isEmpty() && ++propertyDepth > MAX_DEPTH) {
                        throw new MaxDepthMFException(child.getName());
                    }
                    if (chk == cur) {
                        formats.removeAll(Tu.getRootMFsForClasses(child.getName(), documentProperties.getManager()));
                        props.removeAll(Tu.getPropsForClasses(child.getName(), microformats));
                        for (final Microformat format : formats) {
                            if (Tu.mayContain(format, child.getName())) {
                                if (Tu.mayContain(child, format, current) || Tu.mayContain(child, format.getName())) {
                                    oneLevelFormats.add(format);
                                } else {
                                    ok = false;
                                    break;
                                }
                            }
                        }
                        if (!ok) {
                            break;
                        }
                        for (final MFProperty property : props) {
                            if (Tu.mayContain(property, child.getName())) {
                                if (Tu.mayContain(child, property.getName())) {
                                    oneLevelProperties.add(property);
                                } else {
                                    ok = false;
                                    break;
                                }
                            }
                        }
                        if (!ok) {
                            break;
                        }
                    } else {
                        for (final Microformat format : formats) {
                            if (Tu.mayContain(format, child.getName())) {
                                ok = false;
                                break;
                            }
                            for (final Microformat anotherFormat : oneLevelFormats) {
                                if (Tu.mayContain(format, anotherFormat.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                            for (final MFProperty anotherProp : oneLevelProperties) {
                                if (Tu.mayContain(format, anotherProp.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                        }
                        if (!ok) {
                            break;
                        }
                        for (final MFProperty property : props) {
                            if (Tu.mayContain(property, child.getName())) {
                                ok = false;
                                break;
                            }
                            for (final Microformat anotherFormat : oneLevelFormats) {
                                if (Tu.mayContain(property, anotherFormat, cur) ||
                                        Tu.mayContain(property, anotherFormat.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                            for (final MFProperty anotherProp : oneLevelProperties) {
                                if (Tu.mayContain(property, anotherProp.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                        }
                        if (!ok) {
                            break;
                        }
                    }

                chk = chk.getParent();
            }
            if (!ok) {
                propFieldsIt.remove();
            }
        }
        return propFields;
    }

    public static List<TagNode> filterParent(final List<TagNode> input, final TagNode root, final MFProperty parent, final Microformat child, final DocumentProperties documentProperties) throws InvalidActionException {
        final List<TagNode> propFields = new LinkedList<>();
        propFields.addAll(input);
        final Iterator<TagNode> propFieldsIt = propFields.iterator();
        while (propFieldsIt.hasNext()) {
            final TagNode cur = propFieldsIt.next();
            if (cur == root) {
                continue;
            }
            TagNode chk = cur;
            boolean ok = true;
            final List<Microformat> oneLevelFormats = new ArrayList<Microformat>();
            final List<MFProperty> oneLevelProperties = new ArrayList<MFProperty>();
            while (chk != null && !chk.equals(root)) {
                    final TagNode current =  chk;
                    final List<Microformat> formats =
                            Tu.getRootMFsForClasses(current.getAttributeByName("class"), documentProperties.getManager());
                    final Set<Microformat> microformats = new HashSet<Microformat>();
                    TagNode p = chk.getParent();
                    while (p != null && !p.equals(root)) {

                        final TagNode cp = p;
                        microformats.addAll(
                                Tu.getRootMFsForClasses(cp.getAttributeByName("class"), documentProperties.getManager()));

                        p = p.getParent();
                    }
                    if (p != null) {
                        final TagNode cp =  p;
                        microformats.addAll(
                                Tu.getRootMFsForClasses(cp.getAttributeByName("class"), documentProperties.getManager()));
                    }
                    final List<MFProperty> props = Tu.getPropsForClasses(current.getAttributeByName("class"), microformats);
                    if (chk == cur) {
                        formats.removeAll(Tu.getRootMFsForClasses(child.getName(), documentProperties.getManager()));
                        props.removeAll(Tu.getPropsForClasses(child.getName(), microformats));
                        for (final Microformat format : formats) {
                            if (Tu.mayContain(format, child.getName())) {
                                if (Tu.mayContain(child, format.getName())) {
                                    oneLevelFormats.add(format);
                                } else {
                                    ok = false;
                                    break;
                                }
                            }
                        }
                        if (!ok) {
                            break;
                        }
                        for (final MFProperty property : props) {
                            if (Tu.mayContain(property, child.getName())) {
                                if (Tu.mayContain(child, property.getName())) {
                                    oneLevelProperties.add(property);
                                } else {
                                    ok = false;
                                    break;
                                }
                            }
                        }
                        if (!ok) {
                            break;
                        }
                    } else {
                        for (final Microformat format : formats) {
                            if (Tu.mayContain(format, child.getName())) {
                                ok = false;
                                break;
                            }
                            for (final Microformat anotherFormat : oneLevelFormats) {
                                if (Tu.mayContain(format, anotherFormat.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                            for (final MFProperty anotherProp : oneLevelProperties) {
                                if (Tu.mayContain(format, anotherProp.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                        }
                        if (!ok) {
                            break;
                        }
                        for (final MFProperty property : props) {
                            if (Tu.mayContain(property, child.getName())) {
                                ok = false;
                                break;
                            }
                            for (final Microformat anotherFormat : oneLevelFormats) {
                                if (Tu.mayContain(property, anotherFormat, cur) ||
                                        Tu.mayContain(property, anotherFormat.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                            for (final MFProperty anotherProp : oneLevelProperties) {
                                if (Tu.mayContain(property, anotherProp.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                        }
                        if (!ok) {
                            break;
                        }
                    }

                chk = chk.getParent();
            }
            if (!ok) {
                propFieldsIt.remove();
            }
        }
        return propFields;
    }

    public static List<Element> filterParent(final List<Element> input, final Node root, final Microformat parent, final MFProperty child, final DocumentProperties documentProperties) throws InvalidActionException, MaxDepthMFException {
        final List<Element> propFields = new LinkedList<Element>();
        propFields.addAll(input);
        final Iterator<Element> propFieldsIt = propFields.iterator();
        while (propFieldsIt.hasNext()) {
            final Element cur = propFieldsIt.next();
            if (cur == root) {
                continue;
            }
            Node chk = cur;
            boolean ok = true;
            final List<Microformat> oneLevelFormats = new ArrayList<Microformat>();
            final List<MFProperty> oneLevelProperties = new ArrayList<MFProperty>();
            int propertyDepth = 0;
            while (chk != null && !chk.equals(root)) {
                if (chk instanceof Element) {
                    final Element current = (Element) chk;
                    final List<Microformat> formats =
                            Tu.getRootMFsForClasses(current.getAttribute("class"), documentProperties.getManager());
                    final Set<Microformat> microformats = new HashSet<Microformat>();
                    microformats.add(parent);
                    Node p = chk.getParentNode();
                    int microformatsDepth = 0;
                    while (p != null && !p.equals(root)) {
                        if (p instanceof Element) {
                            final Element cp = (Element) p;
                            final List<Microformat> found =
                                    Tu.getRootMFsForClasses(cp.getAttribute("class"), documentProperties.getManager());
                            if (!found.isEmpty()) {
                                ++microformatsDepth;
                                microformats.addAll(found);
                                if (microformatsDepth > MAX_DEPTH) {
                                    throw new MaxDepthMFException(child.getName());
                                }
                            }
                        }
                        p = p.getParentNode();
                    }
                    if (p != null && p instanceof Element) {
                        final Element cp = (Element) p;
                        microformats.addAll(
                                Tu.getRootMFsForClasses(cp.getAttribute("class"), documentProperties.getManager()));
                    }
                    final List<MFProperty> props = Tu.getPropsForClasses(current.getAttribute("class"), microformats);
                    if (!props.isEmpty() && ++propertyDepth > MAX_DEPTH) {
                        throw new MaxDepthMFException(child.getName());
                    }
                    if (chk == cur) {
                        formats.removeAll(Tu.getRootMFsForClasses(child.getName(), documentProperties.getManager()));
                        props.removeAll(Tu.getPropsForClasses(child.getName(), microformats));
                        for (final Microformat format : formats) {
                            if (Tu.mayContain(format, child.getName())) {
                                if (Tu.mayContain(child, format, current) || Tu.mayContain(child, format.getName())) {
                                    oneLevelFormats.add(format);
                                } else {
                                    ok = false;
                                    break;
                                }
                            }
                        }
                        if (!ok) {
                            break;
                        }
                        for (final MFProperty property : props) {
                            if (Tu.mayContain(property, child.getName())) {
                                if (Tu.mayContain(child, property.getName())) {
                                    oneLevelProperties.add(property);
                                } else {
                                    ok = false;
                                    break;
                                }
                            }
                        }
                        if (!ok) {
                            break;
                        }
                    } else {
                        for (final Microformat format : formats) {
                            if (Tu.mayContain(format, child.getName())) {
                                ok = false;
                                break;
                            }
                            for (final Microformat anotherFormat : oneLevelFormats) {
                                if (Tu.mayContain(format, anotherFormat.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                            for (final MFProperty anotherProp : oneLevelProperties) {
                                if (Tu.mayContain(format, anotherProp.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                        }
                        if (!ok) {
                            break;
                        }
                        for (final MFProperty property : props) {
                            if (Tu.mayContain(property, child.getName())) {
                                ok = false;
                                break;
                            }
                            for (final Microformat anotherFormat : oneLevelFormats) {
                                if (Tu.mayContain(property, anotherFormat, cur) ||
                                        Tu.mayContain(property, anotherFormat.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                            for (final MFProperty anotherProp : oneLevelProperties) {
                                if (Tu.mayContain(property, anotherProp.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                        }
                        if (!ok) {
                            break;
                        }
                    }
                }
                chk = chk.getParentNode();
            }
            if (!ok) {
                propFieldsIt.remove();
            }
        }
        return propFields;
    }

    public static List<Element> filterParent(final List<Element> input, final Node root, final MFProperty parent, final Microformat child, final DocumentProperties documentProperties) throws InvalidActionException {
        final List<Element> propFields = new LinkedList<Element>();
        propFields.addAll(input);
        final Iterator<Element> propFieldsIt = propFields.iterator();
        while (propFieldsIt.hasNext()) {
            final Element cur = propFieldsIt.next();
            if (cur == root) {
                continue;
            }
            Node chk = cur;
            boolean ok = true;
            final List<Microformat> oneLevelFormats = new ArrayList<Microformat>();
            final List<MFProperty> oneLevelProperties = new ArrayList<MFProperty>();
            while (chk != null && !chk.equals(root)) {
                if (chk instanceof Element) {
                    final Element current = (Element) chk;
                    final List<Microformat> formats =
                            Tu.getRootMFsForClasses(current.getAttribute("class"), documentProperties.getManager());
                    final Set<Microformat> microformats = new HashSet<Microformat>();
                    Node p = chk.getParentNode();
                    while (p != null && !p.equals(root)) {
                        if (p instanceof Element) {
                            final Element cp = (Element) p;
                            microformats.addAll(
                                    Tu.getRootMFsForClasses(cp.getAttribute("class"), documentProperties.getManager()));
                        }
                        p = p.getParentNode();
                    }
                    if (p != null && p instanceof Element) {
                        final Element cp = (Element) p;
                        microformats.addAll(
                                Tu.getRootMFsForClasses(cp.getAttribute("class"), documentProperties.getManager()));
                    }
                    final List<MFProperty> props = Tu.getPropsForClasses(current.getAttribute("class"), microformats);
                    if (chk == cur) {
                        formats.removeAll(Tu.getRootMFsForClasses(child.getName(), documentProperties.getManager()));
                        props.removeAll(Tu.getPropsForClasses(child.getName(), microformats));
                        for (final Microformat format : formats) {
                            if (Tu.mayContain(format, child.getName())) {
                                if (Tu.mayContain(child, format.getName())) {
                                    oneLevelFormats.add(format);
                                } else {
                                    ok = false;
                                    break;
                                }
                            }
                        }
                        if (!ok) {
                            break;
                        }
                        for (final MFProperty property : props) {
                            if (Tu.mayContain(property, child.getName())) {
                                if (Tu.mayContain(child, property.getName())) {
                                    oneLevelProperties.add(property);
                                } else {
                                    ok = false;
                                    break;
                                }
                            }
                        }
                        if (!ok) {
                            break;
                        }
                    } else {
                        for (final Microformat format : formats) {
                            if (Tu.mayContain(format, child.getName())) {
                                ok = false;
                                break;
                            }
                            for (final Microformat anotherFormat : oneLevelFormats) {
                                if (Tu.mayContain(format, anotherFormat.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                            for (final MFProperty anotherProp : oneLevelProperties) {
                                if (Tu.mayContain(format, anotherProp.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                        }
                        if (!ok) {
                            break;
                        }
                        for (final MFProperty property : props) {
                            if (Tu.mayContain(property, child.getName())) {
                                ok = false;
                                break;
                            }
                            for (final Microformat anotherFormat : oneLevelFormats) {
                                if (Tu.mayContain(property, anotherFormat, cur) ||
                                        Tu.mayContain(property, anotherFormat.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                            for (final MFProperty anotherProp : oneLevelProperties) {
                                if (Tu.mayContain(property, anotherProp.getName())) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (!ok) {
                                break;
                            }
                        }
                        if (!ok) {
                            break;
                        }
                    }
                }
                chk = chk.getParentNode();
            }
            if (!ok) {
                propFieldsIt.remove();
            }
        }
        return propFields;
    }

    public static List<TagNode> findByAnyClass(final TagNode root, final List<String> classesName, final boolean includeRoot) {
        final LinkedList<TagNode> result = new LinkedList<>();
        if (includeRoot) {
            final String classAtt = root.getAttributeByName("class");
            if (classAtt != null) {
                for (final String classPart : splitBySpaces(classAtt)) {
                    if (classesName.contains(classPart)) {
                        result.add(root);
                        break;
                    }
                }
            }

        }
        for(TagNode nd: root.getChildTags()){

                final String classAtt = nd.getAttributeByName("class");
                if (classAtt != null) {
                    for (final String classPart : splitBySpaces(classAtt)) {
                        if (classesName.contains(classPart)) {
                            result.add(nd);
                            break;
                        }
                    }
                }
                result.addAll(findByAnyClass(nd, classesName, false));

        }
        return result;
    }

    public static List<Element> findByAnyClass(final Node root, final List<String> classesName, final boolean includeRoot) {
        final LinkedList<Element> result = new LinkedList<Element>();
        if (includeRoot) {
            if (root instanceof Element) {
                final String classAtt = ((Element) root).getAttribute("class");
                if (classAtt != null) {
                    for (final String classPart : splitBySpaces(classAtt)) {
                        if (classesName.contains(classPart)) {
                            result.add((Element) root);
                            break;
                        }
                    }
                }
            }
        }
        Node nd = root.getFirstChild();
        while (nd != null) {
            if (nd instanceof Element) {
                final String classAtt = ((Element) nd).getAttribute("class");
                if (classAtt != null) {
                    for (final String classPart : splitBySpaces(classAtt)) {
                        if (classesName.contains(classPart)) {
                            result.add((Element) nd);
                            break;
                        }
                    }
                }
                result.addAll(findByAnyClass(nd, classesName, false));
            }
            nd = nd.getNextSibling();
        }
        return result;
    }

    private final static List<String> VALUE_TITLES = Cf.list("value-title");

    public static String stringifyNode(final TagNode root, final boolean valueExerpt, final boolean abbrExerpt, final boolean ignoreRemoved) {
        final TagNode rootAsElement = root;
        if (valueExerpt && rootAsElement != null) {
            final String classValue = rootAsElement.getAttributeByName("class");
            if (containsAnyClass(classValue,VALUE_TITLES)) {
                final String titleValue = rootAsElement.getAttributeByName("title");
                if (titleValue != null) {
                    return titleValue;
                }
            }
        }
        if (abbrExerpt && rootAsElement != null && "abbr".equalsIgnoreCase(rootAsElement.getName())) {
            final String titleValue = rootAsElement.getAttributeByName("title");
            if (!isEmpty(titleValue)) {
                return titleValue;
            }
        }
        if (valueExerpt) {
            final StringBuilder sb = new StringBuilder();

            for(TagNode child : root.getChildTags()){
                final String classValue = ((Element) child).getAttribute("class");
                if ("value-title".equals(classValue)) {
                    final String titleValue = ((Element) child).getAttribute("title");
                    if (titleValue != null) {
                        return titleValue;
                    }
                }
                if ("value".equals(classValue)) {
                    sb.append(stringifyNode(child, false, abbrExerpt, ignoreRemoved).trim());
                }
            }

            if (sb.length() != 0) {
                return sb.toString();
            }
        }
        return _stringifyNode(root, ignoreRemoved, true);
    }
    public static String stringifyNode(final Node root, final boolean valueExerpt, final boolean abbrExerpt, final boolean ignoreRemoved) {
        final Element rootAsElement = (root instanceof Element) ? (Element) root : null;
        if (valueExerpt && rootAsElement != null) {
            final String classValue = rootAsElement.getAttribute("class");
            if (containsAnyClass(classValue,VALUE_TITLES)) {
                final String titleValue = rootAsElement.getAttribute("title");
                if (titleValue != null) {
                    return titleValue;
                }
            }
        }
        if (abbrExerpt && rootAsElement != null && "abbr".equalsIgnoreCase(rootAsElement.getTagName())) {
            final String titleValue = rootAsElement.getAttribute("title");
            if (!isEmpty(titleValue)) {
                return titleValue;
            }
        }
        if (valueExerpt) {
            final StringBuilder sb = new StringBuilder();
            Node child = root.getFirstChild();
            while (child != null) {
                if (child instanceof Element) {
                    final String classValue = ((Element) child).getAttribute("class");
                    if ("value-title".equals(classValue)) {
                        final String titleValue = ((Element) child).getAttribute("title");
                        if (titleValue != null) {
                            return titleValue;
                        }
                    }
                    if ("value".equals(classValue)) {
                        sb.append(stringifyNode(child, false, abbrExerpt, ignoreRemoved).trim());
                    }
                }
                child = child.getNextSibling();
            }
            if (sb.length() != 0) {
                return sb.toString();
            }
        }
        return _stringifyNode(root, ignoreRemoved, true);
    }

    public static String _stringifyNode(final Node root, final boolean ignoreRemoved, final boolean firstLevel) {
        if (root.getUserData("stop") != null && ignoreRemoved && !firstLevel) {
            return "";
        }
        if (root instanceof Text) {
            return ((Text) root).getWholeText();
        }
        final StringBuilder sb = new StringBuilder();
        if (root instanceof Element) {
            final Element rootAsElement = (Element) root;
            if ("img".equalsIgnoreCase(rootAsElement.getTagName())) {
                final String text = rootAsElement.getAttribute("alt");
                return text == null ? "" : text;
            }
            if ("input".equalsIgnoreCase(rootAsElement.getTagName())) {
                final String text = rootAsElement.getAttribute("value");
                return text == null ? "" : text;
            }
            if ("br".equalsIgnoreCase(rootAsElement.getTagName())) {
                return "\n";
            }
            if ("del".equalsIgnoreCase(rootAsElement.getTagName())) {
                return "";
            }
            if ("li".equalsIgnoreCase(rootAsElement.getTagName())) {
                sb.append("\n");
            }
        }
        Node child = root.getFirstChild();
        while (child != null) {
            sb.append(_stringifyNode(child, false, false));
            child = child.getNextSibling();
        }
        return sb.toString();
    }


    public static String _stringifyNode(final TagNode root, final boolean ignoreRemoved, final boolean firstLevel) {
        if (root.hasAttribute(ParseTagNodeContext.INCREADIBLE_STOP_ATTRIBUTE) && ignoreRemoved && !firstLevel) {
            return "";
        }
        final StringBuilder sb = new StringBuilder();
        if(root.getChildTags().length ==0){
            return String.valueOf(root.getText());
        }

        else{

            if ("img".equalsIgnoreCase(root.getName())) {
                final String text = root.getAttributeByName("alt");
                return text == null ? "" : text;
            }
            if ("input".equalsIgnoreCase(root.getName())) {
                final String text = root.getAttributeByName("value");
                return text == null ? "" : text;
            }
            if ("br".equalsIgnoreCase(root.getName())) {
                return "\n";
            }
            if ("del".equalsIgnoreCase(root.getName())) {
                return "";
            }
            if ("li".equalsIgnoreCase(root.getName())) {
                sb.append("\n");
            }
        }
        for(TagNode nd: root.getChildTags()){
            sb.append(_stringifyNode(nd,false,false));
        }
        return sb.toString();
    }

    private static LSSerializer getSerializer() {
        LSSerializer serializer = null;
        try {
            serializer =
                    ((DOMImplementationLS) DocumentBuilderFactory.newInstance().newDocumentBuilder().getDOMImplementation()).createLSSerializer();
            serializer.getDomConfig().setParameter("format-pretty-print", true);
            serializer.getDomConfig().setParameter("xml-declaration", false);
        } catch (ParserConfigurationException e) {
            Logger.getLogger(Tu.class).fatal("LSSerializer can't be created: " + e.getMessage());
        }
        return serializer;
    }

    public static String extractHtmlContent(final Node root) {
        final Node data = root.cloneNode(true);
        eraseAllClassesInside(data);
        final StringBuilder sb = new StringBuilder();
        Node child = data.getFirstChild();
        while (child != null) {
            sb.append(getSerializer().writeToString(child));
            child = child.getNextSibling();
        }
        return sb.toString();
    }

    public static String extractHtmlContent(final TagNode root) {
        final TagNode data = root.makeCopy();
        eraseAllClassesInside(data);
        final StringBuilder sb = new StringBuilder();
        final Writer writer = new StringWriter();
        final CleanerProperties properties = new CleanerProperties();
        final Serializer serializer = new SimpleHtmlSerializer(properties);
        for(BaseToken child : data.getAllChildren()) {
            try {
                child.serialize(serializer,writer);
            } catch (IOException e) {
                e.printStackTrace();
            }
            sb.append(writer.toString());
        }


        return sb.toString();
    }

    private static void eraseAllClassesInside(final Node root) {
        if (root instanceof Element) {
            ((Element) root).removeAttribute("class");
        }
        Node nd = root.getFirstChild();
        while (nd != null) {
            eraseAllClassesInside(nd);
            nd = nd.getNextSibling();
        }
    }

    private static void eraseAllClassesInside(final TagNode root) {

        root.removeAttribute("class");

        for(TagNode nd : root.getChildTags()){
            eraseAllClassesInside(nd);
        }
    }

    public static String serializeDoc(final Node doc) {
        return getSerializer().writeToString(doc);
    }

    private static final String[] compoundMicroformats =
            {"mfo", "vcard", "adr", "geo", "hreview", "xfolkentry", "hresume", "biota", "vcalendar", "vevent", "vtodo",
                    "valarm", "vfreebusy", "hfeed", "hentry", "hslice", "haudio", "hmeasure", "hangle", "hmoney",
                    "hlisting", "figure", "hproduct", "hmedia"};

    public static void destroyer(final Node root, final List<String> saveFormats) {
        final List<String> toDelete = new LinkedList<String>();
        for (final String mf : compoundMicroformats) {
            if (!saveFormats.contains(mf)) {
                toDelete.add(mf);
            }
        }
        destroyElements(root, toDelete);
    }

    public static void destroyElements(final Node root, final List<String> toDelete) {
        Node child = root.getFirstChild();
        while (child != null) {
            if (child instanceof Element) {
                final Element childAsElement = (Element) child;
                final String classAtt = childAsElement.getAttribute("class");
                if (classAtt != null && containsAnyClass(classAtt, toDelete)) {
                    destroyElement(child);
                    final StringBuilder sb = new StringBuilder();
                    for (final String part : splitBySpaces(classAtt)) {
                        if (!toDelete.contains(part)) {
                            sb.append(part).append(" ");
                        }
                    }
                    childAsElement.setAttribute("class", sb.toString());
                } else {
                    destroyElements(child, toDelete);
                }
            }
            child = child.getNextSibling();
        }
    }

    public static void destroyElement(final Node root) {
        Node child = root.getFirstChild();
        while (child != null) {
            if (child instanceof Element) {
                final Element childAsElement = (Element) child;
                childAsElement.setAttribute("class", "");
                childAsElement.setAttribute("rel", "");
                childAsElement.setAttribute("rev", "");
                destroyElement(childAsElement);
            }
            child = child.getNextSibling();
        }
    }

    private final static List<String> VALUES_CLASSES = Cf.list("value","value-title");

    public static List<Element> getValueChildren(final Node root) {
        return findByAnyClass(root,VALUES_CLASSES,false);
    }

    public static List<TagNode> getValueChildren(final TagNode root) {
        return findByAnyClass(root,VALUES_CLASSES,false);
    }

    public static List<Node> getFirstNodesWithClass(final Node root, final List<String> classNames) {
        final LinkedList<Node> result = new LinkedList<Node>();
        Node child = root.getFirstChild();
        while (child != null) {
            if (child instanceof Element) {
                final Element childAsElement = (Element) child;
                final String classAtt = childAsElement.getAttribute("class");
                if (classAtt != null && containsAnyClass(classAtt, classNames)) {
                    result.add(child);
                } else {
                    result.addAll(getFirstNodesWithClass(child, classNames));
                }
            }
            child = child.getNextSibling();
        }
        return result;
    }

    public static boolean containsAnyClass(final String classAtt, final List<String> classNames) {
        final String[] classParts = splitBySpaces(classAtt);
        for (final String part : classParts) {
            if (classNames.contains(part.toLowerCase())) {
                return true;
            }
        }
        return false;
    }

    private static final String[][] uriValueAttributes =
            {{"a", "href"}, {"area", "href"}, {"img", "src", "srclow"}, {"form", "action"}, {"object", "data"}};
    private static final HashMap<String, String[]> uriValueAttMap = new HashMap<String, String[]>();
    public static final List<String> SAVE_FOR_HCARD =
            Collections.unmodifiableList(CollectionFactory.list("adr", "org"));

    public static Microformat getMFForClasses(final String classes, final MicroformatsManager manager) {
        if (classes == null) {
            return null;
        }
        final String[] classParts = splitBySpaces(classes);
        for (final String classPart : classParts) {
            final Microformat format = manager.get(classPart);
            if (format != null) {
                return format;
            }
        }
        return null;
    }

    public static List<MFProperty> getPropsForClasses(final String classes, final Microformat microformat) {
        final List<MFProperty> res = new ArrayList<MFProperty>();
        if (classes == null) {
            return res;
        }
        final String[] classParts = splitBySpaces(classes);
        for (final String classPart : classParts) {
            final MFProperty prop;
            try {
                prop = microformat.getProperty(classPart);
                if (prop != null) {
                    res.add(prop);
                }
            } catch (InvalidActionException e) {
                e.printStackTrace();  //cannot be
            }
        }
        return res;
    }

    public static List<MFProperty> getPropsForClasses(final String classes, final Set<Microformat> microformats) {
        final List<MFProperty> res = new ArrayList<MFProperty>();
        if (classes == null) {
            return res;
        }
        final String[] classParts = splitBySpaces(classes);
        for (final String classPart : classParts) {
            for (final Microformat microformat : microformats) {
                final MFProperty prop;
                try {
                    prop = microformat.getProperty(classPart);
                    if (prop != null) {
                        res.add(prop);
                    }
                } catch (InvalidActionException e) {
                    e.printStackTrace();  //cannot be
                }
            }
        }
        return res;
    }

    public static boolean mayContain(final Microformat format, final String propertyName) {
        try {
            return format.getProperty(propertyName) != null;
        } catch (InvalidActionException e) {
            e.printStackTrace();  //cannot be
        }
        return false;
    }

    public static boolean mayContain(final MFProperty prop, final Microformat format, final Element value) {
        try {
            return prop.getPossibleTypes().contains(format) &&
                    Tu.findByClass(value, format.getName(), true).contains(value);
        } catch (InvalidActionException e) {
            e.printStackTrace();  //cannot be.
        }
        return false;
    }

    public static boolean mayContain(final MFProperty prop, final Microformat format, final TagNode value) {
        try {
            return prop.getPossibleTypes().contains(format) &&
                    Tu.findByClass(value, format.getName(), true).contains(value);
        } catch (InvalidActionException e) {
            e.printStackTrace();  //cannot be.
        }
        return false;
    }


    public static boolean mayContain(final MFProperty prop, final String propertyName) {
        for (final MFType type : prop.getPossibleTypes()) {
            if (type instanceof Microformat) {
                final Microformat format = (Microformat) type;
                try {
                    if (format.getName().equals(prop.getName()) && mayContain(format, propertyName)) {
                        return true;
                    }
                } catch (InvalidActionException e) {
                    e.printStackTrace();  //cannot be
                }
            }
        }
        return false;
    }

    public static List<Microformat> getMFsForClasses(final String classes, final MicroformatsManager manager) {
        final LinkedList<Microformat> result = new LinkedList<Microformat>();
        if (classes == null) {
            return null;
        }
        final String[] classParts = splitBySpaces(classes);
        for (final String classPart : classParts) {
            final Microformat format = manager.get(classPart);
            if (format != null) {
                result.add(format);
            }
        }
        return result;
    }

    public static List<Microformat> getRootMFsForClasses(final String classes, final MicroformatsManager manager) {
        final LinkedList<Microformat> result = new LinkedList<Microformat>();
        if (classes == null) {
            return null;
        }
        final String[] classParts = splitBySpaces(classes);
        for (final String classPart : classParts) {
            final Microformat format = manager.get(classPart);
            if (format != null && format.isRoot()) {
                result.add(format);
            }
        }
        return result;
    }

    public static List<MFType> getCommontypesForMF(final Microformat format, final String classAtt, final boolean onlyComplex) {
        final String[] classParts = splitBySpaces(classAtt);
        final LinkedList<MFType> result = new LinkedList<MFType>();
        final LinkedList<MFProperty> props = new LinkedList<MFProperty>();
        for (final String classPart : classParts) {
            try {
                final MFProperty prop = format.getProperty(classPart);
                if (prop != null) {
                    props.add(prop);
                }
            } catch (InvalidActionException e) {
                //can't be
            }
        }
        if (props.isEmpty()) {
            return Collections.EMPTY_LIST;
        }
        result.addAll(props.get(0).getPossibleTypes());
        final Iterator<MFProperty> propIt = props.iterator();
        propIt.next();
        while (propIt.hasNext()) {
            final List<MFType> curTypes = propIt.next().getPossibleTypes();
            final Iterator<MFType> propTypeIt = result.iterator();
            while (propTypeIt.hasNext()) {
                final MFType curType = propTypeIt.next();
                if (!curTypes.contains(curType)) {
                    propTypeIt.remove();
                }
            }
        }
        if (onlyComplex) {
            final Iterator<MFType> resultIt = result.iterator();
            while (resultIt.hasNext()) {
                final MFType type = resultIt.next();
                if (!(type instanceof Microformat)) {
                    resultIt.remove();
                }
            }
        }
        return result;
    }


    static {
        for (final String[] tags : uriValueAttributes) {
            final String[] atts = new String[tags.length - 1];
            System.arraycopy(tags, 1, atts, 0, tags.length - 1);
            uriValueAttMap.put(tags[0], atts);
        }
    }

    public static String getUriElementValue(final Element node, final boolean useId) {
        final String[] atts = uriValueAttMap.get(node.getTagName().toLowerCase());
        if (atts != null) {
            for (final String att : atts) {
                final String val = node.getAttribute(att);
                if (val != null) {
                    return stripPrefix(val);
                }
            }
        }
        if (useId) {
            final String id = node.getAttribute("id");
            if (!"#".equals(id)) {
                return "#" + id;
            }
        }
        return null;
    }

    public static String getUriElementValue(final TagNode node, final boolean useId) {
        final String[] atts = uriValueAttMap.get(node.getName().toLowerCase());
        if (atts != null) {
            for (final String att : atts) {
                final String val = node.getAttributeByName(att);
                if (val != null) {
                    return stripPrefix(val);
                }
            }
        }
        if (useId) {
            final String id = node.getAttributeByName("id");
            if (!"#".equals(id)) {
                return "#" + id;
            }
        }
        return null;
    }

    private static String stripPrefix(final String val) {
        if (val.startsWith("mailto:")) {
            return val.substring("mailto:".length());
        }
        return val;
    }

    public static String getUriValue(final Element root, final boolean useId) {
        final String rootValue = getUriElementValue(root, useId);
        if (rootValue != null) {
            return rootValue;
        }
        final List<Element> valueChilds = getValueChildren(root);
        for (final Element el : valueChilds) {
            final String value = getUriElementValue(el, useId);
            if (value != null) {
                return value;
            }
        }
        return stringifyNode(root, true, true, false);
    }

    public static String getUriValue(final TagNode root, final boolean useId) {
        final String rootValue = getUriElementValue(root, useId);
        if (rootValue != null) {
            return rootValue;
        }
        final List<TagNode> valueChilds = getValueChildren(root);
        for (final TagNode el : valueChilds) {
            final String value = getUriElementValue(el, useId);
            if (value != null) {
                return value;
            }
        }
        return stringifyNode(root, true, true, false);
    }

    public static String getTextContent(final Node node) {
        final StringBuilder sb = new StringBuilder();
        Node child = node.getFirstChild();
        while (child != null) {
            if ("#text".equals(child.getNodeName())) {
                sb.append(child.getNodeValue());
            }
            if (child instanceof Element) {
                final Element elem = (Element) child;
                if (containsOnlyRels(elem.getAttribute("class"))) {
                    sb.append(getTextContent(child));
                }
            }
            child = child.getNextSibling();
        }
        return sb.toString();
    }

    private static boolean containsOnlyRels(final String classes) {
        final String[] classParts = splitBySpaces(classes);
        for (final String s : classParts) {
            if (!s.isEmpty() && !s.startsWith("rel-")) {
                return false;
            }
        }
        return true;
    }

    public static String getDatetimeContent(final Element node) {
        final String dateAtt = node.getAttribute("datetime");
        if (dateAtt != null && !dateAtt.isEmpty()) {
            return dateAtt;
        }
        final List<Element> valNodes = getValueChildren(node);
        final LinkedList<String> vals = new LinkedList<String>();
        for (final Element element : valNodes) {
            final String childDateAtt = element.getAttribute("datetime");
            if (childDateAtt != null && !childDateAtt.isEmpty()) {
                vals.add(childDateAtt);
            } else {
                vals.add(stringifyNode(element, true, true, false));
            }
        }
        if (vals.isEmpty()) {
            vals.add(stringifyNode(node, false, true, false).trim());
        }
        return Su.join(vals, "");
    }

    public static String getDatetimeContent(final TagNode node) {
        final String dateAtt = node.getAttributeByName("datetime");
        if (dateAtt != null && !dateAtt.isEmpty()) {
            return dateAtt;
        }
        final List<TagNode> valNodes = getValueChildren(node);
        final LinkedList<String> vals = new LinkedList<String>();
        for (final TagNode element : valNodes) {
            final String childDateAtt = element.getAttributeByName("datetime");
            if (childDateAtt != null && !childDateAtt.isEmpty()) {
                vals.add(childDateAtt);
            } else {
                vals.add(stringifyNode(element, true, true, false));
            }
        }
        if (vals.isEmpty()) {
            vals.add(stringifyNode(node, false, true, false).trim());
        }
        return Su.join(vals, "");
    }

    public static String stripClassFromClasses(final String classes, final String classToDelete) {
        final StringBuilder sb = new StringBuilder();
        for (final String part : splitBySpaces(classes)) {
            if (!part.equalsIgnoreCase(classToDelete)) {
                sb.append(part).append(" ");
            }
        }
        return sb.toString();
    }

    public static String stringifyForPostprocess(final Node root) {
        if (root.getUserData("stop") != null) {
            return "";
        }
        if (root instanceof Text) {
            return ((Text) root).getWholeText();
        }
        final StringBuilder sb = new StringBuilder();
        Node child = root.getFirstChild();
        while (child != null) {
            sb.append(stringifyForPostprocess(child));
            child = child.getNextSibling();
        }
        return sb.toString();
    }


}
                                                                            