package ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.jsonld;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import org.htmlcleaner.TagNode;
import ru.yandex.common.util.collections.MultiMap;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.location.EntityLocation;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.data.*;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.exceptions.JsonCollidingKeywordsRDFaException;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @author Ivan Nikolaev <ivannik@yandex-team.ru>
 */
public final class RDFaEntityJsonLdDeserialilizer extends StdDeserializer<RDFaEntity> {

    private static final long serialVersionUID = 1L;

    private final List<JsonCollidingKeywordsRDFaException> duplicateFieldsExceptions;

    private final TagNode parentNode;

    public RDFaEntityJsonLdDeserialilizer() {
        this(new ArrayList<JsonCollidingKeywordsRDFaException>(), null);
    }

    public RDFaEntityJsonLdDeserialilizer(List<JsonCollidingKeywordsRDFaException> duplicateFieldsExceptions, TagNode node) {
        super(RDFaEntity.class);
        this.duplicateFieldsExceptions = duplicateFieldsExceptions;
        this.parentNode = node;
    }



    @Override
    public RDFaEntity deserialize(JsonParser jp, DeserializationContext ctxt) throws IOException {
        if (jp.getCurrentToken() == JsonToken.START_OBJECT) {
            jp.nextToken();
            return deserializeObject(jp, ctxt, null);
        }
        if (jp.getCurrentToken() == JsonToken.FIELD_NAME) {
            return deserializeObject(jp, ctxt, null);
        }
        throw ctxt.mappingException(RDFaEntity.class);
    }

    protected final RDFaEntity deserializeObject(JsonParser jp, DeserializationContext ctxt, RDFaEntity root)
            throws IOException {
        EntityLocation location =
                new EntityLocation(parentNode.getRow() + jp.getCurrentLocation().getLineNr() - 1, jp.getCurrentLocation().getColumnNr());
        JSONLDEntity md = new JSONLDEntity(null, null, location);
        md.setParentNodeLocation(new EntityLocation(parentNode.getRow(), parentNode.getCol()));
        RDFaEntity rootNode = root == null ? md : root;
        JsonToken t = jp.getCurrentToken();
        if (t == JsonToken.START_OBJECT) {
            t = jp.nextToken();
        }
        Set<String> keys = new HashSet<>();
        for (; t == JsonToken.FIELD_NAME; t = jp.nextToken()) {
            String key = jp.getCurrentName();
            EntityLocation currentLocation = new EntityLocation(parentNode.getRow() + jp.getCurrentLocation().getLineNr() - 1,
                    jp.getCurrentLocation().getColumnNr());
            if (!keys.add(key)) {
                duplicateFieldsExceptions.add(new JsonCollidingKeywordsRDFaException(false, rootNode, key, currentLocation));
                jp.nextToken();
                jp.skipChildren();
                continue;
            }
            switch (jp.nextToken()) {

                case START_OBJECT:
                    RDFaEntity value = deserializeObject(jp, ctxt, rootNode);
                    md.appendProperty(new RDFaComplexProperty(key, value));
                    break;
                case START_ARRAY:
                    List<RDFaProperty> values = deserializeArray(key, jp, ctxt, rootNode);
                    md.appendProperties(values);
                    break;
                case VALUE_NULL:
                    md.addProperty(new RDFaValueProperty(key, null, null, "null", currentLocation));
                    break;
                default:
                    String text = jp.getText();
                    md.addProperty(new RDFaValueProperty(key, text, null, text, currentLocation));
            }
        }
        return md;
    }

    protected final List<RDFaProperty> deserializeArray(String key, JsonParser jp, DeserializationContext ctxt, RDFaEntity root)
            throws IOException {
        List<RDFaProperty> rdFaProperties = new ArrayList<>();
        while (true) {
            JsonToken t = jp.nextToken();
            if (t == null) {
                throw ctxt.mappingException("Unexpected end-of-input when binding data into List<RDFaProperty>");
            }
            EntityLocation currentLocation = new EntityLocation(parentNode.getRow() + jp.getCurrentLocation().getLineNr() - 1,
                    jp.getCurrentLocation().getColumnNr());
            switch (t) {
                case START_OBJECT:
                    RDFaEntity value = deserializeObject(jp, ctxt, root);
                    rdFaProperties.add(new RDFaComplexProperty(key, value));
                    break;
                case START_ARRAY:
                    rdFaProperties.addAll(deserializeArray(key, jp, ctxt, root));
                    break;
                case END_ARRAY:
                    return rdFaProperties;
                case VALUE_NULL:
                    rdFaProperties.add(new RDFaValueProperty(key, null, null, "null", currentLocation));
                    break;
                default:
                    String text = jp.getText();
                    rdFaProperties.add(new RDFaValueProperty(key, text, null, text, currentLocation));
                    break;
            }
        }
    }
}
