package ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.validator;

import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.util.Su;
import ru.yandex.common.util.collections.Cf;
import ru.yandex.webmaster3.core.semantic.schema_org_information_extractor.SchemaClass;
import ru.yandex.webmaster3.core.semantic.schema_org_information_extractor.SchemaField;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.MicrodataUtils;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.data.*;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.exceptions.*;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.validators.ISO8601Validator;

import java.util.*;
import java.util.regex.Pattern;

/**
 * Created by aleksart on 04.03.14.
 */
public class GoodRelationsValidator implements RDFaValidator {
    private String vocab;

    @Required
    public void setVocab(final String vocab) {
        this.vocab = vocab;
    }

    private final Map<String, SchemaClass> classMaps;

    public GoodRelationsValidator(final Map<String, SchemaClass> classMap) {
        this.classMaps = classMap;
    }

    @Override
    public List<RDFaException> validate(RDFaEntity data) {
        final List<String> types = Su.split(data.type);
        final Map<String, SchemaClass> classMap = classMaps;
        final List<RDFaException> exceptions = new LinkedList<RDFaException>();
        for (String hrefType : types) {
            String httpType = MicrodataUtils.addPrefixIfEmpty(hrefType);
            String vocabType = MicrodataUtils.addPrefixIfEmpty(vocab);
            List<String> vocabTypes = MicrodataUtils.protocolAliases(vocabType);
            boolean rightVocab = false;
            for(String vocab : vocabTypes){
                if(httpType.startsWith(vocab)){
                    rightVocab = true;
                }
            }
            if (rightVocab) {
                final List<String> allTypes = Su.split(data.type);


                final List<SchemaClass> additionalClasses = new LinkedList<SchemaClass>();
                for (final String addType : allTypes) {
                    if (classMaps.containsKey(addType)) {
                        final SchemaClass addCls = classMap.get(addType);
                        if (addCls != null) {
                            additionalClasses.add(addCls);
                        }
                    }
                }
                for (String type : allTypes) {
                    String vocabPrefix = MicrodataUtils.addPrefixIfEmpty(hrefType);
                    if (!type.startsWith(vocabPrefix)) {
                        continue;
                    }
                    final SchemaClass addCls = classMap.get(type);
                    if (addCls == null) {
                        final String fixedType = MicrodataUtils.addPrefixIfEmpty(type);
                        boolean found = false;
                        for (final String klass : classMap.keySet()) {
                            if (fixedType.toLowerCase().equals(klass.toLowerCase())) {
                                exceptions.add(
                                        new SchemaMisspellRDFaValidatorException("type", true, data, type + "$$" + klass));
                                found = true;
                            }
                        }
                        if (!found) {
                            exceptions.add(new NoSuchSchemaClassRDFaValidatorException(false, data, type));
                        }
                    }

                }
                if (additionalClasses.isEmpty()) {
                    return exceptions;
                }
                validate(data, hrefType, classMap, exceptions, additionalClasses);

            }

        }
        return exceptions;

    }

    private void validate(RDFaEntity data, String hrefType, Map<String, SchemaClass> classMap, List<RDFaException> exceptions, List<SchemaClass> types) {

        for (final RDFaProperty property : data.getValuePairs()) {
            Pattern vocab_pattern;
            if(vocab.equals("schema.org")) {
                vocab_pattern = Pattern.compile("https?://(www.)?" + vocab + ".*");
            }
            else{
                vocab_pattern = Pattern.compile("https?://" + vocab + ".*");
            }
            if (vocab_pattern.matcher(property.propId).matches()) {

                String fieldName = property.propId;
                if ("itemId".equals(fieldName)) {
                    continue;
                }
                boolean foundField = false;
                final List<SchemaField> fields = new LinkedList<SchemaField>();
                for (final SchemaClass curCls : types) {
                    final SchemaField fld = curCls.fields.get(fieldName);
                    if (fld == null) {
                        continue;
                    }
                    foundField = true;
                    fields.add(fld);
                }
                if (!foundField) {
                    SchemaClass foundScheme = null;
                    String foundFieldName = null;
                    for (final SchemaClass curCls : types) {
                        // field name to long
                        for (final String field : curCls.fields.keySet()) {
                            if (field.toLowerCase().equals(fieldName.toLowerCase())) {
                                foundScheme = curCls;
                                foundFieldName = field;
                                foundField = true;
                                break;
                            }
                        }
                        if (foundField) {
                            break;
                        }
                    }
                    if (foundField) {
                        exceptions.add(new SchemaMisspellRDFaValidatorException("field", true, data,
                                fieldName + "$$" + foundScheme.name + "$$" + foundFieldName, property.getLocation()));
                    } else {
                        exceptions.add(new NoSuchSchemaFieldRDFaValidatorException(false, data, fieldName + "$$" + hrefType, property.getLocation()));
                    }
                } else {
                    if (property instanceof RDFaValueProperty) {

                        String textValue = ((RDFaValueProperty) property).textValue;
                        String hrefValue = ((RDFaValueProperty) property).hrefValue;
                        if (hrefValue == null) {
                            hrefValue = "";
                        }

                        if (onlyType(fields, "http://www.w3.org/2001/XMLSchema#dateTime")) {
                            if (!ISO8601Validator.validate(textValue)) {
                                exceptions.add(new DatetimeFormatRDFaValidatorException(false, data,
                                        textValue + "$$" + fieldName));
                            }
                        } else if (onlyType(fields, "http://www.w3.org/2001/XMLSchema#time")) {
                            if (!ISO8601Validator.validate(textValue)) {
                                exceptions.add(new DatetimeFormatRDFaValidatorException(false, data,
                                        textValue + "$$" + fieldName));
                            }
                        } else if (onlyType(fields, "http://www.w3.org/2001/XMLSchema#float")) {
                            if (!FLOAT_REGEXP.matcher(textValue).matches()) {
                                exceptions.add(new SchemaTypeRDFaValidatorException("float", true, data,
                                        textValue + "$$" + fieldName));
                            }
                        } else if (onlyType(fields, "http://www.w3.org/2001/XMLSchema#int")) {
                            if (!INTEGER_REGEXP.matcher(textValue).matches()) {
                                exceptions.add(new SchemaTypeRDFaValidatorException("number", true, data,
                                        textValue + "$$" + fieldName));
                            }
                        } else if (onlyType(fields, "http://www.w3.org/2001/XMLSchema#boolean")) {
                            if (!"True".equals(textValue) &&
                                    !"False".equals(textValue)) {
                                exceptions.add(new SchemaTypeRDFaValidatorException("boolean", true, data,
                                        textValue + "$$" + fieldName));
                            }
                        }
                        // text value ok for all fields
                    } else if (property instanceof RDFaComplexProperty) {
                        boolean foundType = false;
                        ArrayList<String> validTypes = new ArrayList<String>();
                        for (final SchemaField fld : fields) {
                            if ((fld.ranges.size() == 1) && (fld.ranges.get(0).endsWith("#"))) {

                                String range = fld.ranges.get(0);

                                final SchemaClass schemaClass = classMap.get(range);
                                validate(((RDFaComplexProperty) property).entity, "", classMap, exceptions, Cf.list(schemaClass));

                            }
                            final SchemaClass chkCls = classMap.get(((RDFaComplexProperty) property).entity.type);
                            if (chkCls != null && checkClass(chkCls, fld)) {
                                foundType = true;
                            } else {
                                for (String rng : fld.ranges) {
                                    validTypes.add(rng);
                                }
                            }

                        }
                        if (!foundType) {
                            exceptions.add(new WrongClassForFieldRDFaValidatorException(false, data,
                                    fieldName + "$$" + (((RDFaComplexProperty) property).entity.type + "$$" + Su.join(validTypes, ", "))));
                        }
                    }
                }
            }
        }

    }

    private final static Pattern FLOAT_REGEXP = Pattern.compile("^[-+]?[0-9]*\\.?[0-9]+([eE][-+]?[0-9]+)?$");
    private final static Pattern INTEGER_REGEXP = Pattern.compile("^[-+]?[0-9]+$");

    private boolean onlyType(final List<SchemaField> fields, final String type) {
        final Set<String> classNames = new HashSet<String>();
        for (final SchemaField field : fields) {
            classNames.addAll(field.ranges);
        }
        for(String addType: MicrodataUtils.protocolAliases(type)) {
            if (classNames.contains(addType)){
                return true;
            }
        }
        return false;
    }

    private boolean checkClass(final SchemaClass chkCls, final SchemaField fld) {
        for (final String range : fld.ranges) {
            if (range.equals(chkCls.name)) {
                return true;
            }
        }
        if (chkCls != null) {
            for (final SchemaClass parent : chkCls.subClassOf) {
                if (checkClass(parent, fld)) {
                    return true;
                }
            }
        }
        return false;
    }
}
