package ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.validator;


import ru.yandex.common.util.collections.Cf;
import ru.yandex.common.util.collections.Pair;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.MicrodataUtils;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.data.RDFaComplexProperty;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.data.RDFaEntity;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.data.RDFaProperty;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.data.RDFaValueProperty;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.exceptions.OGPVideoRDFaException;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.exceptions.RDFaException;

import java.util.*;
import java.util.regex.Pattern;

/**
 * Created by IntelliJ IDEA.
 * User: rasifiel
 * Date: 12.09.12
 * Time: 20:12
 */
public class OGPVideoRDFaValidator implements RDFaValidator {

    private final List<Pair<String, String>> fields;
    private final List<String> validTypes;

    public OGPVideoRDFaValidator(final Map<String, String> fields, List<String> validTypes) {
        this.fields = new ArrayList<>();
        for (String key : fields.keySet()) {
            this.fields.add(Pair.of(key, fields.get(key)));
        }
        this.validTypes = validTypes;
    }

    private static Pattern OGP_PROTOCOL_PATTERN = Pattern.compile("https?://ogp.me/ns#.*");

    @Override
    public List<RDFaException> validate(final RDFaEntity data) {

        if (data.isRoot && isOgEntity(data)) {

            List<RDFaEntity> addEntities;
            final String httpType = "http://ogp.me/ns#type";
            final List<RDFaProperty> types = new ArrayList<>();
            for (String httpTypeAlias : MicrodataUtils.protocolAliases(httpType)) {
                types.addAll(data.getProperty(httpTypeAlias));
            }
            if (types.size() == 1) {
                RDFaProperty type = types.get(0);

                String value = null;
                if (type instanceof RDFaValueProperty) {
                    RDFaValueProperty typeProperty = (RDFaValueProperty) type;
                    value = typeProperty.textValue;
                } else if (type instanceof RDFaComplexProperty) {
                    RDFaComplexProperty entityProp = (RDFaComplexProperty) type;
                    RDFaEntity entity = entityProp.entity;
                    if (entity.hasProperty("@value")) {
                        RDFaProperty typeProperty = entity.getProperty("@value").get(0);
                        if (typeProperty instanceof RDFaValueProperty) {
                            value = ((RDFaValueProperty) typeProperty).textValue;
                        }
                    }
                }
                if (null != value) {
                    if (Cf.set(MicrodataUtils.protocolAliases(validTypes)).contains(value)) {
                        final List<RDFaException> exceptions = new LinkedList<>();
                        final List<String> missingFields = new LinkedList<>();

                        addEntities = exploreData(data, new ArrayList<>());
                        for (final Pair<String, String> field : fields) {
                            if (data.getProperty(field.getFirst()).isEmpty()) {
                                boolean empty = true;
                                for (RDFaEntity entity : addEntities) {
                                    if (!entity.getProperty(field.getFirst()).isEmpty()) {
                                        empty = false;
                                        break;
                                    }
                                }
                                if (empty) {
                                    missingFields.add(field.getSecond());
                                }
                            }
                        }
                        if (!missingFields.isEmpty()) {
                            StringBuilder resultBuilder = new StringBuilder();
                            for (final String field : missingFields) {
                                if (resultBuilder.length() == 0) {
                                    resultBuilder.append(field);
                                } else {
                                    resultBuilder.append(", ").append(field);
                                }
                            }
                            exceptions.add(new OGPVideoRDFaException(false, data, resultBuilder.toString()));
                        }
                        return exceptions;
                    }
                }
            }
        }
        return Collections.emptyList();

    }

    private ArrayList<RDFaEntity> exploreData(RDFaEntity data, ArrayList<RDFaEntity> entities) {

        for (RDFaProperty property : data.getValuePairs()) {
            if (property instanceof RDFaComplexProperty) {
                entities.add(((RDFaComplexProperty) property).entity);
                entities = exploreData(((RDFaComplexProperty) property).entity, entities);
            }
        }
        return entities;
    }

    private boolean isOgEntity(RDFaEntity data) {
        if (data.hasProperty("@type")) {
            List<RDFaProperty> properties = data.getProperty("@type");
            return properties.size() > 0 &&
                    properties.get(0) instanceof RDFaValueProperty &&
                    "og".equals(((RDFaValueProperty) properties.get(0)).getValue());
        } else {
            for (final String field : data.getPropertyList()) {
                if (OGP_PROTOCOL_PATTERN.matcher(field).matches()) {
                    return true;
                }
            }
        }
        return false;
    }
}
