package ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.validator;

import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.util.Su;
import ru.yandex.common.util.collections.Cf;
import ru.yandex.common.util.collections.Cu;
import ru.yandex.common.util.functional.Function;
import ru.yandex.webmaster3.core.semantic.schema_org_information_extractor.SchemaClass;
import ru.yandex.webmaster3.core.semantic.schema_org_information_extractor.SchemaField;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.MicrodataUtils;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.validators.ISO4217Validatior;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.data.*;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.exceptions.*;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.validators.ISO8601Validator;

import java.util.*;
import java.util.regex.Pattern;

/**
 * Created with IntelliJ IDEA.
 * User: aleksart
 * Date: 20.11.13
 * Time: 20:07
 * To change this template use File | Settings | File Templates.
 */
public class SchemaValidator implements RDFaValidator {
    private String vocab;
    @Required
    public void setVocab(final String vocab) {
        this.vocab = vocab;
    }

    private final Map<String, SchemaClass> classMaps;

    private final Comparator exceptionComparator = new RDFaExceptionComparator();

    public SchemaValidator(final Map<String, SchemaClass> classMap) {
        this.classMaps = classMap;
    }

    @Override
    public List<RDFaException> validate(RDFaEntity data) {
        if(!data.isRoot && !(data instanceof JSONLDEntity)){
            return Collections.EMPTY_LIST;
        }
        final List<String> types = Su.split(data.type);
        if (types.isEmpty()) {
            List<RDFaProperty> typesList = data.getProperty("@type");
            for (RDFaProperty type : typesList) {
                if (type != null && type instanceof RDFaValueProperty) {
                    types.add(((RDFaValueProperty) type).getValue());
                }
            }
        }
        final Map<String, SchemaClass> classMap = classMaps;
        final List<RDFaException> exceptions = new LinkedList<RDFaException>();
        final List<String> allTypes = new ArrayList<String>();
        for (String hrefType : types) {
            String httpType = MicrodataUtils.addPrefixIfEmpty(hrefType);
            String vocabType = MicrodataUtils.addPrefixIfEmpty(vocab);
            List<String> vocabTypes = MicrodataUtils.protocolAliases(vocabType);
            boolean rightVocab = false;
            for(String vocab : vocabTypes){
                if(httpType.startsWith(vocab)){
                    rightVocab = true;
                }
            }
            if (rightVocab) {
                allTypes.add(httpType);
            }

        }

        for (String hrefType : types) {
            String httpType = MicrodataUtils.addPrefixIfEmpty(hrefType);
            String vocabType = MicrodataUtils.addPrefixIfEmpty(vocab);
            List<String> vocabTypes = MicrodataUtils.protocolAliases(vocabType);
            boolean rightVocab = false;
            for(String vocab : vocabTypes){
                if(httpType.startsWith(vocab)){
                    rightVocab = true;
                }
            }
            if (rightVocab) {

                final List<SchemaClass> additionalClasses = new LinkedList<SchemaClass>();
                for (final String addType : allTypes) {
                    if (classMaps.containsKey(addType)) {
                        final SchemaClass addCls = classMap.get(addType);
                        if (addCls != null) {
                            additionalClasses.add(addCls);
                        }
                    }
                }
                for (String type : allTypes) {
                    String vocabPrefix = MicrodataUtils.addPrefixIfEmpty(vocab);
                    List<String> vocabTypes1 = MicrodataUtils.protocolAliases(vocabPrefix);
                    boolean rightVocab1 = false;
                    for(String vocab : vocabTypes1){
                        if(httpType.startsWith(vocab)){
                            rightVocab1 = true;
                        }
                    }
                    if (rightVocab1) {
                        final SchemaClass addCls = classMap.get(type);
                        if (addCls == null) {
                            final String fixedType = MicrodataUtils.addPrefixIfEmpty(type);
                            boolean found = false;
                            for (final String klass : classMap.keySet()) {
                                if (fixedType.toLowerCase().equals(klass.toLowerCase())) {
                                    exceptions.add(new SchemaMisspellRDFaValidatorException("type", true, data,
                                            type + "$$" + klass + "$$" + vocabPrefix));
                                    found = true;
                                }
                            }
                            if (!found) {
                                exceptions.add(
                                        new NoSuchSchemaClassRDFaValidatorException(false, data, type + "$$" + vocab));
                            }
                        }
                    }

                }
                if (additionalClasses.isEmpty()) {

                    return Cu.uniq(exceptionComparator, exceptionComparator ,exceptions);
                }
                validate(data, hrefType, classMap, exceptions, additionalClasses);
            }
        }
        return Cu.uniq(exceptionComparator, exceptionComparator ,exceptions);
    }


    private final static Set<String> ALLOWED_BOOL =
            Cf.set("true", "false", "True", "False", "http://schema.org/True", "http://schema.org/False", "https://schema.org/True", "https://schema.org/False" ,
                    "http://www.schema.org/True", "http://www.schema.org/False","https://www.schema.org/True", "https://www.schema.org/False");

    private void validate(RDFaEntity data, String hrefType, Map<String, SchemaClass> classMap, List<RDFaException> exceptions, List<SchemaClass> types) {

        for (final RDFaProperty property : data.getValuePairs()) {
            Pattern vocab_pattern;
            if(vocab.equals("schema.org")) {
                vocab_pattern = Pattern.compile("https?://(www.)?" + vocab + ".*");
            }
            else{
                vocab_pattern = Pattern.compile("https?://" + vocab + ".*");
            }
            if (vocab_pattern.matcher(property.propId).matches()) {
                String fieldNameRaw = property.propId;
                final String fieldName;
                final String suffix;
                if (start_with_schema.matcher(fieldNameRaw).matches()) {
                    String[] parts = Su.split(fieldNameRaw, '-', 2);
                    fieldName = parts[0];
                    if (parts.length > 1) {
                        suffix = parts[1];
                    } else {
                        suffix = "";
                    }
                } else {
                    fieldName = fieldNameRaw;
                    suffix = "";
                }
                if ("itemId".equals(fieldName)) {
                    continue;
                }
                boolean foundField = false;
                final List<SchemaField> fields = new LinkedList<SchemaField>();
                for (final SchemaClass curCls : types) {
                    final SchemaField fld = curCls.fields.get(fieldName);
                    if (fld == null) {
                        continue;
                    }
                    foundField = true;
                    fields.add(fld);
                }
                if (!foundField) {
                    SchemaClass foundScheme = null;
                    String foundFieldName = null;
                    for (final SchemaClass curCls : types) {
                        // field name to long
                        for (final String field : curCls.fields.keySet()) {
                            if (field.toLowerCase().equals(fieldName.toLowerCase())) {
                                foundScheme = curCls;
                                foundFieldName = field;
                                foundField = true;
                                break;
                            }
                        }
                        if (foundField) {
                            break;
                        }
                    }
                    if (foundField) {
                        exceptions.add(new SchemaMisspellRDFaValidatorException("field", true, data,
                                fieldName + "$$" + foundScheme.name + "$$" + foundFieldName, property.getLocation()));
                    } else {
                        exceptions.add(
                                new NoSuchSchemaFieldRDFaValidatorException(false, data, fieldName + "$$" + hrefType, property.getLocation()));
                    }
                } else {
                    if (property instanceof RDFaValueProperty) {
                        validateSimpleData(data, exceptions, (RDFaValueProperty) property, fieldName, MAP_RANGES.map(fields));
                    } else if (isJSONLDValueProp(property)) {
                        validateJSONLDSimpeData(data, exceptions, (RDFaComplexProperty) property, fieldName, fields);
                    } else if (property instanceof RDFaComplexProperty) {
                        boolean foundType = false;
                        ArrayList<String> validTypes = new ArrayList<String>();
                        List<String> fieldTypes = Su.split(((RDFaComplexProperty) property).entity.type);
                        List<RDFaProperty> typesList = ((RDFaComplexProperty) property).entity.getProperty("@type");
                        for (RDFaProperty type : typesList) {
                            if (type != null && type instanceof RDFaValueProperty) {
                                fieldTypes.add(((RDFaValueProperty) type).getValue());
                            }
                        }
                        List<SchemaClass> classes = new ArrayList<SchemaClass>(fieldTypes.size());
                        for (final String range : fieldTypes) {
                            final SchemaClass schemaClass = classMap.get(range);
                            if (schemaClass != null) {
                                classes.add(schemaClass);
                            }
                        }
                        if (isRole(classes)) {
                            classes.addAll(types);
                        } else {
                            for (final SchemaField fld : fields) {

                                List<String> ranges = fld.ranges;
                                if ("input".equals(suffix) || "output".equals(suffix)) {
                                    ranges = Cu.join(ranges, Cf.list("http://schema.org/PropertyValueSpecification","https://schema.org/PropertyValueSpecification"));
                                }
                                for (final String fieldType : fieldTypes) {
                                    final SchemaClass chkCls = classMap.get(fieldType);
                                    if (chkCls != null && checkClass(chkCls, ranges)) {
                                        foundType = true;
                                    }
                                }
                                if (!foundType) {

                                    for (String rng : ranges) {
                                        validTypes.add(rng);
                                    }
                                }


                            }
                            if (!foundType) {
                                exceptions.add(
                                        new WrongClassForFieldRDFaValidatorException(false, data, fieldName + "$$" +
                                                fieldTypesOrNull(fieldTypes) + "$$" + Su.join(MicrodataUtils.httpPrefixOnly(validTypes), ", "), property.getLocation()));
                            }
                        }
                        if (!classes.isEmpty()) {
                            validate(((RDFaComplexProperty) property).entity, Su.join(MicrodataUtils.httpPrefixOnly(fieldTypes), " "), classMap,
                                    exceptions, classes);
                        }
                    }
                }
            }
        }

    }
    private String fieldTypesOrNull(List<String> fieldTypes){
        if(fieldTypes.size()==1){
            if(fieldTypes.get(0).matches("\\s") || fieldTypes.get(0).isEmpty()){
                return "null";
            }
        }
        if(fieldTypes.size() > 0){
            return Su.join(fieldTypes, " ");
        }
        return "null";
    }

    public static void validateJSONLDSimpeData(final RDFaEntity data, final List<RDFaException> exceptions, final RDFaComplexProperty property, final String fieldName, final List<SchemaField> fields) {
        List<RDFaProperty> values = property.entity.getProperty("@value");
        if (values.isEmpty()) {
            values = property.entity.getProperty("@id");
        }
        List<String> fieldTypes = new ArrayList<String>();
        if (property.entity.hasProperty("@type")) {
            RDFaProperty type = property.entity.getFirstOrNull("@type");
            if (type instanceof RDFaValueProperty) {
                fieldTypes.add(((RDFaValueProperty) type).getValue());
            }
        }

        for (final RDFaProperty p : values) {
            if (p instanceof RDFaValueProperty) {
                RDFaValueProperty pv = (RDFaValueProperty) p;
                validateSimpleData(data, exceptions, new RDFaValueProperty(pv.propId, pv.textValue,
                        Su.isEmpty(pv.hrefValue) ? pv.textValue : pv.hrefValue, pv.htmlValue, pv.getLocation()),
                        fieldName, fields.isEmpty()?Cf.<List<String>>list(fieldTypes):MAP_RANGES.map(fields));
            }
//            if( p instanceof RDFaComplexProperty){
//                validate(((RDFaComplexProperty) p).entity,);
//            }
        }
    }

    private final static Function<SchemaField,List<String>> MAP_RANGES = new Function<SchemaField, List<String>>() {
        @Override
        public List<String> apply(final SchemaField schemaField) {
            return schemaField.ranges;
        }
    };

    private List<SchemaClass> makeAllowed(final Collection<String> ranges, final Map<String, SchemaClass> classMap, final SchemaClass... cls) {
        return Cu.join(Cu.project(classMap, ranges).values(), Cf.list(cls));
    }

    private boolean isRole(final List<SchemaClass> clss) {
        if (clss.isEmpty() || clss.size() > 1) {
            return false;
        }
        return isRole(clss.get(0));
    }

    private static Pattern start_with_schema = Pattern.compile("http?://(www.)?schema.org.*");

    private boolean isRole(final SchemaClass cls) {
        return start_with_schema.matcher(cls.name).matches() && cls.name.endsWith("Role");
    }

    public static void validateSimpleData(final RDFaEntity data, final List<RDFaException> exceptions, final RDFaValueProperty property, final String fieldName, final List<List<String>> fields) {
        String textValue = property.textValue;
        String hrefValue = property.hrefValue;
        String htmlValue = property.htmlValue;
        if (hrefValue == null) {
            hrefValue = "";
        }
        if (textValue == null) {
            textValue = hrefValue;
        }
        boolean good = false;
        List<RDFaException> exceptionsLocal = new ArrayList<RDFaException>();
        if (containsOnlyType(fields, "http://schema.org/Date")) {
            if (!ISO8601Validator.validate(textValue)) {
                exceptionsLocal.add(
                        new DatetimeFormatRDFaValidatorException(false, data, textValue + "$$" + fieldName, property.getLocation()));
            } else {
                good = true;
            }
        }
        if (containsOnlyType(fields, "http://schema.org/Number")) {
            if (!FLOAT_REGEXP.matcher(textValue).matches()) {
                exceptionsLocal.add(
                        new SchemaTypeRDFaValidatorException("integer", false, data, textValue + "$$" + fieldName, property.getLocation()));
            } else {
                good = true;
            }
        }
        if (containsOnlyType(fields, "http://schema.org/Float")) {
            if (!FLOAT_REGEXP.matcher(textValue).matches()) {
                exceptionsLocal.add(
                        new SchemaTypeRDFaValidatorException("float", false, data, textValue + "$$" + fieldName, property.getLocation()));
            } else {
                good = true;
            }
        }

        if (containsOnlyType(fields, "http://schema.org/Integer")) {
            if(fieldName.matches("https?://(www.)?schema.org/fileSize")){
                if(!FILESIZE_REGEXP.matcher(textValue).matches()){
                    exceptionsLocal.add(
                            new SchemaTypeRDFaValidatorException("number", false, data, textValue + "$$" + fieldName, property.getLocation()));
                }
            }
            else if (!INTEGER_REGEXP.matcher(textValue).matches()) {
                exceptionsLocal.add(
                        new SchemaTypeRDFaValidatorException("number", false, data, textValue + "$$" + fieldName, property.getLocation()));
            } else {
                good = true;
            }
        }
        if (containsOnlyType(fields, "http://schema.org/URL")) {
//            if (htmlValue.matches("^\\s*<\\s*meta.*")){
//                if (textValue.isEmpty()) {
//                    exceptionsLocal.add(
//                            new SchemaTypeRDFaValidatorException("url", false, data, hrefValue + "$$" + fieldName));
//                }
//            }
//            else
            if (hrefValue.isEmpty()) {
                exceptionsLocal.add(
                        new SchemaTypeRDFaValidatorException("url", false, data, hrefValue + "$$" + fieldName, property.getLocation()));
            } else {
                good = true;
            }
        }
        if (containsOnlyType(fields, "http://schema.org/Boolean")) {
            if (!ALLOWED_BOOL.contains(hrefValue) && !ALLOWED_BOOL.contains(textValue)) {
                exceptionsLocal.add(
                        new SchemaTypeRDFaValidatorException("boolean", false, data, textValue + "$$" + fieldName, property.getLocation()));
            } else {
                good = true;
            }
        }
        if(containsOnlyType(fields, "http://schema.org/Text")){
            good = true;
        }
        if(fieldName.equals("priceCurrency")){
            if(!ISO4217Validatior.validate(textValue)){
                exceptions.add(new SchemaTypeRDFaValidatorException("currency", false,data,
                        textValue + "$$" + fieldName, property.getLocation()));
            }
        }
        if (!good) {
            exceptions.addAll(exceptionsLocal);
        }
    }

    public static boolean isJSONLDValueProp(final RDFaProperty property) {
        if (property instanceof RDFaComplexProperty) {
            RDFaEntity entity = ((RDFaComplexProperty) property).entity;
            return entity instanceof JSONLDEntity && (entity.hasProperty("@value") || entity.hasProperty("@id") && !entity.hasProperty("@type"));
        }
        return false;
    }

    private final static Pattern FLOAT_REGEXP = Pattern.compile("^[-+]?[0-9]*\\.?[0-9]+([eE][-+]?[0-9]+)?$");
    private final static Pattern INTEGER_REGEXP = Pattern.compile("^[-+]?[0-9]+$");
    private final static Pattern FILESIZE_REGEXP = Pattern.compile("^[-+]?[0-9]+\\s*[gGkKmMгГмМкКtTтТ]?[bBбБ]?\\s*$");

    private static boolean containsOnlyType(final List<List<String>> types, final String type) {
        final Set<String> classNames = new HashSet<String>();
        for (final List<String> field : types) {
            classNames.addAll(transformTypes(field));
        }
        if(classNames.size() > 0) {
            for (String addType : MicrodataUtils.protocolAliases(type)) {
                if (!classNames.remove(addType)){
                    return false;
                }
            }
            return classNames.isEmpty();
        }
        return false;
    }

    public static Collection<? extends String> transformTypes(final List<String> ranges) {
        return TRANSFORM_TYPE.map(ranges);
    }

    private final static Map<String, String> TYPE_MAP = Cf.newConcurrentMap();

    static {
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#integer", "http://schema.org/Integer");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#int", "http://schema.org/Integer");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#long", "http://schema.org/Integer");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#boolean", "http://schema.org/Boolean");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#dateTime", "http://schema.org/Date");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#time", "http://schema.org/Date");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#double", "http://schema.org/Float");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#float", "http://schema.org/Float");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#decimal", "http://schema.org/Float");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#duration", "http://schema.org/Date");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#anyURI", "http://schema.org/URL");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#byte", "http://schema.org/Integer");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#negativeInteger", "http://schema.org/Integer");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#nonNegativeInteger", "http://schema.org/Integer");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#nonPositiveInteger", "http://schema.org/Integer");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#positiveInteger", "http://schema.org/Integer");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#short", "http://schema.org/Integer");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#unsignedLong", "http://schema.org/Integer");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#unsignedInt", "http://schema.org/Integer");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#unsignedShort", "http://schema.org/Integer");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#unsignedByte", "http://schema.org/Integer");
        TYPE_MAP.put("http://www.w3.org/2001/XMLSchema#unsignedInt", "http://schema.org/Integer");
        for(String key: TYPE_MAP.keySet()){
            TYPE_MAP.put(Su.replaceFirst("http://",key,"https://"),Su.replaceFirst("http://",TYPE_MAP.get(key),"https://"));
        }

    }


    private final static Function<String, String> TRANSFORM_TYPE = new Function<String, String>() {
        @Override
        public String apply(final String s) {
            return TYPE_MAP.containsKey(s) ? TYPE_MAP.get(s) : s;
        }
    };

    private boolean checkClass(final SchemaClass chkCls, final List<String> ranges) {
        for (final String range : ranges) {
            if (range.equals(chkCls.name)) {
                return true;
            }
        }
        if (chkCls != null) {
            for (final SchemaClass parent : chkCls.subClassOf) {
                if (checkClass(parent, ranges)) {
                    return true;
                }
            }
        }
        return false;
    }

}
