package ru.yandex.webmaster3.core.semantic.semantic_document_parser.struct_data;

import org.jetbrains.annotations.Nullable;
import org.json.JSONException;
import org.json.JSONObject;
import ru.yandex.common.util.Su;
import ru.yandex.common.util.collections.Cf;
import ru.yandex.common.util.collections.Cu;
import ru.yandex.common.util.collections.Either;
import ru.yandex.common.util.collections.Pair;
import ru.yandex.common.util.functional.Function;
import ru.yandex.common.util.functional.PartialFunction;
import ru.yandex.common.util.xml.Xmler;
import ru.yandex.webmaster3.core.semantic.review_business.auto.model.agg.AggAutoReview;
import ru.yandex.webmaster3.core.semantic.review_business.auto.model.agg.impl.wrapper.AggAutoReviewWrapper;
import ru.yandex.webmaster3.core.semantic.review_business.auto.model.impl.HProductAttributes;
import ru.yandex.webmaster3.core.semantic.review_business.auto.model.impl.json.HProductJsonConversions;
import ru.yandex.webmaster3.core.semantic.review_business.auto.model.impl.wrapper.HProductWrapper;
import ru.yandex.webmaster3.core.semantic.review_business.biz.model.BizReview;
import ru.yandex.webmaster3.core.semantic.review_business.biz.model.Tag;
import ru.yandex.webmaster3.core.semantic.review_business.biz.model.impl.wrapper.*;
import ru.yandex.webmaster3.core.semantic.review_business.model.Author;
import ru.yandex.webmaster3.core.semantic.review_business.model.impl.wrapper.AuthorWrapper;
import ru.yandex.webmaster3.core.semantic.review_business.util.DateHelper;
import ru.yandex.webmaster3.core.semantic.review_business.auto.model.HProduct;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.MicrodataUtils;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.classificator.ItemTypeClassificator;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.data.ComplexMicrodata;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.data.Microdata;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.data.MutableComplexMicrodataProxy;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.data.TextMicrodata;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.validators.AutoReviewCheckingMicrodataValidatorWrapper;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.validators.BusinessReviewCheckingMicrodataValidatorWrapper;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.validators.MicrodataValidator;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.validators.YandexSOrgReviewMicrodataValidator;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.data.MFAnyData;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.data.MicroformatData;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.data.property_types.TextData;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.data.property_types.URIData;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.spec.instances.HCard;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.FrontEnd;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.ogp.OGPUtils;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.ogp.data.OGPData;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.*;
import java.util.regex.Pattern;

import static ru.yandex.common.util.StringUtils.isEmpty;
import static ru.yandex.common.util.collections.CollectionFactory.newHashSet;
import static ru.yandex.common.util.collections.CollectionFactory.pair;
import static ru.yandex.common.util.xml.Xmler.tag;
import static ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.MicrodataUtils.extractNameList;
import static ru.yandex.webmaster3.core.semantic.semantic_document_parser.microdata.MicrodataUtils.extractStringList;
import static ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.MicroformatsUtils.*;

/**
 * Created by IntelliJ IDEA.
 * User: rasifiel
 * Date: 8/8/11
 * Time: 6:43 PM
 * To change this template use File | Settings | File Templates.
 */
public class StructuredDataTransformers {


    private FrontEnd frontEnd;

    public static Organization fromSchemaOrgOrganization(final ComplexMicrodata data, final String sourceUrl) {
        Pattern Prop_Pattern = Pattern.compile("https?://(www.)?schema.org/.*");
        if (data == null || data.getType() == null || !Prop_Pattern.matcher(data.getType()).matches()) {
            return null;
        }
        final String kind = MicrodataUtils.extractType(data.getType());
        final String rootClass = ItemTypeClassificator.getRoot(kind);
        if ("Organization".equals(rootClass) || "Place".equals(rootClass)) {
            final List<Microdata> addresses = data.getPropAsList("address");
            final String name = MicrodataUtils.getTextOrName(data, "name");
            final List<Microdata> urls = data.getPropAsList("url");
            final Microdata location = data.getFirst("location");
            final Microdata directGeo = data.getFirst("geo");
            final Microdata geo;
            if (directGeo != null) {
                geo = directGeo;
            } else if (location != null) {
                if (location instanceof ComplexMicrodata) {
                    final ComplexMicrodata locationAsComplex = (ComplexMicrodata) location;
                    geo = locationAsComplex.getFirst("geo");
                } else {
                    geo = null;
                }
            } else {
                geo = null;
            }
            final List<Microdata> emails = data.getPropAsList("email");
            final List<Microdata> phones = data.getPropAsList("telephone");
            final List<Microdata> faxes = data.getPropAsList("faxNumber");
            if (isEmpty(name)) {
                return null;
            }
            final List<Phone> resultPhones = new LinkedList<Phone>();
            for (final Microdata phone : phones) {
                if (phone instanceof TextMicrodata) {
                    resultPhones.add(new Phone("voice", phone.toString()));
                }
            }
            for (final Microdata fax : faxes) {
                if (fax instanceof TextMicrodata) {
                    resultPhones.add(new Phone("fax", fax.toString()));
                }
            }
            final List<String> resultEmails = new LinkedList<String>();
            for (final Microdata email : emails) {
                if (email instanceof TextMicrodata) {
                    resultEmails.add(email.toString());
                }
            }
            final List<String> resultUrls = new LinkedList<String>();
            for (final Microdata url : urls) {
                if (url instanceof TextMicrodata) {
                    resultUrls.add(url.toString());
                }
            }
            Geo resultGeo = null;
            if (geo instanceof ComplexMicrodata) {
                final ComplexMicrodata geoMD = (ComplexMicrodata) geo;
                final Microdata geoLat = geoMD.getFirst("latitude");
                final Microdata geoLong = geoMD.getFirst("longitude");
                if (geoLat != null && geoLong != null && geoLat instanceof TextMicrodata &&
                        geoLong instanceof TextMicrodata) {
                    resultGeo = new Geo(geoLat.toString(), geoLong.toString());
                }
            }
            final List<Address> resultAddresses = new LinkedList<Address>();
            for (final Microdata address : addresses) {
                if (address instanceof TextMicrodata) {
                    resultAddresses.add(new Address(address.toString()));
                } else if (address instanceof ComplexMicrodata) {
                    final ComplexMicrodata addressMD = (ComplexMicrodata) address;
                    List<String> adrphones = MicrodataUtils.extractStringList(addressMD, "telephone");
                    for (final String phone : adrphones) {
                        resultPhones.add(new Phone("voice", phone));
                    }
                    resultAddresses.add(new Address(addressMD.getFirstAsText("addressCountry"),
                            addressMD.getFirstAsText("addressLocality"), addressMD.getFirstAsText("addressRegion"),
                            addressMD.getFirstAsText("postalCode"), addressMD.getFirstAsText("streetAddress"), null,
                            addressMD.getFirstAsText("postOfficeBoxNumber")));
                }
            }
            final List<Pair<String, String>> optParts = new LinkedList<Pair<String, String>>();
            final String openingHours = data.getFirstAsText("openingHours");
            final String catId = ItemTypeClassificator.getCat(kind);
            optParts.add(pair("source-format", "schema.org"));
            return new Organization(name, resultUrls, resultPhones, resultEmails, resultGeo, resultAddresses,
                    openingHours, null, catId, null, null, null, null, null, null, null, null, null, null, null,
                    optParts, sourceUrl);
        }
        return null;
    }

    public static Organization fromDataVocabOrganization(final ComplexMicrodata data, final String sourceUrl) {

        Pattern data_voc = Pattern.compile("https?://data-vocabulary.org/.*");
        if (data.getType() == null || !data_voc.matcher(data.getType()).matches()) {
            return null;
        }
        final String kind = MicrodataUtils.extractType(data.getType());
        if ("Organization".equals(kind)) {
            final List<Microdata> addresses = Cf.list(data.getPropAsList("address"));
            addresses.addAll(data.getPropAsList("adr"));
            final List<Microdata> name = Cf.list(data.getPropAsList("name"));
            name.addAll(data.getPropAsList("fn"));
            final List<Microdata> urls = data.getPropAsList("url");
            final List<Microdata> geo = data.getPropAsList("geo");
            final List<Microdata> emails = data.getPropAsList("email");
            final List<Microdata> phones = data.getPropAsList("tel");
            if (name.isEmpty() || !(name.get(0) instanceof TextMicrodata)) {
                return null;
            }
            final List<Phone> resultPhones = new LinkedList<Phone>();
            for (final Microdata phone : phones) {
                if (phone instanceof TextMicrodata) {
                    resultPhones.add(new Phone("voice", phone.toString()));
                }
            }
            final List<String> resultEmails = new LinkedList<String>();
            for (final Microdata email : emails) {
                if (email instanceof TextMicrodata) {
                    resultEmails.add(email.toString());
                }
            }
            final List<String> resultUrls = new LinkedList<String>();
            for (final Microdata url : urls) {
                if (url instanceof TextMicrodata) {
                    resultUrls.add(url.toString());
                }
            }
            Geo resultGeo = null;
            if (!geo.isEmpty() && geo.get(0) instanceof ComplexMicrodata) {
                final ComplexMicrodata geoMD = (ComplexMicrodata) geo.get(0);
                final Microdata geoLat = geoMD.getFirst("latitude");
                final Microdata geoLong = geoMD.getFirst("latitude");
                if (geoLat != null && geoLong != null && geoLat instanceof TextMicrodata &&
                        geoLong instanceof TextMicrodata) {
                    resultGeo = new Geo(geoLat.toString(), geoLong.toString());
                }
            }
            final String resultName = name.get(0).toString();
            final List<Address> resultAddresses = new LinkedList<Address>();
            for (final Microdata address : addresses) {
                if (address instanceof TextMicrodata) {
                    resultAddresses.add(new Address(address.toString()));
                } else if (address instanceof ComplexMicrodata) {
                    final ComplexMicrodata addressMD = (ComplexMicrodata) address;
                    resultAddresses.add(
                            new Address(addressMD.getFirstAsText("country-name"), addressMD.getFirstAsText("locality"),
                                    addressMD.getFirstAsText("region"), addressMD.getFirstAsText("postal-code"),
                                    addressMD.getFirstAsText("street-address"), null, null));
                }
            }
            final List<Pair<String, String>> optParts = new LinkedList<Pair<String, String>>();
            optParts.add(pair("source-format", "datavocab"));
            return new Organization(resultName, resultUrls, resultPhones, resultEmails, resultGeo, resultAddresses,
                    null, null, data.getType(), null, null, null, null, null, null, null, null, null, null, null,
                    optParts, sourceUrl);
        }
        return null;
    }

    public static Organization fromOGPOrg(final OGPData data, final String sourceUrl) {
        if (!OGPUtils.isOrg(data.getFirstOrNull("og:type"))) {
            return null;
        }
        final List<Address> adr = new LinkedList<Address>();
        if (data.containsProp("og:street-address") || data.containsProp("og:locality") ||
                data.containsProp("og:region") ||
                data.containsProp("og:postal-code") || data.containsProp("og:country-name")) {
            adr.add(new Address(data.getFirstOrNull("og:country-name"), data.getFirstOrNull("og:locality"),
                    data.getFirstOrNull("og:region"), data.getFirstOrNull("og:postal-code"),
                    data.getFirstOrNull("og:street-address"), null, null));
        }
        Geo geo = null;
        if (data.containsProp("og:latitude") || data.containsProp("og:longitude")) {
            geo = new Geo(data.getFirstOrNull("og:latitude"), data.getFirstOrNull("og:longitude"));
        }
        final List<Phone> phones = new LinkedList<Phone>();
        if (data.containsProp("og:phone_number")) {
            phones.add(new Phone("voice", data.getFirstOrNull("og:phone_number")));
        }
        if (data.containsProp("og:fax_number")) {
            phones.add(new Phone("fax", data.getFirstOrNull("og:fax_number")));
        }
        final List<String> emails = data.getData("og:email");
        final List<String> urls = data.getData("og:url");
        final String cat = OGPUtils.getCat(data.getFirstOrNull("og:type"));
        final List<Pair<String, String>> optParts = new LinkedList<Pair<String, String>>();
        optParts.add(pair("source-format", "ogp"));
        return new Organization(data.getFirstOrNull("og:title"), urls, phones, emails, geo, adr, null, null, cat, null,
                null, null, null, null, null, null, null, null, null, null, optParts, sourceUrl);
    }


    public static Organization fromHCard(final MicroformatData data, final String sourceUrl) {
        if (!HCard.getInstance().isOrganization(data)) {
            return null;
        }
        return new Organization(extractPropertyToSingleString(data, "fn"), extractPropertyToListString(data, "url"),
                extractPropertyToList(data, "tel", new PartialFunction<MFAnyData, Phone>() {
                    @Override
                    public Phone apply(final MFAnyData tel) {
                        if (tel instanceof TextData || tel instanceof URIData) {
                            return new Phone("voice", tel.toString());
                        } else if (tel instanceof MicroformatData) {
                            final MicroformatData data = (MicroformatData) tel;
                            return new Phone(extractPropertyToSingleString(data, "type"),
                                    extractPropertyToSingleString(data, "value"));
                        }
                        throw new IllegalArgumentException();
                    }
                }), extractPropertyToListString(data, "email"),
                extractPropertyToSingle(data, "geo", new Function<MFAnyData, Geo>() {
                    @Override
                    public Geo apply(final MFAnyData geo) {
                        if (geo instanceof MicroformatData) {
                            final MicroformatData data = (MicroformatData) geo;
                            return new Geo(extractPropertyToSingleString(data, "latitude"),
                                    extractPropertyToSingleString(data, "longitude"));
                        }
                        return null;
                    }
                }), extractPropertyToList(data, "adr", new PartialFunction<MFAnyData, Address>() {
            @Override
            public Address apply(final MFAnyData adr) {
                if (adr instanceof TextData) {
                    return new Address(adr.toString());
                } else if (adr instanceof MicroformatData) {
                    final MicroformatData data = (MicroformatData) adr;
                    return new Address(extractPropertyToSingleString(data, "country-name"),
                            extractPropertyToSingleString(data, "locality"),
                            extractPropertyToSingleString(data, "region"),
                            extractPropertyToSingleString(data, "postal-code"),
                            extractPropertyToSingleString(data, "street-address"),
                            extractPropertyToSingleString(data, "extended-address"),
                            extractPropertyToSingleString(data, "post-office-box"));
                }
                throw new IllegalArgumentException();
            }
        }), extractPropertyToSingleString(data, "workhours"), extractPropertyToListString(data, "category"), null,
                extractPropertyToList(data, "agent"), extractPropertyToSingleString(data, "class"),
                extractPropertyToListString(data, "key"), extractPropertyToListString(data, "label"),
                extractPropertyToListString(data, "logo"), extractPropertyToListString(data, "mailer"),
                extractPropertyToListString(data, "note"), extractPropertyToListString(data, "photo"),
                extractPropertyToSingleString(data, "rev"), extractPropertyToListString(data, "title"),
                extractPropertyToListString(data, "rel-tag"),
                Cf.list(pair("raw", MFAnyData.asStringOrNull(data.getFirstOrNull("raw"))),
                        pair("source-format", "hcard")), sourceUrl);
    }

    public static Recipe fromHRecipe(final MicroformatData data, final String sourceUrl) {
        return new Recipe(extractPropertyToSingleString(data, "fn"),
                extractPropertyAndTextToListXML(data, "ingredient"), extractPropertyToSingleString(data, "category"),
                extractPropertyToSingle(data, "instructions", new Function<MFAnyData, List<String>>() {
                    @Override
                    public List<String> apply(final MFAnyData instructions) {
                        if (instructions == null) {
                            return null;
                        }
                        if (instructions instanceof MicroformatData) {
                            final MicroformatData data = (MicroformatData) instructions;
                            return extractPropertyToListString(data, "instruction");
                        } else {
                            return Cf.list(toExportString(instructions));
                        }
                    }
                }), extractPropertyToSingleString(data, "yield"), extractPropertyToSingleString(data, "duration"),
                extractPropertyToListString(data, "photo"), extractPropertyToSingleString(data, "result-photo"),
                extractPropertyToList(data, "author"), extractPropertyToListXML(data, "nutrition"),
                extractPropertyToSingleString(data, "sub-category"),
                extractPropertyToSingleString(data, "cuisine-type"), extractPropertyToSingleString(data, "weight"),
                sourceUrl);
    }

    private static final Map<String, String> HPRODUCT_FIELDS =
            Cu.zipMap(pair("prodyear", HProductAttributes.PRODYEAR), pair("body-type", HProductAttributes.BODY_TYPE),
                    pair("displacement", HProductAttributes.DISPLACEMENT),
                    pair("engine-type", HProductAttributes.ENGINE_TYPE),
                    pair("gear-type", HProductAttributes.GEAR_TYPE),
                    pair("transmission", HProductAttributes.TRANSMISSION),
                    pair("steering-wheel", HProductAttributes.STEERING_WHEEL),
                    pair("configuration-name", HProductAttributes.CONF_NAME));

    public static HProduct fromHProduct(final MicroformatData data, final String sourceUrl) {
        JSONObject identObj = new JSONObject();
        if (!"auto".equals(extractPropertyToSingleString(data, "category"))) {
            return null;
        }
        final List<MFAnyData> idents = data.getDataAsList("identifier");
        try {
            for (final MFAnyData ident : idents) {
                if (ident instanceof MicroformatData) {
                    MicroformatData md = (MicroformatData) ident;
                    String value = extractPropertyToSingleString(md, "value");
                    String type = extractPropertyToSingleString(md, "type");
                    if (HPRODUCT_FIELDS.containsKey(type)) {
                        identObj.put(HPRODUCT_FIELDS.get(type), value);
                    }
                }
            }
            HProductWrapper result = new HProductWrapper(HProductJsonConversions.fromJson(identObj));
            result = result.setFn(extractPropertyToSingleString(data, "fn")).setBrand(
                    extractPropertyToSingleString(data, "brand")).setUrl(extractPropertyToSingleString(data, "url"));
            /*if (result.getFn()==null)
                return null;*/
            return result;
        } catch (JSONException e) {
            throw new RuntimeException(e);
        }
/*
        return new Product(extractPropertyToListString(data, "category"), extractPropertyToListString(data, "photo"),
                extractPropertyToSingleString(data, "url"), extractPropertyToSingleString(data, "brand"),
                extractPropertyToSingleString(data, "fn"), extractPropertyAndTextToListXML(data, "identifier"),
                extractPropertyToSingleString(data, "price"), extractPropertyToSingleString(data, "description"));
*/
    }

    public static Review fromHReview(final MicroformatData data, final String sourceUrl) {
        return new ReviewBuilder().setSummaries(extractPropertyToListString(data, "summary")).setDescriptions(
                extractPropertyToListString(data, "description")).setPros(
                extractPropertyToListString(data, "pro")).setContras(
                extractPropertyToListString(data, "contra")).setDtreviewed(
                extractPropertyToSingleString(data, "dtreviewed")).setRatings(
                extractPropertyToList(data, "rating", new Function<MFAnyData, Rating>() {
                    @Override
                    public Rating apply(final MFAnyData arg) {
                        if (arg instanceof MicroformatData) {
                            final MicroformatData data = (MicroformatData) arg;
                            return new Rating(extractPropertyToSingleString(data, "value"),
                                    extractPropertyToSingleString(data, "worst"),
                                    extractPropertyToSingleString(data, "best"),
                                    extractPropertyToSingleString(data, "name"),
                                    extractPropertyToSingleString(data, "average"),
                                    extractPropertyToListString(data, "rel-tag"));
                        } else {
                            return new Rating(toExportString(arg), "1.0", "5.0", null, null, null);
                        }
                    }
                })).setPermalink(extractPropertyToSingleString(data, "permalink")).setReviewers(
                extractPropertyToList(data, "reviewer", new PartialFunction<MFAnyData, Author>() {
                    @Override
                    public Author apply(final MFAnyData arg) throws IllegalArgumentException {
                        if (arg instanceof TextData) {
                            return AuthorWrapper.newAuthor().setFn(arg.toString());
                        }
                        if (arg instanceof MicroformatData) {
                            MicroformatData data = (MicroformatData) arg;
                            final String url = extractPropertyToSingleString(data, "url");
                            final String permalink = extractPropertyToSingleString(data, "permalink");
                            return AuthorWrapper.newAuthor().setFn(extractPropertyToSingleString(data, "fn")).setEmail(
                                    extractPropertyToSingleString(data, "email")).setUrl(
                                    permalink != null ? permalink : url);
                        }
                        throw new IllegalArgumentException();
                    }
                })).setTypes(extractPropertyToListString(data, "type")).setItems(
                new Function<MicroformatData, Either<List<HProduct>, Organization>>() {

                    @Override
                    public Either<List<HProduct>, Organization> apply(final MicroformatData arg) {
                        final List<MFAnyData> items = arg.getDataAsList("item");
                        if (!items.isEmpty()) {
                            if (items.get(0) instanceof MicroformatData &&
                                    items.get(0).getSpec() == HCard.getInstance()) {
                                return Either.right(fromHCard((MicroformatData) items.get(0), sourceUrl));
                            }
                        }
                        return Either.left(
                                extractPropertyToList(data, "item", new PartialFunction<MFAnyData, HProduct>() {
                                    @Override
                                    public HProduct apply(final MFAnyData arg) throws IllegalArgumentException {
                                        if (arg instanceof MicroformatData) {
                                            return fromHProduct((MicroformatData) arg, sourceUrl);
                                        }
                                        throw new IllegalArgumentException();
                                    }
                                }));
                    }
                }.apply(data)).setOwningTime(extractPropertyToSingleString(data, "owning-time")).setReviewsurls(
                extractPropertyToListString(data, "reviewsurl")).setTag(
                extractPropertyToList(data, "tag", new Function<MFAnyData, Rating>() {
                    @Override
                    public Rating apply(final MFAnyData arg) {
                        if (arg instanceof MicroformatData) {
                            final MicroformatData data = (MicroformatData) arg;
                            return new Rating(extractPropertyToSingleString(data, "value"),
                                    extractPropertyToSingleString(data, "worst"),
                                    extractPropertyToSingleString(data, "best"),
                                    extractPropertyToSingleString(data, "name"),
                                    extractPropertyToSingleString(data, "average"),
                                    extractPropertyToListString(data, "rel-tag"));
                        } else {
                            return new Rating(toExportString(arg), "1.0", "5.0", null, null, null);
                        }
                    }
                })).setPhotos(extractPropertyToListString(data, "photo")).setTags(
                extractPropertyToListString(data, "rel-tag")).setBookmarks(
                extractPropertyToListString(data, "rel-bookmark")).setLicense(
                extractPropertyToSingleString(data, "rel-license")).setSourceUrl(sourceUrl).createReview();
    }

    public static BizReview fromBizHReview(final MicroformatData data, final String sourceUrl) {
        final Review review = fromHReview(data, sourceUrl);
        if (review != null && !review.getItems().isLeftNotRight()) {
            return fromOrgReview(review, sourceUrl);
        } else {
            return null;
        }
    }

    public static AggAutoReview fromAutoHReview(final MicroformatData data, final String sourceUrl) {
        final Review review = fromHReview(data, sourceUrl);
        if (review != null && review.getItems().isLeftNotRight()) {
            return fromAutoReview(review, sourceUrl);
        } else {
            return null;
        }
    }

    private static AggAutoReview fromAutoReview(final Review review, final String sourceUrl) {
        if (review == null || review.getItems().asLeft().isEmpty()) {
            return null;
        }
        HProduct item = review.getItems().asLeft().get(0);
        if (item == null) {
            return null;
        }
        AggAutoReviewWrapper result = AggAutoReviewWrapper.newReview();
        result = result.setReviewedDate(DateHelper.read(review.getDtreviewed())).setSummary(
                review.getSummaries().isEmpty() ? null : Su.join(review.getSummaries(), "")).setUrl(
                review.getPermalink()).setPros(review.getPros()).setContras(review.getContras()).setDescription(
                review.getDescriptions().isEmpty() ? null : Su.join(review.getDescriptions(), "")).setItem(
                item).setReviewer(review.getReviewers().isEmpty() ? null : review.getReviewers().get(0)).setReviewsUrl(
                review.getReviewsurls());
        if (result.getReviewer() == null || result.getReviewedDate() == null || result.getType() == null) {
            return null;
        }
        if (!review.getRatings().isEmpty()) {
            final Rating structRating = review.getRatings().get(0);
            if (structRating != null) {
                {
                    final Float best = toFloat(structRating.getBest());
                    if (best != null) {
                        result = result.setBestRating(best);
                    }
                    final Float worst = toFloat(structRating.getWorst());
                    if (worst != null) {
                        result = result.setWorstRating(worst);
                    }
                    final Float value = toFloat(structRating.getValue());
                    if (value != null) {
                        result = result.setRating(value);
                    }
                }
            }
        }

        return result;
    }

    public static BizReview fromOrgReview(final Review review, final String sourceUrl) {
        if (review == null) {
            return null;
        }
        //check for bizreview, not auto review
        if (review.getItems().isLeftNotRight() || review.getItems().asRight() == null) {
            return null;
        }
        BizReviewWrapper result = BizReviewWrapper.newBizReview();
        result = result.setReviewedDate(DateHelper.read(review.getDtreviewed())).setSummary(
                review.getSummaries().isEmpty() ? null : Su.join(review.getSummaries(), "")).setUrl(
                review.getPermalink()).setPros(review.getPros()).setContras(review.getContras()).setDescription(
                review.getDescriptions().isEmpty() ? null : Su.join(review.getDescriptions(), "")).setItem(
                ORG_TO_REVIEW_HCARD.apply(review.getItems().asRight())).setTags(
                RATING_TO_REVIEW_TAG.map(review.getTag())).setReviewer(
                review.getReviewers().isEmpty() ? null : review.getReviewers().get(0));
        if (result.getUrl() == null) {
            return null;
        }
        if (!review.getRatings().isEmpty()) {
            final Rating structRating = review.getRatings().get(0);
            if (structRating != null) {
                {
                    final Float best = toFloat(structRating.getBest());
                    if (best != null) {
                        result = result.setBestRating(best);
                    }
                    final Float worst = toFloat(structRating.getWorst());
                    if (worst != null) {
                        result = result.setWorstRating(worst);
                    }
                    final Float value = toFloat(structRating.getValue());
                    if (value != null) {
                        result = result.setRating(value);
                    }
                }
            }
        }

        return result;
    }

    private static String getFirst(final List<String> list) {
        return !list.isEmpty() ? list.get(0) : null;
    }

    private static final String[] CONTENT_TYPES =
            {"carbohydrate", "cholesterol", "fat", "fiber", "protein", "saturatedFat", "sodium", "sugar", "transFat",
                    "unsaturatedFat"};

    public static Recipe fromSchemaOrgRecipe(final ComplexMicrodata microdata, final String sourceUrl) {
        Pattern Prop_Pattern = Pattern.compile("https?://(www.)?schema.org/Recipe");
        if (microdata.getType() == null || !Prop_Pattern.matcher(microdata.getType()).matches()) {
            return null;
        }
        final String fn = microdata.getFirstAsText("name");
        final List<String> ingredientsRaw = extractNameList(microdata, "ingredients");
        final List<Xmler.Tag> ingredients = new LinkedList<Xmler.Tag>();
        for (final String ingredientRaw : ingredientsRaw) {
            ingredients.add(tag("ingredient", tag("name", ingredientRaw)));
        }
        final String category = microdata.getFirstAsText("recipeCategory");
        final List<String> instructions = extractStringList(microdata, "recipeInstructions");
        final String yield = microdata.getFirstAsText("recipeYield");
        final String totalTime = microdata.getFirstAsText("totalTime");
        final String duration;
        if (totalTime != null) {
            duration = totalTime;
        } else {
            duration = microdata.getFirstAsText("cookTime");
        }
        final List<String> photos = extractStringList(microdata, "image");
        final List<Microdata> authorsMD = microdata.getPropAsList("author");
        final String resultPhoto = microdata.getFirstAsText("resultPhoto");
        final List<Either<Xmler.Tag, String>> authors = new LinkedList<Either<Xmler.Tag, String>>();
        for (final Microdata authorMD : authorsMD) {
            if (authorMD instanceof TextMicrodata) {
                authors.add(Either.<Xmler.Tag, String>right(authorMD.toString()));
            } else {
                authors.add(Either.<Xmler.Tag, String>left(((ComplexMicrodata) authorMD).toXmlTag("author")));
            }
        }
        final List<Xmler.Tag> nutritions = new LinkedList<Xmler.Tag>();
        final Microdata nutritionMD = microdata.getFirst("nutrition");
        if (nutritionMD instanceof ComplexMicrodata) {
            final ComplexMicrodata nutritionCMD = (ComplexMicrodata) nutritionMD;
            for (final String content_key : CONTENT_TYPES) {
                final String contentVal = nutritionCMD.getFirstAsText(content_key + "Content");
                if (contentVal != null) {
                    nutritions.add(tag(content_key, contentVal));
                }
            }
        }
        final String cuisine = microdata.getFirstAsText("recipeCuisine");
        return new Recipe(fn, ingredients, category, instructions, yield, duration, photos, resultPhoto, authors,
                nutritions, null, cuisine, null, sourceUrl);
    }

    public static MusicGroup fromSchemaOrgMusicGroup(final ComplexMicrodata microdata, final String sourceUrl) {
        Pattern Prop_Pattern = Pattern.compile("https?://(www.)?schema.org/MusicGroup");
        if (microdata.getType() == null || !Prop_Pattern.matcher(microdata.getType()).matches()) {
            return null;
        }
        final String name = microdata.getFirstAsText("name");
        final String image = microdata.getFirstAsText("image");
        final String playCount;
        if (microdata.getFirstAsText("playCount") != null) {
            playCount = microdata.getFirstAsText("playCount");
        } else {
            final String interactioncount = microdata.getFirstAsText("interactionCount");
            if (interactioncount != null && interactioncount.startsWith("UserPlays:")) {
                final String plays = interactioncount.substring("UserPlays:".length());
                playCount = plays;
            } else {
                playCount = null;
            }
        }
        final List<MusicRecording> tracks = new LinkedList<MusicRecording>();
        for (Microdata track : microdata.getPropAsList("tracks")) {
            if (track instanceof ComplexMicrodata) {
                MusicRecording trackRecording = fromSchemaOrgMusicRecording((ComplexMicrodata) track, sourceUrl);
                if (trackRecording != null) {
                    tracks.add(trackRecording);
                }
            }
        }
        return new MusicGroup(name, image, playCount, tracks, sourceUrl);
    }

    public static MusicRecording fromSchemaOrgMusicRecording(final ComplexMicrodata microdata, final String sourceUrl) {
        Pattern Prop_Pattern = Pattern.compile("https?://(www.)?schema.org/MusicRecording");
        if (microdata.getType() == null || !Prop_Pattern.matcher(microdata.getType()).matches()) {
            return null;
        }
        final String name = microdata.getFirstAsText("name");
        if (name == null) {
            return null;
        }
        final String url = microdata.getFirstAsText("url");
        final String inAlbum = microdata.getFirstAsText("inAlbum");
        final String duration = microdata.getFirstAsText("duration");
        final String playCount;
        if (microdata.getFirstAsText("playCount") != null) {
            playCount = microdata.getFirstAsText("playCount");
        } else {
            final String interactioncount = microdata.getFirstAsText("interactionCount");
            if (interactioncount != null && interactioncount.startsWith("UserPlays:")) {
                final String plays = interactioncount.substring("UserPlays:".length());
                playCount = plays;
            } else {
                playCount = null;
            }
        }
        return new MusicRecording(name, duration, inAlbum, url, playCount);
    }

    public static EncArticle fromYandexMicrodataEncArticle(final ComplexMicrodata microdata, final String sourceUrl) {
        Pattern Prop_Pattern = Pattern.compile("https?://webmaster.yandex.ru/vocabularies/enc-article.xml");
        if (microdata.getType() == null || !Prop_Pattern.matcher(microdata.getType()).matches()) {
            return null;
        }
        return new EncArticle(microdata.getFirstAsText("title"), microdata.getFirstAsHtml("content") == null ? null :
                Su.leaveOnlyOneSpaceCharBetweenWords(microdata.getFirstAsHtml("content")),
                MicrodataUtils.extractStringList(microdata, "category"), microdata.getFirstAsText("references"),
                microdata.getFirstAsText("author"), microdata.getFirstAsText("source"),
                microdata.getFirstAsText("source-date"), MicrodataUtils.extractStringList(microdata, "photo"),
                sourceUrl, "enc-article");
    }

    public static TermDef fromYandexMicrodataTermDef(final ComplexMicrodata microdata, final String sourceUrl) {
        Pattern Prop_Pattern = Pattern.compile("https?://webmaster.yandex.ru/vocabularies/term-def.xml");
        if (microdata.getType() == null || !Prop_Pattern.matcher(microdata.getType()).matches()) {
            return null;
        }

        return new TermDef(microdata.getFirstAsText("term"), MicrodataUtils.extractHtmlList(microdata, "definition"),
                microdata.getFirstAsText("author"), microdata.getFirstAsText("source"),
                microdata.getFirstAsText("source-date"), sourceUrl);
    }

    public static EncArticle fromSchemaOrgScholarlyArticle(final ComplexMicrodata microdata, final String sourceUrl) {
        Pattern Prop_Pattern = Pattern.compile("https?://(www.)?schema.org/ScholarlyArticle");
        if (microdata.getType() == null || !Prop_Pattern.matcher(microdata.getType()).matches()) {
            return null;
        }
        return new EncArticle(microdata.getFirstAsText("headline"), microdata.getFirstAsHtml("articleBody"),
                MicrodataUtils.extractStringList(microdata, "articleSection"), microdata.getFirstAsText("references"),
                MicrodataUtils.getTextOrName(microdata, "author"), microdata.getFirstAsText("source"),
                microdata.getFirstAsText("datePublished"), MicrodataUtils.extractStringList(microdata, "image"),
                sourceUrl, "scholarly-article");
    }

    public static Movie fromSchemaOrgMovie(final ComplexMicrodata microdata, final @Nullable String sourceUrl, final @Nullable String zoraLang) {
        Pattern Prop_Pattern = Pattern.compile("https?://(www.)?schema.org/Movie");
        if (microdata.getType() == null || !Prop_Pattern.matcher(microdata.getType()).matches()) {
            return null;
        }
        return new MovieBuilder().setName(microdata.getFirstAsText("name")).setDescription(
                MicrodataUtils.extractStringList(microdata, "description")).setActors(
                Cu.join(MicrodataUtils.extractNameHrefList(microdata, "actors"),
                        MicrodataUtils.extractNameHrefList(microdata, "actor"))).setDirector(
                MicrodataUtils.getNameHrefPair(microdata, "director")).setDuration(
                microdata.getFirstAsText("duration")).setMusicBy(
                MicrodataUtils.getNameHrefPair(microdata, "musicBy")).setProducer(
                MicrodataUtils.getNameHrefPair(microdata, "producer")).setProductionCompany(
                MicrodataUtils.getNameHrefPair(microdata, "productionCompany")).setTrailer(
                microdata.getFirstAsText("trailer")).setSourceUrl(sourceUrl).setAlternativeHeadline(
                microdata.getFirstAsText("alternativeHeadline")).setAwards(
                Cu.join(MicrodataUtils.extractNameHrefList(microdata, "award"),
                        MicrodataUtils.extractNameHrefList(microdata, "awards"))).setContentRating(
                MicrodataUtils.getNameHrefPair(microdata, "contentRating")).setDateCreated(
                microdata.getFirstAsText("dateCreated")).setDateModified(
                microdata.getFirstAsText("dateModified")).setDatePublished(
                microdata.getFirstAsText("datePublished")).setFamilyFriendly(
                MicrodataUtils.getNameHrefPair(microdata, "isFamilyFriendly")).setGenre(
                MicrodataUtils.extractNameHrefList(microdata, "genre")).setInLanguage(
                MicrodataUtils.getNameHrefPair(microdata, "inLanguage")).setVersion(
                MicrodataUtils.getNameHrefPair(microdata, "version")).setLang(zoraLang).createMovie();
    }

    public static AggregateRating fromSchemaOrgAggregateRating(final ComplexMicrodata microdata, final String sourceUrl) {
        Pattern Agg_Rate = Pattern.compile("https?://(www.)?schema.org/AggregateRating");
        if (microdata.getType() == null || !Agg_Rate.matcher(microdata.getType()).matches()) {
            return null;
        }
        return new AggregateRatingBuilder().setBestRating(microdata.getFirstAsText("bestRating")).setRatingCount(
                microdata.getFirstAsText("ratingCount")).setRatingValue(
                microdata.getFirstAsText("ratingValue")).setReviewCount(
                microdata.getFirstAsText("reviewCount")).setSourceUrl(sourceUrl).
                setWorstRating(microdata.getFirstAsText("worstRating")).createAggregateRating();
    }

    private static Set<String> CREATIVE_WORK_TYPES = newHashSet(
            Arrays.asList("CreativeWork", "Article", "BlogPosting", "NewsArticle", "ScholarlyArticle",
                    "MedicalScholarlyArticle", "Blog", "Book", "Diet", "ExercisePlan", "MediaObject", "AudioObject",
                    "ImageObject", "MusicVideoObject", "VideoObject", "Painting", "Photograph", "Sculpture",
                    "TVEpisode", "TVSeason", "TVSeries"));

    public static CreativeWork fromSchemaOrgCreativeWork(final ComplexMicrodata microdata, final String sourceUrl, String zoraLang) {
        if (microdata.getType() == null) {
            return null;
        }
        String clearType= microdata.getType();
        for(String prefix : MicrodataUtils.protocolAliases("http://schema.org")) {
            clearType = clearType.replace(prefix, "");
        }
        if (!CREATIVE_WORK_TYPES.contains(clearType)) {
            return null;
        }
        return new CreativeWork.CreativeWorkBuilder().setSourceUrl(sourceUrl).setLang(zoraLang).setAuthor(
                MicrodataUtils.extractNameList(microdata, "author")).setDescription(
                MicrodataUtils.extractStringList(microdata, "description")).setGenre(
                MicrodataUtils.extractNameList(microdata, "genre")).setHeadline(
                microdata.getFirstAsText("headline")).setName(microdata.getFirstAsText("name")).setMainType(
                clearType).createCreativeWork();
    }

    private final static MicrodataValidator BIZREVIEW_VALIDATOR = new BusinessReviewCheckingMicrodataValidatorWrapper(
            Cf.<MicrodataValidator>list(new YandexSOrgReviewMicrodataValidator()));

    public static BizReview fromSchemaOrgReview(final ComplexMicrodata microdata, final String sourceUrl) {
        if (!BIZREVIEW_VALIDATOR.validate(microdata).isEmpty()) {
            return null;
        }
        try {
            BizReviewWrapper review;
            try {
                review = BizReviewWrapper.newBizReview().setItem(ORG_TO_REVIEW_HCARD.apply(
                        fromSchemaOrgOrganization((ComplexMicrodata) microdata.getFirst("itemReviewed"), sourceUrl)));
            } catch (IllegalArgumentException e) {
                return null;
            } catch (ClassCastException e) {
                return null;
            }
            review = review.setUrl(microdata.getFirstAsText("url"));
            if (review.getUrl() == null) {
                return null;
            }
            review = review.setPros(MicrodataUtils.extractStringList(microdata, "pro"));
            review = review.setContras(MicrodataUtils.extractStringList(microdata, "contra"));
            review = review.setDescription(microdata.getFirstAsText("reviewBody"));
            if (review.getDescription() == null) {
                review = review.setDescription(microdata.getFirstAsText("description"));
            }
            review = review.setSummary(microdata.getFirstAsText("summary"));
            review = review.setReviewsUrl(microdata.getFirstAsText("reviewsUrl"));
            Microdata author = microdata.getFirst("author");
            try {
                review = review.setReviewer(SCHEMA_PERSON_TO_REVIEW_AUTHOR.apply(author));
            }
            catch (IllegalArgumentException ignored){
            }
            final Microdata rating = microdata.getFirst("reviewRating");
            if (rating != null) {
                if (rating instanceof ComplexMicrodata) {
                    final Rating structRating = SCHEMA_RATING_TO_REVIEW_RATING.apply((ComplexMicrodata) rating);
                    if (structRating != null) {
                        final Float best = toFloat(structRating.getBest());
                        if (best != null) {
                            review = review.setBestRating(best);
                        }
                        final Float worst = toFloat(structRating.getWorst());
                        if (worst != null) {
                            review = review.setWorstRating(worst);
                        }
                        final Float value = toFloat(structRating.getValue());
                        if (value != null) {
                            review = review.setRating(value);
                        }
                    }
                } else {
                    final Float value = toFloat(rating.toString());
                    if (value != null) {
                        review = review.setRating(value);
                    }
                }
            }
            final Microdata dateMD = microdata.getFirst("datePublished");
            if (dateMD != null) {
                final Date date = DateHelper.read(dateMD.toString());
                review = review.setReviewedDate(date);
            }
            final Microdata visitDateMD = microdata.getFirst("dateVisited");
            if (visitDateMD != null) {
                review = review.setVisitedDate(visitDateMD.toString());
            }

            review =
                    review.setTags(Cu.mapWhereDefined(SCHEMA_RATING_TAG_TO_REVIEW_TAG, microdata.getPropAsList("tag")));
            return review;
        } catch (IllegalArgumentException e) {
            return null;
        }
    }

    private static Float toFloat(final String best) {
        if (best == null) {
            return null;
        }
        try {
            return Float.valueOf(best.replace(',', '.'));
        } catch (Exception e) {
            return null;
        }
    }

    private static final PartialFunction<Rating, Tag> RATING_TO_REVIEW_TAG = new PartialFunction<Rating, Tag>() {
        @Override
        public Tag apply(final Rating arg) throws IllegalArgumentException {
            TagWrapper tag = TagWrapper.newTag();
            tag = tag.setName(arg.getName());
            final Float val = toFloat(arg.getValue());
            if (val == null) {
                throw new IllegalArgumentException("Wrong rating value");
            }
            Float best = toFloat(arg.getBest());
            best = best == null ? 5 : best;
            Float worst = toFloat(arg.getWorst());
            worst = worst == null ? 1 : worst;
            if (val < worst || val > best || worst.equals(best)) {
                throw new IllegalArgumentException("Value out of bound");
            }
            if ((best == 1) && (worst == -1)) {
                tag = tag.setValue(Float.toString(Math.round(val))).setType(Tag.Type.ATTITUDE);
            } else if (best == 5 && worst == 1) {
                tag = tag.setValue(Float.toString(Math.round(val))).setType(Tag.Type.RATING5);
            } else {
                final float normalizedValue = Math.round((val - worst) / (best - worst) * 9) + 1;
                tag = tag.setValue(Float.toString(normalizedValue)).setType(Tag.Type.RATING10);
            }
            return tag;
        }
    };

    private static final PartialFunction<Microdata, Tag> SCHEMA_RATING_TAG_TO_REVIEW_TAG =
            new PartialFunction<Microdata, Tag>() {
                @Override
                public Tag apply(final Microdata arg) throws IllegalArgumentException {
                    if (!(arg instanceof ComplexMicrodata)) {
                        final Float val = toFloat(arg.toString());
                        if (val != null) {
                            final String roundedVal = Float.toString(Math.round(val));
                            if (TagWrapper.Type.RATING5.validateValue(roundedVal)) {
                                return TagWrapper.newTag().setType(Tag.Type.RATING5).setValue(roundedVal);
                            }
                        }
                        throw new IllegalArgumentException("Wrong rating value");
                    } else {
                        TagWrapper tag = TagWrapper.newTag();
                        tag = tag.setName(((ComplexMicrodata) arg).getFirstAsText("ratingTarget"));
                        final Float val = toFloat(((ComplexMicrodata) arg).getFirstAsText("ratingValue"));
                        if (val == null) {
                            throw new IllegalArgumentException("Wrong rating value");
                        }
                        Float best = toFloat(((ComplexMicrodata) arg).getFirstAsText("bestRating"));
                        best = best == null ? 5 : best;
                        Float worst = toFloat(((ComplexMicrodata) arg).getFirstAsText("worstRating"));
                        worst = worst == null ? 1 : worst;
                        if (val < worst || val > best || worst.equals(best)) {
                            throw new IllegalArgumentException("Value out of bound");
                        }
                        if ((best == 1) && (worst == -1)) {
                            tag = tag.setValue(Float.toString(Math.round(val))).setType(Tag.Type.ATTITUDE);
                        } else if (best == 5 && worst == 1) {
                            tag = tag.setValue(Float.toString(Math.round(val))).setType(Tag.Type.RATING5);
                        } else {
                            final float normalizedValue = Math.round((val - worst) / (best - worst) * 9) + 1;
                            tag = tag.setValue(Float.toString(normalizedValue)).setType(Tag.Type.RATING10);
                        }
                        return tag;
                    }
                }
            };

    private static final Function<ComplexMicrodata, Rating> SCHEMA_RATING_TO_REVIEW_RATING =
            new Function<ComplexMicrodata, Rating>() {
                @Override
                public Rating apply(final ComplexMicrodata arg) {
                    return new Rating(arg.getFirstAsText("ratingValue"), arg.getFirstAsText("worstRating"),
                            arg.getFirstAsText("bestRating"), null, null, Collections.<String>emptyList());
                }
            };

    private static final PartialFunction<Microdata, Author> SCHEMA_PERSON_TO_REVIEW_AUTHOR =
            new PartialFunction<Microdata, Author>() {
                @Override
                public Author apply(final Microdata arg) throws IllegalArgumentException {
                    if (arg instanceof ComplexMicrodata) {
                        return AuthorWrapper.newAuthor().setFn(
                                ((ComplexMicrodata) arg).forceFirstAsText("name")).setUrl(
                                ((ComplexMicrodata) arg).getFirstAsText("url"));
                    } else if (arg instanceof TextMicrodata) {
                        return AuthorWrapper.newAuthor().setFn(((TextMicrodata) arg).toDataString());
                    } else {
                        throw new IllegalArgumentException();
                    }
                }
            };

    private static final PartialFunction<Organization, ru.yandex.webmaster3.core.semantic.review_business.biz.model.HCard> ORG_TO_REVIEW_HCARD =
            new PartialFunction<Organization, ru.yandex.webmaster3.core.semantic.review_business.biz.model.HCard>() {
                @Override
                public ru.yandex.webmaster3.core.semantic.review_business.biz.model.HCard apply(final Organization arg) throws IllegalArgumentException {
                    try {
                        HCardWrapper card = HCardWrapper.newHCard().setFn(arg.getName()).setAddresses(
                                ORG_ADDRESS_TO_REVIEW_ADDRESS.map(arg.getAddresses()));
                        if (arg.getGeo() != null) {
                            try {
                                card = card.setGeo(GeoWrapper.newGeo().set(toFloat(arg.getGeo().getLatitude()),
                                        toFloat(arg.getGeo().getLongitude())));
                            } catch (Exception e) {
                                //failed geo parsing, ignore
                            }
                        }
                        card = card.setWorkHours(arg.getWorkhours());
                        if (!arg.getEmails().isEmpty()) {
                            card = card.setEmail(arg.getEmails().get(0));
                        }
                        if (!arg.getUrls().isEmpty()) {
                            card = card.setUrl(arg.getUrls().get(0));
                        }
                        if (!arg.getPhones().isEmpty()) {
                            card = card.setPhoneNumber(arg.getPhones().get(0).getPhone());
                        }
                        if (!arg.getPhotos().isEmpty()) {
                            card = card.setPhotoUrl(arg.getPhotos().get(0));
                        }
                        if (!arg.getLogos().isEmpty()) {
                            card = card.setLogoUrl(arg.getLogos().get(0));
                        }
                        return card;
                    } catch (Exception e) {
                        throw new IllegalArgumentException(e);
                    }
                }
            };

    private static final PartialFunction<Address, ru.yandex.webmaster3.core.semantic.review_business.biz.model.Address>
            ORG_ADDRESS_TO_REVIEW_ADDRESS =
            new PartialFunction<Address, ru.yandex.webmaster3.core.semantic.review_business.biz.model.Address>() {
                @Override
                public ru.yandex.webmaster3.core.semantic.review_business.biz.model.Address apply(final Address arg) throws IllegalArgumentException {
                    if (arg.getOnlyString() != null) {
                        return AddressWrapper.newAddress().setStreetAddress(arg.getOnlyString());
                    }
                    return AddressWrapper.newAddress().setCountryName(arg.getCountry()).setExtendedAddress(
                            arg.getExtended()).setLocality(arg.getLocality()).setPostalCode(
                            arg.getPostalCode()).setRegion(arg.getRegion()).setStreetAddress(arg.getStreet());
                }
            };

    public static List<BizReview> reviewsFromOrg(final ComplexMicrodata data, final String url) {
        final List<BizReview> result = new ArrayList<BizReview>();
        Pattern Prop_Pattern = Pattern.compile("https?://(www.)?schema.org/.*");
        if (data == null || data.getType() == null || !Prop_Pattern.matcher(data.getType()).matches()) {
            return Collections.emptyList();
        }
        final String kind = MicrodataUtils.extractType(data.getType());
        final String rootClass = ItemTypeClassificator.getRoot(kind);
        if ("Organization".equals(rootClass) || "Place".equals(rootClass)) {
            List<Microdata> reviews = data.getPropAsList("review");
            for (final Microdata review : reviews) {
                if (review instanceof ComplexMicrodata) {
                    if (((ComplexMicrodata) review).getPropAsList("itemReviewed").isEmpty()) {
                        MutableComplexMicrodataProxy reviewProxy =
                                new MutableComplexMicrodataProxy((ComplexMicrodata) review);
                        reviewProxy.addPropValue("itemReviewed", data);
                        BizReview resReview = fromSchemaOrgReview(reviewProxy, url);
                        if (resReview != null) {
                            result.add(resReview);
                        }
                    }
                }
            }
        }
        return result;
    }

    private final static MicrodataValidator AUTOREVIEW_VALIDATOR = new AutoReviewCheckingMicrodataValidatorWrapper(
            Cf.<MicrodataValidator>list(new YandexSOrgReviewMicrodataValidator()));

    public static AggAutoReviewWrapper fromSchemaOrgAutoReview(final ComplexMicrodata microdata, final String sourceUrl) {
        if (!AUTOREVIEW_VALIDATOR.validate(microdata).isEmpty()) {
            return null;
        }
        try {
            AggAutoReviewWrapper review;
            try {
                review = AggAutoReviewWrapper.newReview().setItem(
                        AUTO_TO_REVIEW_HPRODUCT.apply(((ComplexMicrodata) microdata.getFirst("itemReviewed"))));
            } catch (IllegalArgumentException e) {
                return null;
            } catch (ClassCastException e) {
                return null;
            }
            review = review.setUrl(microdata.getFirstAsText("url"));
            review = review.setPros(MicrodataUtils.extractStringList(microdata, "pro"));
            review = review.setContras(MicrodataUtils.extractStringList(microdata, "contra"));
            review = review.setDescription(microdata.getFirstAsText("reviewBody"));
            if (review.getDescription() == null) {
                review = review.setDescription(microdata.getFirstAsText("description"));
            }
            review = review.setSummary(microdata.getFirstAsText("name"));
            Microdata author = microdata.getFirst("author");
            try {
                review = review.setReviewer(SCHEMA_PERSON_TO_REVIEW_AUTHOR.apply(author));
            } catch (IllegalArgumentException e) {
                //ignore
            }
            final Microdata rating = microdata.getFirst("reviewRating");
            if (rating != null) {
                if (rating instanceof ComplexMicrodata) {
                    final Rating structRating = SCHEMA_RATING_TO_REVIEW_RATING.apply((ComplexMicrodata) rating);
                    if (structRating != null) {
                        final Float best = toFloat(structRating.getBest());
                        if (best != null) {
                            review = review.setBestRating(best);
                        }
                        final Float worst = toFloat(structRating.getWorst());
                        if (worst != null) {
                            review = review.setWorstRating(worst);
                        }
                        final Float value = toFloat(structRating.getValue());
                        if (value != null) {
                            review = review.setRating(value);
                        }
                    }
                } else {
                    final Float value = toFloat(rating.toString());
                    if (value != null) {
                        review = review.setRating(value);
                    }
                }
            }
            final Microdata dateMD = microdata.getFirst("datePublished");
            if (dateMD != null) {
                final Date date = DateHelper.read(dateMD.toString());
                review = review.setReviewedDate(date);
            }

            return review;
        } catch (IllegalArgumentException e) {
            return null;
        }
    }

    private static final PartialFunction<ComplexMicrodata, HProduct> AUTO_TO_REVIEW_HPRODUCT =
            new PartialFunction<ComplexMicrodata, HProduct>() {

                @Override
                public HProduct apply(final ComplexMicrodata arg) throws IllegalArgumentException {
                    if (arg == null) {
                        throw new IllegalArgumentException();
                    }
                    if (!"http://schema.org/Auto".equals(arg.getType()) && !"http://schema.org/Car".equals(arg.getType())) {
                        throw new IllegalArgumentException();
                    }
                    HProductWrapper x =
                            HProductWrapper.newHProduct().setBodyType(arg.forceFirstAsText("bodyType")).setBrand(
                                    arg.forceFirstAsText("brand")).setModel(arg.forceFirstAsText("model")).setConfName(
                                    arg.forceFirstAsText("configurationName")).setDisplacement(
                                    arg.forceFirstAsText("displacement")).setEngineType(
                                    arg.forceFirstAsText("engineType")).setFn(arg.forceFirstAsText("name")).setGearType(
                                    arg.forceFirstAsText("gearType")).setProdyear(
                                    arg.forceFirstAsText("prodyear")).setSteeringWheel(
                                    arg.forceFirstAsText("steeringWheel")).setTransmission(
                                    arg.forceFirstAsText("transmission")).setUrl(arg.getFirstAsText("url"));
                    return x;
                }
            };

    public static Chords fromSchemaOrgChords(final ComplexMicrodata microdata, final String sourceUrl, final String lang) {
        if (microdata.getType() == null) {
            return null;
        }
        String clearType= microdata.getType();
        for(String prefix : MicrodataUtils.protocolAliases("http://schema.org")) {
            clearType = clearType.replace(prefix, "");
        }
        if (!"MusicRecording".equals(clearType)) {
            return null;
        }
        if (microdata.getFirst("chordsLine") == null && microdata.getFirst("chordsBlock") == null) {
            return null;
        }
        final AccordsBuilder builder =
                new AccordsBuilder().setArtistName(MicrodataUtils.getTextOrName(microdata, "byArtist")).setTitle(
                        microdata.forceFirstAsText("name")).setSourceUrl(sourceUrl).setLang(lang);
        String chordsBlock = microdata.getFirstAsHtml("chordsBlock");
        if (chordsBlock != null) {
            builder.setRaw(chordsBlock.replaceAll("<[^>]*>", ""));
        }
        for (final Microdata m : microdata.getPropAsList("chordsLine")) {
            if (m instanceof ComplexMicrodata) {
                try {
                    final String chords = ((ComplexMicrodata) m).getFirstAsHtml("chords").replaceAll("<[^>]*>", "");
                    final String text = ((ComplexMicrodata) m).getFirstAsHtml("text").replaceAll("<[^>]*>", "");
                    builder.addChordsAndTextLines(pair(chords, text));
                }
                catch (NullPointerException e){
                    // impossible to extract one or both chord and text arguments
                }

            }
        }
        return builder.createAccords();
    }


    public static WebForm fromJSONLDWebForm(final ComplexMicrodata microdata, final String sourceUrl, final String lang) {
        if (microdata.getType() == null) {
            return null;
        }
        String clearType= microdata.getType();
        for(String prefix : MicrodataUtils.protocolAliases("http://schema.org")) {
            clearType = clearType.replace(prefix, "");
        }
        if (!"WebFormHandler".equals(clearType)) {
            return null;
        }
        final String schemeUrl = microdata.getFirstAsText("specificationUrl");
        if (isEmpty(schemeUrl)) {
            return null;
        }
        final Map<String, String> defVals = new HashMap<String, String>();
        for (Microdata defVal : microdata.getPropAsList("defaultProperty")) {
            if (defVal instanceof ComplexMicrodata) {
                ComplexMicrodata compVal = (ComplexMicrodata) defVal;
                final String name = compVal.forceFirstAsText("name");
                final String defaultValue = compVal.forceFirstAsText("defaultValue");
                if (name != null && defaultValue != null) {
                    defVals.put(name, defaultValue);
                }
            }
        }
        return new WebForm(schemeUrl, defVals, sourceUrl);
    }

    public static WebForm fromYaInteractionWebForm(final OGPData data, final String sourceUrl) {
        if (!"XML_FORM".equals(data.getFirstOrNull("ya:interaction"))) {
            return null;
        }
        String url = null;
        Map<String, String> vals = new HashMap<String, String>();
        String propName = null;
        for (final Pair<String, String> pair : data.getAllData()) {
            if ("ya:interaction:url".equals(pair.first)) {
                if (url != null) {
                    return null;
                }
                url = pair.second;
            }
            if ("ya:interaction:property".equals(pair.first)) {
                propName = pair.second;
            }
            if ("ya:interaction:property:default_value".equals(pair.first)) {
                if (propName != null) {
                    vals.put(propName, pair.second);
                    propName = null;
                }
            }
        }
        if (url != null) {
            String absUrl;
            try {
                absUrl = new URI(sourceUrl).resolve(url).toString();
            } catch (URISyntaxException e) {
                throw new RuntimeException(e); // cant be - urls from robot
            }
            return new WebForm(absUrl, vals, sourceUrl);
        } else {
            return null;
        }
    }

    public void setFrontEnd(FrontEnd frontEnd) {
        this.frontEnd = frontEnd;
    }
}
