package ru.yandex.webmaster3.core.sitemap;

import java.util.Set;
import java.util.UUID;

import org.joda.time.DateTime;

import ru.yandex.webmaster3.core.codes.LinkType;
import ru.yandex.webmaster3.core.data.HttpCodeInfo;
import ru.yandex.webmaster3.core.sitemap.raw.HostRelatedSitemaps;

/**
 * @author aherman
 */
public class HostSitemap {
    private final UUID sitemapId;
    private final String sitemapUrl;
    private final Info info;
    private boolean deletable;
    private boolean deleted;

    public HostSitemap(UUID sitemapId, String sitemapUrl, Info info, boolean deletable, boolean deleted) {
        this.sitemapId = sitemapId;
        this.sitemapUrl = sitemapUrl;
        this.info = info;
        this.deletable = deletable;
        this.deleted = deleted;
    }

    public HostSitemap(UUID sitemapId, String sitemapUrl, Set<HostRelatedSitemaps.SitemapSource> sources,
                       UUID parentSitemapId, DateTime lastAccessDate, HttpCodeInfo httpCodeInfo, String redirectTarget,
                       boolean index, long urlsCount, int errorsCount, int childrenCount) {
        this(sitemapId, sitemapUrl, new Info(sources, parentSitemapId, lastAccessDate, httpCodeInfo, redirectTarget, index,
                urlsCount, errorsCount, childrenCount), false, false);
    }

    public HostSitemap(UserSitemap userSitemap) {
        this(userSitemap.getSitemapId(), userSitemap.getSitemapUrl(), null, true, false);
    }

    public UUID getSitemapId() {
        return sitemapId;
    }

    public String getSitemapUrl() {
        return sitemapUrl;
    }

    public boolean isDeletable() {
        return deletable;
    }

    public void setDeletable(boolean deletable) {
        this.deletable = deletable;
    }

    public void setDeleted(boolean deleted) {
        this.deleted = deleted;
    }

    public boolean isDeleted() {
        return deleted;
    }

    public Info getInfo() {
        return info;
    }

    public HostSitemap copy() {
        return new HostSitemap(sitemapId, sitemapUrl, info, deletable, deleted);
    }

    public static boolean isRedirect(int code) {
        return LinkType.REDIRECT.contains(code) || code == 2004;
    }

    public static boolean isRedirect(HttpCodeInfo code) {
        return (code != null) && (LinkType.REDIRECT.contains(code.getCode()) || code.getCode() == 2004);
    }

    public static String getRedirTargetSafe(String redirTarget, int code) {
        return isRedirect(code) ? redirTarget : null;
    }

    public static String getRedirTargetSafe(String redirTarget, HttpCodeInfo code) {
        return isRedirect(code) ? redirTarget : null;
    }

    public static class Info {
        private final UUID parentSitemapId;
        private final DateTime lastAccessDate;
        private final HttpCodeInfo httpCodeInfo;
        private final String redirectTarget;
        private final boolean index;
        private final long urlsCount;
        private final int errorsCount;
        private final int childrenCount;
        private final Set<HostRelatedSitemaps.SitemapSource> sources;

        public Info(Set<HostRelatedSitemaps.SitemapSource> sources, UUID parentSitemapId, DateTime lastAccessDate,
                    HttpCodeInfo httpCodeInfo, String redirectTarget, boolean index, long urlsCount, int errorsCount, int childrenCount) {
            this.sources = sources;
            this.parentSitemapId = parentSitemapId;
            this.lastAccessDate = lastAccessDate;
            this.httpCodeInfo = httpCodeInfo;
            this.redirectTarget = getRedirTargetSafe(redirectTarget, httpCodeInfo);
            this.index = index;
            this.urlsCount = urlsCount;
            this.errorsCount = errorsCount;
            this.childrenCount = (index || isRedirect(httpCodeInfo)) ? childrenCount : 0;
        }

        public Set<HostRelatedSitemaps.SitemapSource> getSources() {
            return sources;
        }

        public UUID getParentSitemapId() {
            return parentSitemapId;
        }

        public DateTime getLastAccessDate() {
            return lastAccessDate;
        }

        public HttpCodeInfo getHttpCodeInfo() {
            return httpCodeInfo;
        }

        public String getRedirectTarget() {
            return redirectTarget;
        }

        public boolean isIndex() {
            return index;
        }

        public long getUrlsCount() {
            return urlsCount;
        }

        public int getErrorsCount() {
            if (getRedirectTarget() == null && httpCodeInfo != null && httpCodeInfo.getCode() >= 400) {
                // плохой код, не редирект. лемур стал отгружать ошибки парсинга 404 html
                // - а на фронте ошибки парсинга имеют приоритет над ошибочным статусом
                return 0;
            } else {
                return errorsCount;
            }
        }

        public int getChildrenCount() {
            return childrenCount;
        }
    }
}
