package ru.yandex.webmaster3.core.solomon.metric;

import com.fasterxml.jackson.annotation.JsonCreator;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * @author avhaliullin
 */
public final class SolomonKey {

    public static final String LABEL_INDICATOR = SolomonSensor.LABEL_INDICATOR;
    public static final String LABEL_TIME_BUCKET = SolomonSensor.LABEL_TIME_BUCKET;

    private final Map<String, String> labels;

    private SolomonKey(Map<String, String> labels) {
        this.labels = Collections.unmodifiableMap(labels);
    }

    public Map<String, String> getLabels() {
        return labels;
    }

    public SolomonKey withLabel(String name, String value) {
        if (value == null) {
            return withoutLabel(name);
        }
        HashMap<String, String> result = new HashMap<>(labels);
        result.put(
                toSolomonName(name),
                toSolomonValue(name, value)
        );
        return new SolomonKey(result);
    }

    public SolomonKey withoutLabel(String name) {
        HashMap<String, String> result = new HashMap<>(labels);
        result.remove(toSolomonName(name));
        return new SolomonKey(result);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        SolomonKey that = (SolomonKey) o;

        return labels.equals(that.labels);

    }

    @Override
    public int hashCode() {
        return labels.hashCode();
    }

    @Override
    public String toString() {
        return labels.toString();
    }

    private static String toSolomonName(String name) {
        return name.replace('.', '_').replace('-', '_');
    }

    private static String toSolomonValue(String name, String value) {
        return name.equals("host") ? value : toSolomonName(value);
    }

    public static SolomonKey create() {
        return new SolomonKey(Collections.emptyMap());
    }

    public static SolomonKey create(String labelName, String labelValue) {
        HashMap<String, String> labels = new HashMap<>();
        labels.put(
                toSolomonName(labelName),
                toSolomonValue(labelName, labelValue)
        );
        return new SolomonKey(labels);
    }

    @JsonCreator
    public static SolomonKey create(Map<String, String> labels) {
        HashMap<String, String> preparedLabels = new HashMap<>();
        for (Map.Entry<String, String> entry : labels.entrySet()) {
            preparedLabels.put(toSolomonName(entry.getKey()), toSolomonValue(entry.getKey(), entry.getValue()) );
        }
        return new SolomonKey(preparedLabels);
    }
}
