package ru.yandex.webmaster3.core.sup;


import java.io.IOException;
import java.nio.charset.Charset;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpHeaders;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.sup.model.BellQuery;
import ru.yandex.webmaster3.core.sup.model.Query;
import ru.yandex.webmaster3.core.util.json.JsonMapping;

/**
 * ishalaru
 * 17.03.2020
 **/
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class SupIntegrationService {
    private static final String WEBMASTER_ICON_LINK = "https://avatars.mds.yandex.net/get-turbo/3117766/2a00000172db23ba47455f8ad019381e6657/orig";
    private static final String AUTHORIZATION = "Authorization";
    private static final int SOCKET_TIMEOUT = 10_000;
    private static final String POLICY_USER = "webmaster_user_limit_policy";
    private static final String POLICY_CONTENT = "webmaster_content_limit_policy";
    private static final int RETRIES = 5;
    @Value("${webmaster3.core.sup.oauth.secret}")
    private final String oauth;
    @Value("${webmaster3.core.sup.url}")
    private final String supUrl;
    @Value("${webmaster3.core.sup.projectId}")
    private final String projectId;
    @Value("${webmaster3.core.sup.ttl}")
    private Long ttl;

    private CloseableHttpClient httpClient;

    public void init() {

        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(HttpConstants.DEFAULT_CONNECTION_REQUEST_TIMEOUT)
                .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT)
                .build();

        httpClient = HttpClientBuilder.create()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionTimeToLive(10, TimeUnit.MINUTES)
                .build();
    }

    public boolean sendBell(List<Long> uids, BellQuery.BellNotification bellMeta) {
        HttpPost httpPost = new HttpPost(supUrl);
        httpPost.addHeader(AUTHORIZATION, oauth);
        httpPost.addHeader(HttpHeaders.CONTENT_TYPE, ContentType.APPLICATION_JSON.getMimeType());
        List<String> workUids = uids.stream().map(BellQuery.UID::fromUID).collect(Collectors.toList());
        Query.ThrottlePolicies throttlePolicies = new Query.ThrottlePolicies(POLICY_USER, POLICY_CONTENT);
        BellQuery query = new BellQuery(workUids, ttl, projectId, bellMeta, throttlePolicies, "now", false);
        final String s = JsonMapping.writeValueAsString(query);
        StringEntity entity = new StringEntity(s, Charset.defaultCharset());
        httpPost.setEntity(entity);
        try (final CloseableHttpResponse execute = httpClient.execute(httpPost)) {
            log.info("Send bell notification {}, to {}", bellMeta, uids);
            if (log.isTraceEnabled()) {
                log.trace("Result: {}", execute.getStatusLine());
                log.trace("Entity: {}", execute.getEntity());
                log.trace("Response body:{}", new String(execute.getEntity().getContent().readAllBytes(), Charset.defaultCharset()));
            }
            if (execute.getStatusLine().getStatusCode() / 100 == 2) {
                return true;
            } else {
                //В ситуации когда мы посылаем слишком много уведомлений, sup может перестать их обрабатывать и нужно немного подождать
                Thread.sleep(100);
            }

        } catch (ClientProtocolException e) {
            log.error(e.getMessage(), e);
        } catch (IOException e) {
            log.error(e.getMessage(), e);
        } catch (InterruptedException e) {
            log.error(e.getMessage(), e);
        }
        return false;
    }

    public boolean send(PushData pushData) {
        HttpPost httpPost = new HttpPost(supUrl);
        httpPost.addHeader(AUTHORIZATION, oauth);
        httpPost.addHeader(HttpHeaders.CONTENT_TYPE, ContentType.APPLICATION_JSON.getMimeType());
        Query query = getQuery(pushData);
        final String s = JsonMapping.writeValueAsString(query);
        StringEntity entity = new StringEntity(s, Charset.defaultCharset());
        httpPost.setEntity(entity);
        for (int i = 0; i < RETRIES; i++) {
            try (final CloseableHttpResponse execute = httpClient.execute(httpPost)) {
                log.info("Send sup notification {}, to {}", pushData.body, pushData.uids);
                if (log.isTraceEnabled()) {
                    log.trace("Result: {}", execute.getStatusLine());
                    log.trace("Entity: {}", execute.getEntity());
                    log.trace("Response body:{}", new String(execute.getEntity().getContent().readAllBytes(), Charset.defaultCharset()));
                }
                if (execute.getStatusLine().getStatusCode() / 100 == 2) {
                    return true;
                }

            } catch (ClientProtocolException e) {
                log.error(e.getMessage(), e);
            } catch (IOException e) {
                log.error(e.getMessage(), e);
            }
        }
        return false;
    }

    @NotNull
    public static Query getQuery(PushData pushData, Long ttl, String projectId) {
        final List<String> workUids = pushData.uids.stream().map(Query.UID::fromUID).collect(Collectors.toList());
        Query.Notification notification = new Query.Notification(pushData.title, pushData.body, pushData.link, WEBMASTER_ICON_LINK);
        Query.ThrottlePolicies throttlePolicies = new Query.ThrottlePolicies(POLICY_USER, POLICY_CONTENT);
        //Query query = new Query(workUids, ttl, projectId, notification, bellNotification, throttlePolicies, "now", false);
        return new Query(workUids, ttl, projectId,
                Query.PushData.create(pushData.pushId, pushData.pushId),
                notification,
                Query.AndroidFeatures.create(pushData.projectTitle),
                Query.PushMeta.create(pushData.shortTitle),
                throttlePolicies,
                "now",
                false);
    }

    @NotNull
    private Query getQuery(PushData pushData) {
        return getQuery(pushData, ttl, projectId);
    }

    public boolean sendYtPush(String source) {
        HttpPost httpPost = new HttpPost(supUrl + "/batch");
        httpPost.addHeader(AUTHORIZATION, oauth);
        httpPost.addHeader(HttpHeaders.CONTENT_TYPE, ContentType.APPLICATION_JSON.getMimeType());
        final String s = JsonMapping.writeValueAsString(new YtPushData(source));
        StringEntity entity = new StringEntity(s, Charset.defaultCharset());
        httpPost.setEntity(entity);
        try (final CloseableHttpResponse execute = httpClient.execute(httpPost)) {
            log.info("Send sup YT notification {}", s);
            if (log.isTraceEnabled()) {
                log.trace("Result: {}", execute.getStatusLine());
                log.trace("Entity: {}", execute.getEntity());
                log.trace("Response body:{}", new String(execute.getEntity().getContent().readAllBytes(), Charset.defaultCharset()));
            }
            if (execute.getStatusLine().getStatusCode() / 100 == 2) {
                return true;
            } else {
                log.info("Result: {}", execute.getStatusLine());
                log.info("Entity: {}", execute.getEntity());
                log.info("Response body:{}", new String(execute.getEntity().getContent().readAllBytes(), Charset.defaultCharset()));
            }

        } catch (ClientProtocolException e) {
            log.error(e.getMessage(), e);
        } catch (IOException e) {
            log.error(e.getMessage(), e);
        }
        return false;
    }

    @lombok.Value
    public static class YtPushData {
        String path;
    }

    @lombok.Value
    public static class PushData {
        String pushId;
        String title;
        String projectTitle;
        String shortTitle;
        String body;
        String link;
        List<Long> uids;
    }

}
