package ru.yandex.webmaster3.core.tracer;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;

import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.core.solomon.metric.SolomonKey;
import ru.yandex.webmaster3.core.solomon.metric.SolomonMetricRegistry;
import ru.yandex.webmaster3.core.solomon.metric.SolomonTimer;
import ru.yandex.webmaster3.core.solomon.metric.SolomonTimerConfiguration;

@Service
@Getter
@Slf4j
public class YdbMetricsService {
    private static final List<String> OPERATIONS = List.of(
            "SELECT",
            "INSERT",
            "UPSERT",
            "BATCH_INSERT",
            "BATCH_UPDATE",
            "UPDATE",
            "DELETE"
    );

    private final ApplicationContext applicationContext;
    private final SolomonMetricRegistry solomonMetricRegistry;
    private final SolomonTimerConfiguration solomonTimerConfiguration;
    //tableName, operationType, metric
    private static Map<Pair<String, String>, YdbMetrics> metricsMap;

    public YdbMetricsService(ApplicationContext applicationContext,
                             SolomonMetricRegistry solomonMetricRegistry,
                             @Qualifier("ydbMetricsConfiguration") SolomonTimerConfiguration solomonTimerConfiguration) {
        this.applicationContext = applicationContext;
        this.solomonMetricRegistry = solomonMetricRegistry;
        this.solomonTimerConfiguration = solomonTimerConfiguration;
        setMetricsMap();
    }

    public synchronized void setMetricsMap() {
        if (metricsMap == null) {
            var ydbTables = applicationContext.getBeansOfType(TableNameProvider.class).values();
            log.info("info");
            metricsMap = new HashMap<>(OPERATIONS.size() * ydbTables.size());
            for (TableNameProvider table : ydbTables) {
                for (String operation : OPERATIONS) {
                        metricsMap.put(Pair.of(table.getTableName(), operation),
                                new YdbMetrics(table.getTableName(), operation)
                        );
                }
            }
        }
    }

    public static void saveSolomon(YdbTrace trace) {
        if (trace == null) {
            return;
        }
        try {
            if (metricsMap == null) {
                return;
            }
            updateSolomonSensor(trace.getPrepareTimeNanoByTable(), YdbMetrics::getPrepareTimer);
            updateSolomonSensor(trace.getTimeNanoByTable(), YdbMetrics::getQueryTimer);
        } catch (Exception e) {
            log.error("error while save to solomon ydb stats ", e);
        }
    }

    private static void updateSolomonSensor(Map<String, Map<String, AtomicLong>> statsTable,
                                            Function<YdbMetrics, SolomonTimer> sensor) {
        for (var entry : statsTable.entrySet()) {
            String tableName = entry.getKey();
            for (var iEntry : entry.getValue().entrySet()) {
                String operation = iEntry.getKey();
                Pair<String, String> key = Pair.of(tableName, operation);
                YdbMetrics ydbMetrics = metricsMap.get(key);
                if (ydbMetrics != null) {
                    long millis = iEntry.getValue().get() / 1000000;
                    sensor.apply(ydbMetrics).update(Duration.millis(millis));
                }
            }
        }
    }


    @Getter
    public class YdbMetrics {
        private static final String QUERY = "query";
        private static final String SOLOMON_LABEL_TABLE = "table";
        private static final String PREPARE = "prepare";
        private static final String SOLOMON_LABEL_TARGET = "target";
        private static final String SOLOMON_LABEL_OPERATION_TYPE = "operation";
        private static final String MAIN_LABEL = "ydb";


        private final SolomonTimer queryTimer;
        private final SolomonTimer prepareTimer;

        YdbMetrics(String tableName, String op) {
            SolomonKey baseKey = createBaseKey(tableName, op);
            queryTimer = solomonMetricRegistry.createTimer(
                    solomonTimerConfiguration,
                    baseKey.withLabel(SolomonSensor.LABEL_DATA_TYPE, QUERY)
            );
            prepareTimer = solomonMetricRegistry.createTimer(
                    solomonTimerConfiguration,
                    baseKey.withLabel(SolomonSensor.LABEL_DATA_TYPE, PREPARE)
            );
        }

        @NotNull
        private SolomonKey createBaseKey(String tableName, String op) {
            return SolomonKey.create(SolomonSensor.LABEL_CATEGORY, MAIN_LABEL)
                    .withLabel(SOLOMON_LABEL_TABLE, tableName)
                    .withLabel(SOLOMON_LABEL_OPERATION_TYPE, op);

        }
    }

}
