package ru.yandex.webmaster3.core.tracer;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

import lombok.AccessLevel;
import lombok.Getter;
import lombok.ToString;
import lombok.experimental.FieldDefaults;
import lombok.experimental.NonFinal;

/**
 * @author aherman
 */
@ToString
@Getter
@FieldDefaults(makeFinal = true, level = AccessLevel.PRIVATE)
public class YdbTrace {
    private static final Set<String> READ_TYPES = Set.of("SELECT");

    private AtomicLong readCount = new AtomicLong();
    private AtomicLong totalReadTimeNano = new AtomicLong();
    private AtomicLong totalReadPrepareTimeNano = new AtomicLong();
    private AtomicLong totalReadBytes = new AtomicLong();

    private AtomicLong writeCount = new AtomicLong();
    private AtomicLong totalWriteTimeNano = new AtomicLong();
    private AtomicLong totalWritePrepareTimeNano = new AtomicLong();

    private Map<String, AtomicLong> streamTimeByTable = new ConcurrentHashMap<>();
    //tableName, operationType, <value>
    private Map<String, Map<String, AtomicLong>> bytesByTable = new ConcurrentHashMap<>();
    private Map<String, Map<String, AtomicLong>> timeNanoByTable = new ConcurrentHashMap<>();
    private Map<String, Map<String, AtomicLong>> prepareTimeNanoByTable = new ConcurrentHashMap<>();



    private final long requestStartTimeNano;
    @NonFinal
    private long requestEndTimeNano;

    public YdbTrace(long wallTimeNano) {
        this.requestStartTimeNano = wallTimeNano;
    }

    public synchronized void endRequest(long wallTimeNano) {
        this.requestEndTimeNano = wallTimeNano;
    }


    public void addQueryStats(String table, String operationType, long queryTimeMillis) {
        if (READ_TYPES.contains(operationType)) {
            readCount.incrementAndGet();
            totalReadTimeNano.addAndGet(queryTimeMillis);
        } else {
            writeCount.incrementAndGet();
            totalWriteTimeNano.addAndGet(queryTimeMillis);
        }
        var value = timeNanoByTable.computeIfAbsent(table, x -> new ConcurrentHashMap<>())
                .computeIfAbsent(operationType, y -> new AtomicLong());
        value.addAndGet(queryTimeMillis);
    }

    public void addBytesStats(String table, String operationType, long bytesLength) {
        //todo убедиться что этот метод не жрет много ресурсов
        if (READ_TYPES.contains(operationType)) {
            totalReadBytes.addAndGet(bytesLength);
        } else {
            // todo not implemented
        }
        var value = bytesByTable.computeIfAbsent(table, x -> new ConcurrentHashMap<>())
                .computeIfAbsent(operationType, y -> new AtomicLong());
        value.addAndGet(bytesLength);
    }

    public void addPrepareStats(String table, String operationType, long queryTimeMillis) {
        if (READ_TYPES.contains(operationType)) {
            totalReadPrepareTimeNano.addAndGet(queryTimeMillis);
        } else {
            totalWriteTimeNano.addAndGet(queryTimeMillis);
        }

        var value = prepareTimeNanoByTable.computeIfAbsent(table, x -> new ConcurrentHashMap<>())
                .computeIfAbsent(operationType, y -> new AtomicLong());
        value.addAndGet(queryTimeMillis);
    }

    public void addForEachStats(String table, long queryTimeMillis) {
        var timer = streamTimeByTable.computeIfAbsent(table, y -> new AtomicLong());

        timer.addAndGet(queryTimeMillis);
    }
}
