package ru.yandex.webmaster3.core.turbo.model;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Getter;
import lombok.Value;

import ru.yandex.autodoc.common.doc.annotation.Description;

/**
 * Created by Oleg Bazdyrev on 02/12/2020.
 */
@Value
@Builder(toBuilder = true)
@AllArgsConstructor(onConstructor_ = @JsonCreator)
public class TurboDisplaySettings {

    @JsonProperty("enabled")
    @Description("Включена ли кастомизация")
    boolean enabled;
    @JsonProperty("header")
    @Description("Настройки заголовка")
    TurboHostHeader header;
    @JsonProperty("lightMode")
    @Description("Настройки светлой темы (дефолтной)")
    ColorScheme lightMode;
    @JsonProperty("darkMode")
    @Description("Настройки темной темы")
    ColorScheme darkMode;
    @JsonProperty("headerFontFamily")
    @Description("Шрифт заголовка - с засечками, или без")
    @Builder.Default
    FontFamily headerFontFamily = FontFamily.SERIF;
    @JsonProperty("contentFontFamily")
    @Description("Шрифт осноного текста - с засечками, или без")
    @Builder.Default
    FontFamily contentFontFamily = FontFamily.SERIF;
    @JsonProperty("headerFontSize")
    @Description("Размер шрифта в заголовках")
    @Builder.Default
    FontSize headerFontSize = FontSize.SMALL;
    @JsonProperty("contentFontSize")
    @Description("Размер шрифта для основного текста")
    @Builder.Default
    FontSize contentFontSize = FontSize.SMALL;

    public static TurboDisplaySettingsBuilder builder(TurboDisplaySettings s) {
        return s == null ? new TurboDisplaySettingsBuilder() : s.toBuilder();
    }

    /**
     * Отключение кастомизации (при настройках css)
     */
    public static TurboDisplaySettings disable(TurboDisplaySettings s) {
        if (s == null || s.getLightMode() == null) {
            return s;
        }
        return s.toBuilder().lightMode(s.getLightMode().toBuilder().enabled(false).build()).build();
    }

    @Getter
    @AllArgsConstructor
    public enum FontSize {
        SMALL("small"),
        MEDIUM("medium"),
        LARGE("large"),
        ;

        String value;
    }

    @Getter
    @AllArgsConstructor
    public enum FontFamily {
        SERIF("serif"),
        SANS_SERIF("sans-serif"),
        ;
        String value;

        @Override
        public String toString() {
            return name().toLowerCase();
        }
    }

    @Value
    @Builder(toBuilder = true)
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    public static final class ColorScheme {
        @JsonProperty("enabled")
        @Description("Включена ли цветовая тема (только для темной - светлая всегда включена)")
        boolean enabled;
        @JsonProperty("headerBackgroundColor")
        @Description("Цвет шапки")
        ColorSettings headerBackgroundColor;
        @JsonProperty("headerForegroundColor")
        @Description("Цвет текста и иконок в шапке")
        ColorSettings headerForegroundColor;
        @JsonProperty("contentBackgroundColor")
        @Description("Цвет фона в контенте")
        ColorSettings contentBackgroundColor;
        @JsonProperty("contentTextColor")
        @Description("Цвет заголовков и текста")
        ColorSettings contentTextColor;
        @JsonProperty("contentDateColor")
        @Description("Цвет подписи и даты")
        ColorSettings contentDateColor;
        @JsonProperty("contentLinkColor")
        @Description("Цвет ссылок")
        ColorSettings contentLinkColor;
        @JsonProperty("contentButtonColor")
        @Description("Цвет кнопок в контенте")
        ColorSettings contentButtonColor;
        @JsonProperty("contentTextButtonColor")
        @Description("Цвет текста кнопок в контенте")
        ColorSettings contentTextButtonColor;
        @JsonProperty("logo")
        @Description("Логотип для темы (только для темной - у светлой общий логотип)")
        TurboLogo logo;
    }

    @Value
    @Builder(toBuilder = true)
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    public static final class ColorSettings {
        @JsonProperty("color")
        @Description("Выбранный цвет")
        String color;
        @JsonProperty("customColor")
        @Description("Текущее значение селектора кастомного цвета")
        String customColor;
        @JsonProperty("customSelected")
        @Description("Выбран ли кастомный цвет")
        boolean customSelected;
    }

}
