package ru.yandex.webmaster3.core.turbo.model.advertising;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import com.google.common.base.Strings;
import lombok.EqualsAndHashCode;

import ru.yandex.autodoc.common.doc.annotation.Description;

import static ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingPlacement.CONTENT;
import static ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingPlacement.MANUAL;
import static ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingPlacement.MANUAL_STICKY;
import static ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingPlacement.STICKY;

/**
 * Настройки рекламы для Турбо
 * Created by Oleg Bazdyrev on 05/07/2017.
 */
@EqualsAndHashCode
@JsonIgnoreProperties(ignoreUnknown = true)
public class AdvertisingSettings {

    public static final TypeReference<List<AdvertisingSettings>> LIST_REFERENCE = new TypeReference<>() {
    };
    public static final TypeReference<Map<AdvertisingPlacement, List<AdvertisingSettings>>> MAP_REFERENCE = new TypeReference<>() {
    };

    private final AdvertisingNetworkType type;
    private final AdvertisingPlacement placement;
    private final String alias;
    private final String value;
    private final ObjectNode data;

    //@JsonCreator
    public AdvertisingSettings(@JsonProperty("type") AdvertisingNetworkType type,
                               @JsonProperty("placement") AdvertisingPlacement placement,
                               @JsonProperty("alias") String alias,
                               @JsonProperty("value") String value,
                               @JsonProperty("data") ObjectNode data) {
        this.type = type;
        this.placement = placement;
        this.alias = alias;
        this.value = value;
        this.data = data;
    }

    @JsonCreator
    public AdvertisingSettings(@JsonProperty("type") AdvertisingNetworkType type,
                               @JsonProperty("placement") AdvertisingPlacement placement,
                               @JsonProperty("alias") String alias,
                               @JsonProperty("value") String value,
                               @JsonProperty("data") ObjectNode data,
                               @JsonProperty("displayType") String dsType) {
        this.type = type;
        this.placement = placement;
        this.alias = alias;
        this.value = value;
        this.data = data;

    }

    @Description("Тип рекламной сети")
    public AdvertisingNetworkType getType() {
        return type;
    }

    @Description("Где блок расположен")
    public AdvertisingPlacement getPlacement() {
        return placement;
    }

    @Description("Алиас(псевдоним) рекламного блока")
    public String getAlias() {
        return alias;
    }

    @Description("Человекопонятное описание (id или код блока)")
    public String getValue() {
        return value;
    }

    @Description("Кастомный json с информацией о рекламном блоке")
    public ObjectNode getData() {
        return data;
    }

    @Description("Что представляет из себя value")
    public AdvertisingDisplayType getDisplayType() {
        return type.getDisplayType();
    }

    public AdvertisingSettings withPlacement(AdvertisingPlacement placement) {
        return new AdvertisingSettings(type, placement, placement == MANUAL ? alias : null, value, data);
    }

    /**
     * Преобразование в Json для экспорта
     *
     * @return
     */
    public ObjectNode toJson() {
        if (data == null) {
            return null;
        }
        ObjectNode result = data.deepCopy();

        //костыль!!! турбо принимает формат {"id":"R-A-446132-3","type":"sticky","placed":"manual"}
        //                                                                  ....."placed":"auto"}
        //мы же рассматриваем ее не как другой ти
        if (placement == STICKY || placement == MANUAL_STICKY) {
            result.set("type", new TextNode("sticky"));
        } else {
            result.set("type", new TextNode(type.getDisplayName()));
        }
        if (!Strings.isNullOrEmpty(alias)) {
            result.set("alias", new TextNode(alias));
        }
        // костыль на всякий случай для InStream
        String placed = type == AdvertisingNetworkType.YANDEX_INSTREAM ?
                AdvertisingNetworkType.YANDEX_INSTREAM.getDisplayName() : placement.getDisplayName();
        result.set("placed", new TextNode(placed));
        return result;
    }

    public static Map<AdvertisingPlacement, List<AdvertisingSettings>> adsMapFromList(
            List<AdvertisingSettings> advertisingSettings) {
        if (advertisingSettings == null) {
            return null;
        }

        Map<AdvertisingPlacement, List<AdvertisingSettings>> result = new EnumMap<>(AdvertisingPlacement.class);
        for (AdvertisingSettings as : advertisingSettings) {
            // костыль для старой рекламы
            if (as.getPlacement() == null) {
                result.computeIfAbsent(CONTENT, (k) -> new ArrayList<>()).add(as.withPlacement(CONTENT));
                if (!Strings.isNullOrEmpty(as.getAlias())) {
                    result.computeIfAbsent(MANUAL, (k) -> new ArrayList<>()).add(as.withPlacement(MANUAL));
                }
            } else {
                result.computeIfAbsent(as.getPlacement(), (k) -> new ArrayList<>()).add(as);
            }
        }
        return result;
    }

    public static List<AdvertisingSettings> adsListFromMap(
            Map<AdvertisingPlacement, List<AdvertisingSettings>> advertisingSettings) {
        if (advertisingSettings == null) {
            return Collections.emptyList();
        }
        return advertisingSettings.entrySet().stream()
                .flatMap(entry -> entry.getValue().stream().map(as -> as.withPlacement(entry.getKey())))
                .collect(Collectors.toList());
    }


    public static Set<AdvertisingPlacement> getChangedPlacements(
            Map<AdvertisingPlacement, List<AdvertisingSettings>> newSettings,
            Map<AdvertisingPlacement, List<AdvertisingSettings>> oldSettings) {
        if (newSettings == null) {
            return Collections.emptySet();
        }
        if (oldSettings == null) {
            return newSettings.keySet();
        }
        EnumSet<AdvertisingPlacement> result = EnumSet.noneOf(AdvertisingPlacement.class);
        for (AdvertisingPlacement placement : AdvertisingPlacement.values()) {
            if (!new HashSet<>(newSettings.getOrDefault(placement, Collections.emptyList())).equals(
                    new HashSet<>(oldSettings.getOrDefault(placement, Collections.emptyList())))) {
                result.add(placement);
            }
        }
        return result;
    }

}
