package ru.yandex.webmaster3.core.turbo.model.commerce;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Strings;
import lombok.Getter;
import lombok.With;
import org.apache.commons.lang3.ObjectUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.turbo.model.TurboHostSettingsBlock;
import ru.yandex.webmaster3.core.turbo.model.commerce.delivery.DeliverySection;

/**
 * Created by Oleg Bazdyrev on 03/09/2018.
 * Настройки хоста для e-commerce
 */
@Getter
@With
public class TurboCommerceSettings {

    @Description("url корзины")
    private final String cartUrl;
    @Description("Тип урла корзины (исключительно для фронта)")
    private final CartUrlType cartUrlType;
    @Description("Включены ли покупки при помощи кнопки В корзину")
    private final boolean cartUrlEnabled;
    @Description("Емейл для покупок в один клик (и для турбо-корзины)")
    private final String checkoutEmail;
    @Description("Включены ли покупки в один клик")
    private final boolean checkoutEmailEnabled;
    @Description("список информационных секций")
    private final List<TurboCommerceInfoSection> infoSections;
    @Description("Распаршенный аккордеон, нужен только внутреннего потребления, на фронте никак отображать не надо")
    private final JsonNode parsedAccordion;
    @Description("Включена ли турбо-корзина")
    private final boolean turboCartEnabled;
    @Description("Включены ли турбо-листинги")
    private final boolean turboListingsEnabled;
    @Description("Настройки Яндекс-оплат")
    private final TurboPaymentsSettings paymentsSettings;
    @Description("Список возможны доставок")
    private final DeliverySection deliverySection;
    @Description("Включена ли автоглавная для екоммерс")
    private final Boolean autoMorda;
    @Description("Настройки интеграции с Битриксом")
    private final TurboBitrixSettings bitrixSettings;
    @Description("Минимальная стоимость заказа (WMC-8869)")
    private final BigDecimal minOrderValue;
    @Description("Отображение рейтинга из маркета")
    private final Boolean marketRatingShow;

    @JsonCreator
    public TurboCommerceSettings(@JsonProperty("cartUrl") String cartUrl,
                                 @JsonProperty("cartUrlType") CartUrlType cartUrlType,
                                 @JsonProperty("cartUrlEnabled") Boolean cartUrlEnabled,
                                 @JsonProperty("checkoutEmail") String checkoutEmail,
                                 @JsonProperty("checkoutEmailEnabled") Boolean checkoutEmailEnabled,
                                 @JsonProperty("infoSections") List<TurboCommerceInfoSection> infoSections,
                                 @JsonProperty("parsedAccordion") JsonNode parsedAccordion,
                                 @JsonProperty("turboCartEnabled") Boolean turboCartEnabled,
                                 @JsonProperty("turboListingsEnabled") Boolean turboListingsEnabled,
                                 @JsonProperty("paymentsSettings") TurboPaymentsSettings paymentsSettings,
                                 @JsonProperty("deliverySection") DeliverySection deliverySection,
                                 @JsonProperty("autoMorda") Boolean autoMorda,
                                 @JsonProperty("bitrixSettings") TurboBitrixSettings bitrixSettings,
                                 @JsonProperty("minOrderValue") BigDecimal minOrderValue,
                                 @JsonProperty("marketRatingShow") Boolean marketRatingShow) {
        this.cartUrl = cartUrl;
        this.cartUrlType = cartUrlType;
        this.cartUrlEnabled = ObjectUtils.firstNonNull(cartUrlEnabled, !Strings.isNullOrEmpty(cartUrl));
        this.checkoutEmail = checkoutEmail;
        this.checkoutEmailEnabled = ObjectUtils.firstNonNull(checkoutEmailEnabled, false);
        this.infoSections = infoSections;
        this.parsedAccordion = parsedAccordion;
        this.turboCartEnabled = ObjectUtils.firstNonNull(turboCartEnabled, false);
        this.turboListingsEnabled = ObjectUtils.firstNonNull(turboListingsEnabled, turboCartEnabled, false);
        this.paymentsSettings = paymentsSettings;
        this.deliverySection = deliverySection;
        this.autoMorda = autoMorda;
        this.bitrixSettings = bitrixSettings;
        this.minOrderValue = ObjectUtils.firstNonNull(minOrderValue, BigDecimal.ZERO);
        this.marketRatingShow = marketRatingShow;
    }

    @JsonIgnore
    public boolean isOrderMethodSelected() {
        return turboCartEnabled || cartUrlEnabled || checkoutEmailEnabled;
    }

    @Getter
    public static final class TurboCommerceSettingsBuilder {
        private String cartUrl;
        private CartUrlType cartUrlType;
        private boolean cartUrlEnabled;
        private String checkoutEmail;
        private boolean checkoutEmailEnabled;
        private List<TurboCommerceInfoSection> infoSections;
        private JsonNode parsedAccordion;
        private boolean turboCartEnabled;
        private boolean turboListingsEnabled;
        private TurboPaymentsSettings paymentsSettings;
        private DeliverySection deliverySection;
        private Boolean autoMorda;
        private TurboBitrixSettings bitrixSettings;
        private BigDecimal minOrderValue = BigDecimal.ZERO;
        private Boolean marketRatingShow;

        public TurboCommerceSettingsBuilder() {
        }

        public TurboCommerceSettingsBuilder(TurboCommerceSettings tcs) {
            if (tcs != null) {
                this.cartUrl = tcs.cartUrl;
                this.cartUrlType = tcs.cartUrlType;
                this.cartUrlEnabled = tcs.cartUrlEnabled;
                this.checkoutEmail = tcs.checkoutEmail;
                this.checkoutEmailEnabled = tcs.checkoutEmailEnabled;
                this.infoSections = tcs.infoSections;
                this.parsedAccordion = tcs.parsedAccordion;
                this.turboCartEnabled = tcs.turboCartEnabled;
                this.turboListingsEnabled = tcs.turboListingsEnabled;
                this.paymentsSettings = tcs.paymentsSettings;
                this.deliverySection = tcs.deliverySection;
                this.autoMorda = tcs.autoMorda;
                this.bitrixSettings = tcs.bitrixSettings;
                this.minOrderValue = tcs.minOrderValue;
                this.marketRatingShow = tcs.marketRatingShow;
            }
        }

        public TurboCommerceSettingsBuilder withCheckout(TurboCommerceSettings tcs) {
            if (tcs != null) {
                this.cartUrl = tcs.cartUrl;
                this.cartUrlType = tcs.cartUrlType;
                this.cartUrlEnabled = tcs.cartUrlEnabled;
                this.checkoutEmail = tcs.checkoutEmail;
                this.checkoutEmailEnabled = tcs.checkoutEmailEnabled;
                this.turboCartEnabled = tcs.turboCartEnabled;
                this.minOrderValue = tcs.minOrderValue;
            } else {
                this.cartUrl = null;
                this.cartUrlType = null;
                this.cartUrlEnabled = false;
                this.checkoutEmail = null;
                this.checkoutEmailEnabled = false;
                this.turboCartEnabled = false;
                this.minOrderValue = BigDecimal.ZERO;
            }
            return this;
        }

        public TurboCommerceSettingsBuilder withInfoSections(TurboCommerceSettings tcs) {
            if (tcs != null) {
                this.infoSections = tcs.infoSections;
                this.parsedAccordion = tcs.parsedAccordion;
            } else {
                this.infoSections = Collections.emptyList();
                this.parsedAccordion = null;
            }
            return this;
        }

        public TurboCommerceSettingsBuilder withTurboListings(TurboCommerceSettings tcs) {
            if (tcs != null) {
                this.turboListingsEnabled = tcs.turboListingsEnabled;
            } else {
                this.turboListingsEnabled = false;
            }
            return this;
        }

        public void copyCommerceSettings(@Nullable TurboCommerceSettings tcs, @NotNull Set<TurboHostSettingsBlock> blocks) {
            if (blocks.contains(TurboHostSettingsBlock.CHECKOUT)) {
                withCheckout(tcs);
            }
            if (blocks.contains(TurboHostSettingsBlock.INFOSECTIONS)) {
                withInfoSections(tcs);
            }
            if (blocks.contains(TurboHostSettingsBlock.TURBO_LISTINGS)) {
                withTurboListings(tcs);
            }
            if (blocks.contains(TurboHostSettingsBlock.PAYMENTS)) {
                setPaymentsSettings(tcs == null ? null : tcs.paymentsSettings);
            }
            if (blocks.contains(TurboHostSettingsBlock.DELIVERYSECTION)) {
                withDeliverySection(tcs);
            }
            if (blocks.contains(TurboHostSettingsBlock.AUTO_MORDA_ECOMMERCE)) {
                setAutoMorda(tcs == null ? null : tcs.getAutoMorda());
            }
            if (blocks.contains(TurboHostSettingsBlock.BITRIX)) {
                setBitrixSettings(tcs == null ? null : tcs.bitrixSettings);
            }
        }

        public TurboCommerceSettingsBuilder withDeliverySection(TurboCommerceSettings tcs) {
            if (tcs != null) {
                this.deliverySection = tcs.deliverySection;
            } else {
                this.deliverySection = null;
            }
            return this;
        }

        public TurboCommerceSettingsBuilder setCartUrl(String cartUrl) {
            this.cartUrl = cartUrl;
            return this;
        }

        public TurboCommerceSettingsBuilder setCartUrlType(CartUrlType cartUrlType) {
            this.cartUrlType = cartUrlType;
            return this;
        }

        public TurboCommerceSettingsBuilder setCartUrlEnabled(boolean cartUrlEnabled) {
            this.cartUrlEnabled = cartUrlEnabled;
            return this;
        }

        public TurboCommerceSettingsBuilder setCheckoutEmail(String checkoutEmail) {
            this.checkoutEmail = checkoutEmail;
            return this;
        }

        public TurboCommerceSettingsBuilder setCheckoutEmailEnabled(boolean checkoutEmailEnabled) {
            this.checkoutEmailEnabled = checkoutEmailEnabled;
            return this;
        }

        public TurboCommerceSettingsBuilder setInfoSections(List<TurboCommerceInfoSection> infoSections) {
            this.infoSections = infoSections;
            return this;
        }

        public TurboCommerceSettingsBuilder setParsedAccordion(JsonNode parsedAccordion) {
            this.parsedAccordion = parsedAccordion;
            return this;
        }

        public TurboCommerceSettingsBuilder setTurboCartEnabled(boolean turboCartEnabled) {
            this.turboCartEnabled = turboCartEnabled;
            return this;
        }

        public TurboCommerceSettingsBuilder setTurboListingsEnabled(boolean turboListingsEnabled) {
            this.turboListingsEnabled = turboListingsEnabled;
            return this;
        }

        public TurboCommerceSettingsBuilder setPaymentsSettings(TurboPaymentsSettings paymentsSettings) {
            this.paymentsSettings = paymentsSettings;
            return this;
        }

        public TurboCommerceSettingsBuilder setAutoMorda(Boolean autoMorda) {
            this.autoMorda = autoMorda;
            return this;
        }

        public TurboCommerceSettingsBuilder setBitrixSettings(TurboBitrixSettings bitrixSettings) {
            this.bitrixSettings = bitrixSettings;
            return this;
        }

        public TurboCommerceSettingsBuilder setDeliverySection(DeliverySection deliverySection) {
            this.deliverySection = deliverySection;
            return this;
        }

        public TurboCommerceSettingsBuilder setMarketingRatingShow(Boolean marketRatingShow) {
            this.marketRatingShow = marketRatingShow;
            return this;
        }

        public TurboCommerceSettings build() {
            return new TurboCommerceSettings(cartUrl, cartUrlType, cartUrlEnabled, checkoutEmail, checkoutEmailEnabled, infoSections, parsedAccordion,
                    turboCartEnabled, turboListingsEnabled, paymentsSettings, deliverySection, autoMorda, bitrixSettings, minOrderValue, marketRatingShow);
        }
    }
}
