package ru.yandex.webmaster3.core.turbo.model.commerce.delivery;

import java.util.Collections;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.Builder;
import lombok.Getter;
import org.apache.curator.shaded.com.google.common.base.Objects;

import ru.yandex.webmaster3.core.turbo.TurboConstants;
import ru.yandex.webmaster3.core.turbo.model.commerce.CurrencyEnum;
import ru.yandex.webmaster3.core.turbo.model.commerce.VATValueEnum;

/**
 * @author: ishalaru
 * DATE: 10.07.2019
 */
@Getter
@Builder(toBuilder = true)
public class DeliverySection implements DeliverySettings {
    private static final String PREFIX_TEMPLATE = "%s.%d";
    @JsonProperty("mail")
    private final List<MailDeliverySettings> mail;
    @JsonProperty("pickup")
    private final List<PickupDeliverySettings> pickup;
    @JsonProperty("courier")
    private final List<CourierDeliverySettings> courier;
    @JsonProperty("vat")
    private final VATValueEnum vat;
    @JsonProperty("currency")
    private final CurrencyEnum currency;
    @JsonProperty("marketDeliveryEnabled")
    private final Boolean marketDeliveryEnabled;

    @JsonCreator
    public DeliverySection(@JsonProperty("mail") List<MailDeliverySettings> mail,
                           @JsonProperty("pickup") List<PickupDeliverySettings> pickup,
                           @JsonProperty("courier") List<CourierDeliverySettings> courier,
                           @JsonProperty("vat") VATValueEnum vat,
                           @JsonProperty("currency") CurrencyEnum currency,
                           @JsonProperty("marketDeliveryEnabled") Boolean marketDeliveryEnabled) {
        this.mail = mail;
        this.pickup = pickup;
        this.courier = courier;
        this.vat = Objects.firstNonNull(vat, VATValueEnum.NO_VAT);
        this.currency = Objects.firstNonNull(currency, CurrencyEnum.RUB);
        this.marketDeliveryEnabled = marketDeliveryEnabled;
    }

    @JsonCreator
    public DeliverySection() {
        this(Collections.emptyList(), Collections.emptyList(), Collections.emptyList(), null, null, null);
    }

    public void validate(String prefix) {
        int count = mail != null ? mail.size() : 0;
        count += pickup != null ? pickup.size() : 0;
        count += courier != null ? courier.size() : 0;
        if (count > TurboConstants.MAX_COMMERCE_DELIVERY_SETTINGS) {
            throw invalidParamValue("Too many e-commerce delivery sections", "deliverySectionSize", String.valueOf(count));
        }
        if (mail != null) {
            validate(prefix + ".mail", mail);
        }
        if (pickup != null) {
            validate(prefix + ".pickup", pickup);
        }
        if (courier != null) {
            validate(prefix + ".courier", courier);
        }
        if (vat == null) {
            throw invalidParamValue("VAT is required", "vat", null);
        }
        if (currency == null) {
            throw invalidParamValue("Currency is required", "currency", null);
        }
    }

    private void validate(String prefix, List<? extends DeliverySettings> list) {
        int index = 0;
        for (DeliverySettings deliverySettings : list) {
            deliverySettings.validate(String.format(PREFIX_TEMPLATE, prefix, index));
            index++;
        }
    }
}
