package ru.yandex.webmaster3.core.turbo.model.error;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.collect.Sets;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.util.enums.EnumResolver;

/**
 * Created by Oleg Bazdyrev on 02/10/2017.
 */
public enum TurboErrorType {

    BANNED_DOCUMENT(true, false, "description", "Banned.Document"),
    BANNED_FEED(true, false, null, "Banned.Feed"),
    DOCUMENT_QUOTA_EXCEEDED_IMAGES(true, true, "url", "Document.QuotaExceeded.Images"),
    DUPLICATE_DOCUMENT(false, false, "url", "Duplicate.Document"),
    DUPLICATE_FEED(false, false, null, "Duplicate.Feed"),
    EMAIL_BAD_DOMAIN(true, false, "email", "Email.BadDomain"),
    EMAIL_BAD_DOMAIN_INVALID(true, false, "email", "Email.BadDomain.Invalid"),
    EMAIL_BAD_DOMAIN_NOT_ALLOWED(true, false, "email", "Email.BadDomain.NotAllowed"),
    EMAIL_INVALID_FORMAT(false, false, "email", "Email.InvalidFormat"),
    FETCH_QUOTA_EXCEEDED_FEEDS(true, false, null, "Fetch.QuotaExceeded.Feeds"),
    FETCH_QUOTA_EXCEEDED_IMAGES(false, false, null, "Fetch.QuotaExceeded.Images"),
    FETCH_QUOTA_EXCEEDED_ITEMS(false, false, null, "Fetch.QuotaExceeded.Items"),
    @Description("Фид запрещен настройками robots.txt")
    FETCHER_DISALLOWED_IN_ROBOTS(TurboError.IS_DISALLOWED_IN_ROBOTS, true, false, null, "Fetcher.Http"),
    @Description("Сервер ответил ошибкой 4xx-5xx")
    FETCHER_HTTP(true, false, "http_code", "Parser.Http3xx", "Parser.Http4xx", "Parser.Http5xx"),
    @Description("Сервер не ответил вообще (и другие ошибки с внутренними статусами 1000+)")
    FETCHER_NO_RESPONSE(TurboError.IS_DISALLOWED_IN_ROBOTS.negate(), true, false, null, "Fetcher.Http"),
    IMAGES_FETCH_ERROR(true, true, "url", "Images.FetchError", "Images.Unavailable", "Skip.Images"),
    IMAGES_STILL_WAITING(true, true, "url", "Images.StillWaiting"),
    //IMAGES_ITEM_IMAGES_MISSING(true, "url", "Images.ItemImagesMissing"),
    INTERNAL(true, false, null, "Internal", "Storage.Internal", "Fetcher.Internal", "Parser.Internal",
            "Images.FetchInternalError", "Fetch.Internal", "Images.Internal", "Internal.Error"),
    PARSER_AD_BAD_ID(true, false, "type", "Parser.Ad.BadId"),
    PARSER_AD_BAD_TYPE(false, true, null, "Parser.Ad.BadType"),
    PARSER_AD_DUPLICATE(false, false, "type", "Parser.Ad.Duplicate"),
    PARSER_AD_MISSING_TURBO_ID(false, false, null, "Parser.Ad.MissingTurboId"),
    PARSER_AD_UNKNOWN_REF(true, false, "url", "Parser.Ad.UnknownRef"),
    PARSER_AD_UNKNOWN_TYPE(true, false, "type", "Parser.Ad.UnknownType"),
    PARSER_BAD_FORMAT(true, true, null, "Parser.BadFormat"),
    PARSER_BAD_FORMAT_BROKEN_UTF(false, true, null, "Parser.BadFormat.BrokenUtf"),
    PARSER_BAD_FORMAT_DUPLICATE_TAG(false, true, null, "Parser.BadFormat.DuplicateTag"),
    PARSER_BAD_FORMAT_MISSING_TAG(true, true, null, "Parser.BadFormat.MissingTag"),
    PARSER_BAD_FORMAT_NO_TEXT(false, true, null, "Parser.BadFormat.NoText"),
    PARSER_BAD_FORMAT_TEXT_TOO_LONG(false, true, "context", "Parser.BadFormat.TextTooLong"),
    PARSER_BAD_FORMAT_UNEXPECTED_TAG(false, true, "tag", "Parser.BadFormat.UnexpectedTag"),
    PARSER_BAD_FORMAT_UNKNOWN_PARENT_CATEGORY(false, true, null, "Parser.BadFormat.UnknownParentCategory"),
    PARSER_BAD_LOGO(true, false, null, "Parser.BadLogo"),
    PARSER_DUPLICATE_URLS(true, true, "url", "Parser.DuplicateUrls"),
    PARSER_EMPTY_XML(true, false, null, "Parser.EmptyXML"),
    PARSER_GOAL_BAD_ID(false, true, null, "Parser.Goal.BadId"),
    PARSER_GOAL_DUPLICATE(false, true, null, "Parser.Goal.Duplicate"),
    PARSER_GOAL_NO_ATTR(false, true, null, "Parser.Goal.NoAttr"),
    PARSER_GOAL_UNKNOWN_TYPE(true, true, null, "Parser.Goal.UnknownType"),
    PARSER_ITEM_BAD_ENCLOSURE(false, true, null, "Parser.Item.BadEnclosure"),
    PARSER_ITEM_BAD_PDA_URL(false, true, "url", "Parser.Item.BadPdaUrl"),
    PARSER_ITEM_BAD_PUB_DATE(true, true, "date", "Parser.Item.BadPubDate"),
    PARSER_ITEM_BAD_RELATED(false, true, null, "Parser.Item.BadRelated"),
    PARSER_ITEM_BAD_SOURCE(false, true, "url", "Parser.Item.BadSource"),
    PARSER_ITEM_BAD_TITLE(true, true, "url", "Parser.Item.BadTitle"),
    PARSER_ITEM_BAD_URL(true, true, "url", "Parser.Item.BadUrl"),
    PARSER_ITEM_EXTERNAL_LINK(true, true, "url", "Parser.Item.ExternalLink", "Document.BadDomain", "Document.BadDomain.NotAllowed", "Document.Related.BadDomain.NotAllowed"),
    //PARSER_ITEM_EXTERNAL_RELATED_LINK(false, true, "url", "Parser.Item.ExternalRelatedLink"),
    PARSER_ITEM_HTML_TAGS_IN_TEXT(false, true, "url", "Parser.Item.HtmlTagsInText"),
    PARSER_ITEM_INPAGE_ALIAS_NOT_SPECIFIED(false, true, null, "Parser.Item.InpageAliasNotSpecified"),
    PARSER_ITEM_INVALID_COVER_MARKUP(true, true, null, "Parser.Item.InvalidCoverMarkup"),
    PARSER_ITEM_INVALID_TURBO_MARKUP(true, true, null, "Parser.Item.InvalidTurboMarkup"),
    PARSER_ITEM_MORE_THAN_ONE_INPAGE_AD_BLOCK(false, true, null, "Parser.Item.MoreThanOneInpageAdBlock"),
    PARSER_ITEM_NO_PUB_DATE(true, true, "url", "Parser.Item.NoPubDate"),
    PARSER_ITEM_NO_TEXT(true, true, null, "Parser.Item.NoText"),
    PARSER_ITEM_NO_URL(true, true, null, "Parser.Item.NoUrl"),
    PARSER_ITEM_NON_TURBO(true, true, null, "Parser.Item.NonTurbo"),
    PARSER_ITEM_SAME_INPAGE_ALIAS_AND_MAIN_ALIAS(false, true, null, "Parser.Item.SameInpageAliasAndMainAlias"),
    PARSER_ITEM_TOO_MANY_RELATED_ITEMS(false, true, "url", "Parser.Item.TooManyRelatedItems"),
    PARSER_ITEM_TURBO_CONTENT_BAD_SOCIAL(false, true, "info", "Parser.Item.TurboContent.BadSocial"),
    PARSER_ITEM_TURBO_CONTENT_CONTENT_REQUIRED(false, true, null, "Parser.Item.TurboContent.ContentRequired"),
    PARSER_ITEM_TURBO_CONTENT_CONTRADICTING_ATTRIBUTES(false, true, null, "Parser.Item.TurboContent.ContradictingAttributes"),
    PARSER_ITEM_TURBO_CONTENT_CUSTOM_HTML_INVALID_CLASS_ATTR(false, true, null, "Parser.Item.TurboContent.CustomHtmlInvalidClassAttr"),
    PARSER_ITEM_TURBO_CONTENT_CUSTOM_HTML_INVALID_ID_ATTR(false, true, null, "Parser.Item.TurboContent.CustomHtmlInvalidIdAttr"),
    PARSER_ITEM_TURBO_CONTENT_DUPLICATE_ELEMENT_ID(false, true, null, "Parser.Item.TurboContent.DuplicateElementId"),
    PARSER_ITEM_TURBO_CONTENT_DUPLICATE_ON_PAGE_WIDGET_FEEDBACK(false, true, null, "Parser.Item.TurboContent.DuplicateOnPageWidgetFeedback"),
    PARSER_ITEM_TURBO_CONTENT_EMPTY_BLOCK_IN_DYNAMIC_FORM(false, true, null, "Parser.Item.TurboContent.EmptyBlockInDynamicForm"),
    PARSER_ITEM_TURBO_CONTENT_FOLD_BLOCK_WITH_UNSUPPORTED_TAG(false, true, null, "Parser.Item.TurboContent.FoldBlockWithUnsupportedTag"),
    PARSER_ITEM_TURBO_CONTENT_HTML_TAGS_IN_TEXT(false, true, "url", "Parser.Item.TurboContent.HtmlTagsInText"),
    PARSER_ITEM_TURBO_CONTENT_INVALID_BLOCK_TYPE_IN_DYNAMIC_FORM(false, true, null, "Parser.Item.TurboContent.InvalidBlockTypeInDynamicForm"),
    PARSER_ITEM_TURBO_CONTENT_INVALID_HTML_CLOSING_TAG(false, true, null, "Parser.Item.TurboContent.InvalidHtmlClosingTag"),
    PARSER_ITEM_TURBO_CONTENT_INVALID_LINE_TYPE_IN_DYNAMIC_FORM(false, true, null, "Parser.Item.TurboContent.InvalidLineTypeInDynamicForm"),
    PARSER_ITEM_TURBO_CONTENT_INVALID_VALUE(false, true, "info", "Parser.Item.TurboContent.InvalidValue"),
    PARSER_ITEM_TURBO_CONTENT_INVALID_URL(false, true, null, "Parser.Item.TurboContent.InvalidUrl"),
    PARSER_ITEM_TURBO_CONTENT_INVALID_URL_FRAGMENT(false, true, null, "Parser.Item.TurboContent.InvalidUrlFragment"),
    PARSER_ITEM_TURBO_CONTENT_MULTIPLE_BLOCKS_OF_SAME_TYPE_IN_DYNAMIC_FORM(false, true, null, "Parser.Item.TurboContent.MultipleBlocksOfTheSameTypeInDynamicForm"),
    PARSER_ITEM_TURBO_CONTENT_MULTIPLE_CALL_BUTTON_IN_WIDGET_FEEDBACK(false, true, "url", "Parser.Item.TurboContent.MultipleCallButtonInWidgetFeedback"),
    PARSER_ITEM_TURBO_CONTENT_NO_CARDS(false, true, "url", "Parser.Item.TurboContent.NoCards"),
    PARSER_ITEM_TURBO_CONTENT_NO_ITEMS(false, true, null, "Parser.Item.TurboContent.NoItems"),
    PARSER_ITEM_TURBO_CONTENT_NO_ITEMS_IN_CAROUSEL(false, true, "url", "Parser.Item.TurboContent.NoItemsInCarousel"),
    PARSER_ITEM_TURBO_CONTENT_NO_ITEMS_IN_COLLECTION(false, true, "url", "Parser.Item.TurboContent.NoItemsInCollection"),
    PARSER_ITEM_TURBO_CONTENT_NO_ITEMS_IN_HISTOGRAM(false, true, "url", "Parser.Item.TurboContent.NoItemsInHistogram"),
    PARSER_ITEM_TURBO_CONTENT_NO_ITEMS_IN_TABS(false, true, "url", "Parser.Item.TurboContent.NoItemsInTabs"),
    PARSER_ITEM_TURBO_CONTENT_NO_ITEMS_IN_WIDGET_FEEDBACK(false, true, "url", "Parser.Item.TurboContent.NoItemsInWidgetFeedback"),
    PARSER_ITEM_TURBO_CONTENT_NO_REQUIRED_ATTR(false, true, null, "Parser.Item.TurboContent.NoRequiredAttr"),
    PARSER_ITEM_TURBO_CONTENT_NO_TEXT_AND_ICON(false, true, "url", "Parser.Item.TurboContent.NoTextAndIcon"),
    PARSER_ITEM_TURBO_CONTENT_NO_VALID_SOCIAL(false, true, "info", "Parser.Item.TurboContent.NoValidSocial"),
    PARSER_ITEM_TURBO_CONTENT_NOT_ALL_GROUP_ATTRIBUTES_SPECIFIED(false, true, null, "Parser.Item.NotAllGroupAttributesSpecified", "Parser.Item.TurboContent.NotAllGroupAttributesSpecified"),
    PARSER_ITEM_TURBO_CONTENT_RECURSION_DEPTH_LIMIT_REACHED(true, false, null, "Parser.Item.TurboContent.RecursionDepthLimitReached"),
    PARSER_ITEM_TURBO_CONTENT_SELF_CONSTRUCTED_TURBO_URL(false, true, null, "Parser.Item.TurboContent.SelfConstructedTurboUrl"),
    PARSER_ITEM_TURBO_CONTENT_TOO_FEW_OPTIONS_IN_DYNAMIC_FORM(false, true, null, "Parser.Item.TurboContent.TooFewOptionsInDynamicForm"),
    PARSER_ITEM_TURBO_CONTENT_UNKNOWN_CARD_TYPE(false, true, "url", "Parser.Item.TurboContent.UnknownCardType"),
    PARSER_ITEM_TURBO_CONTENT_UNKNOWN_CAROUSEL_ITEM_TYPE(false, true, "info", "Parser.Item.TurboContent.UnknownCarouselItemType"),
    PARSER_ITEM_TURBO_CONTENT_UNKNOWN_ICON(false, true, "info", "Parser.Item.TurboContent.UnknownIcon"),
    PARSER_ITEM_TURBO_CONTENT_UNKNOWN_WIDGET_FEEDBACK_TYPE(false, true, "info", "Parser.Item.TurboContent.UnknownWidgetFeedbackType"),
    PARSER_ITEM_TURBO_CONTENT_UNSUPPORTED_TAG(true, true, null, "Parser.Item.TurboContent.UnsupportedTag"),
    PARSER_ITEM_TURBO_CONTENT_WALL_OF_TEXT(false, true, "url", "Parser.Item.TurboContent.WallOfText"),
    PARSER_NO_CHANNEL(true, false, null, "Parser.NoChannel"),
    PARSER_NO_ITEMS(true, false, null, "Parser.NoItems"),
    PARSER_NO_ITEMS_DUE_TO_CHANGED_FEED(true, false, null),
    PARSER_NOT_RSS(true, false, null, "Parser.NotRss"),
    PARSER_NOT_YML(true, false, null, "Parser.NotYml"),
    //PARSER_TRACKER_BAD_ID(true, false, "type", "Parser.Tracker.BadId"),
    PARSER_TRACKER_DUPLICATE(false, false, "type", "Parser.Tracker.Duplicate"),
    PARSER_TRACKER_INVALID_JSON(true, false, null, "Parser.Tracker.InvalidJson"),
    PARSER_TRACKER_UNKNOWN_TYPE(true, false, "type", "Parser.Tracker.UnknownType"),
    PARSER_XML_ERROR(true, true, null, "Parser.XmlError"),
    @Description("Фид слишком большой")
    PARSER_XML_TOO_BIG(true, false, null, "Parser.XMLTooBig"),
    UNKNOWN(true, false, null, "Unknown"),
    ;

    public static final Set<String> IGNORED_ERRORS = Sets.newHashSet(
            "Parser.NoChannel",
            //"Parser.Item.NoUrl",
            //"Parser.Item.BadTitle",
            "Fetch.Skip",
            "Images.Skip",
            "Parser.Tracker.BadId",
            "Parser.Item.InvalidCoverMarkup",
            "Parser.Item.ExternalRelatedLink",
            "Images.ItemImagesMissing"
    );

    private final static List<TurboErrorType> UNKNOWN_LIST = Collections.singletonList(UNKNOWN);

    private final boolean fatal;
    private final boolean hasItemUrl;
    private final Predicate<ObjectNode> condition;
    private final String mainParam;
    private final String[] codes;

    private static final Map<String, List<TurboErrorType>> errorByCodes = new HashMap<>();
    static {
        for (TurboErrorType type : values()) {
            for (String code : type.codes) {
                errorByCodes.computeIfAbsent(code, k -> new ArrayList<>()).add(type);
            }
        }
    }

    TurboErrorType(boolean fatal, boolean hasItemUrl, String mainParam, String... codes) {
        this(null, fatal, hasItemUrl, mainParam, codes);
    }

    TurboErrorType(Predicate<ObjectNode> condition, boolean fatal, boolean hasItemUrl, String mainParam, String... codes) {
        this.mainParam = mainParam;
        this.condition = condition;
        this.codes = codes;
        this.fatal = fatal;
        this.hasItemUrl = hasItemUrl;
    }

    public boolean isFatal() {
        return fatal;
    }

    public boolean hasItemUrl() {
        return hasItemUrl;
    }

    public String getMainParam() {
        return mainParam;
    }

    @JsonIgnore
    public String[] getCodes() {
        return codes;
    }

    public static TurboErrorType fromRawError(TurboRawError rawError) {
        if (IGNORED_ERRORS.contains(rawError.getCode())) {
            return UNKNOWN;
        }
        List<TurboErrorType> types = errorByCodes.getOrDefault(rawError.getCode(), UNKNOWN_LIST);
        if (types.size() == 1) {
            return types.get(0);
        }
        return types.stream().filter(type -> type.condition == null || type.condition.test(rawError.getParams()))
                .findFirst().orElse(UNKNOWN);
    }

    public static TurboErrorType fromCode(String code) {
        if (IGNORED_ERRORS.contains(code)) {
            return UNKNOWN;
        }
        return errorByCodes.getOrDefault(code, UNKNOWN_LIST).get(0);
    }

    public static boolean isKnown(TurboRawError rawError) {
        return fromCode(rawError.getCode()) != UNKNOWN;
    }

    public static EnumResolver<TurboErrorType> R = new EnumResolver<>(TurboErrorType.class);
}
