package ru.yandex.webmaster3.core.turbo.model.feed;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import ru.yandex.webmaster3.core.data.HttpCodeInfo;
import ru.yandex.webmaster3.core.turbo.TurboConstants;
import ru.yandex.webmaster3.core.turbo.model.error.TurboError;
import ru.yandex.webmaster3.core.turbo.model.error.TurboErrorType;
import ru.yandex.webmaster3.core.turbo.model.error.TurboRawError;
import ru.yandex.wmtools.common.util.http.YandexHttpStatus;

import java.util.stream.Stream;

/**
 * Created by Oleg Bazdyrev on 26/10/2017.
 */
public class TurboErrorInfo {

    private static final ObjectMapper OM = new ObjectMapper();

    private final TurboErrorType code;
    private final String rawCode;
    private final Integer lineNumber;
    private final Integer posNumber;
    private final String message;
    private final ObjectNode params;

    @JsonCreator
    public TurboErrorInfo(@JsonProperty("code") TurboErrorType code,
                          @JsonProperty("lineNumber") Integer lineNumber,
                          @JsonProperty("posNumber") Integer posNumber,
                          @JsonProperty("message") String message,
                          @JsonProperty("params") ObjectNode params) {
        this.code = code;
        this.rawCode = code.getCodes()[0];
        this.lineNumber = lineNumber;
        this.posNumber = posNumber;
        this.message = message;
        this.params = params;
    }

    public static TurboErrorInfo create(TurboErrorType code, Integer line, Integer column,
                                        String message, ObjectNode params) {
        if (params == null) {
            params = OM.createObjectNode();
        }
        if (code.getMainParam() != null) {
            JsonNode mainParam = params.get(code.getMainParam());
            if (mainParam != null && !mainParam.isNull()) {
                message = mainParam.asText();
            }
        }
        // TODO некрасиво (вероятно нужен полиморфизм)
        if (code == TurboErrorType.FETCHER_HTTP && message != null) {
            int httpCode = Integer.parseInt(message);
            HttpCodeInfo httpCodeInfo = new HttpCodeInfo(httpCode, YandexHttpStatus.parseCode(httpCode));
            params.set("httpCodeInfo", OM.convertValue(httpCodeInfo, JsonNode.class));
        }
        if (code == TurboErrorType.IMAGES_FETCH_ERROR && params.has("http_code")) {
            int httpCode = params.get("http_code").asInt();
            HttpCodeInfo httpCodeInfo = new HttpCodeInfo(httpCode, YandexHttpStatus.parseCode(httpCode));
            params.set("httpCodeInfo", OM.convertValue(httpCodeInfo, JsonNode.class));
        }

        // прячем item_url для ошибок, у которых он не имеет смысла
        if (!code.hasItemUrl()) {
            params.remove(TurboConstants.FIELD_ITEM_URL);
        }
        return new TurboErrorInfo(code, line, column, message, params);
    }

    public static Stream<TurboErrorInfo> fromTurboError(TurboError error) {
        if (error.getType() == null || error.getType() == TurboErrorType.UNKNOWN) {
            return Stream.empty();
        }
        return error.getSamples().stream().map(sample ->
                create(error.getType(), sample.getLine(), sample.getColumn(), sample.getText(), sample.getParams()));
    }

    public static TurboErrorInfo fromRawError(TurboErrorType type, TurboRawError rawError) {
        return create(type, rawError.getLine(), rawError.getColumn(), rawError.getText(), rawError.getParams());
    }

    public TurboErrorType getCode() {
        return code;
    }

    public String getRawCode() {
        return rawCode;
    }

    public Integer getLineNumber() {
        return lineNumber;
    }

    public Integer getPosNumber() {
        return posNumber;
    }

    public String getMessage() {
        return message;
    }

    public ObjectNode getParams() {
        return params;
    }
}
