package ru.yandex.webmaster3.core.turbo.model.feed;

import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.Builder;
import lombok.Getter;
import org.joda.time.DateTime;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedSccStatus;
import ru.yandex.webmaster3.core.util.WwwUtil;

/**
 * Фид одного хоста от какого-либо источника
 * Created by Oleg Bazdyrev on 05/07/2017.
 */
@Getter
public class TurboFeedSettings implements TurboFeedSettingsInfo {

    public static final long MAX_VALIDATION_MILLIS = TimeUnit.MINUTES.toMillis(5L);

    private final String domain;
    private final TurboFeedType type;
    private final String url;
    private final boolean active;
    private final TurboFeedState state;
    private final DateTime addDate;
    private final DateTime validateDate;
    private final Long businessId;
    private final Long partnerId;
    private final Long feedId;
    private final NativeFeedSccStatus status;
    private final String errorScc;
    private final DateTime timeScc;


    @Builder(toBuilder = true)
    private TurboFeedSettings(@JsonProperty("domain") String domain,
                              @JsonProperty("type") TurboFeedType type,
                              @JsonProperty("url") String url,
                              @JsonProperty("active") boolean active,
                              @JsonProperty("state") TurboFeedState state,
                              @JsonProperty("addDate") DateTime addDate,
                              @JsonProperty("validateDate") DateTime validateDate,
                              @JsonProperty("businessId") Long businessId,
                              @JsonProperty("partnerId") Long partnerId,
                              @JsonProperty("feedId") Long feedId,
                              @JsonProperty("status") NativeFeedSccStatus status,
                              @JsonProperty("errorScc") String errorScc,
                              @JsonProperty("timeScc") DateTime timeScc) {
        this.domain = domain;
        this.type = type;
        this.url = url;
        this.active = active;
        this.state = state;
        this.addDate = addDate;
        this.validateDate = validateDate;
        this.businessId = businessId;
        this.partnerId = partnerId;
        this.feedId = feedId;
        this.status = status;
        this.errorScc = errorScc;
        this.timeScc = timeScc;
    }

    @Deprecated
    // TODO temp workaroung
    public static TurboFeedSettings internalFeed(WebmasterHostId hostId, TurboFeedType type, String url, boolean active,
                                                 TurboFeedState state, DateTime addDate, DateTime validateDate) {
        return new TurboFeedSettings(WwwUtil.cutWWWAndM(hostId.getPunycodeHostname()),
                type, url, active, state, addDate, validateDate, null, null, null, null, null, null);
    }

    public static TurboFeedSettings internalFeed(String domain, TurboFeedType type, String url, boolean active,
                                                 TurboFeedState state, DateTime addDate, DateTime validateDate,
                                                 Long businessId, Long partnerId, Long feedId,
                                                 NativeFeedSccStatus statusScc, String errorScc, DateTime timeScc) {
        return new TurboFeedSettings(domain, type, url, active, state, addDate, validateDate, businessId,
                partnerId, feedId, statusScc, errorScc, timeScc);
    }

    @Description("Домен, к которому относится фид (хост без www. и m.)")
    public String getDomain() {
        return domain;
    }

    @Description("Тип источника")
    public TurboFeedType getType() {
        return type;
    }

    @Description("Адрес фида")
    public String getUrl() {
        return url;
    }

    @Description("Активен ли данный источник")
    public boolean isActive() {
        return active;
    }

    @Description("Текущий статус источника")
    public TurboFeedState getState() {
        return state;
    }

    @JsonIgnore
    public boolean isNotDeleted() {
        return state != TurboFeedState.DELETED;
    }

    @JsonIgnore
    public boolean isValidating() {
        if (active || state != TurboFeedState.VALIDATING) {
            return false;
        }
        if (validateDate == null || (System.currentTimeMillis() - validateDate.getMillis()) > MAX_VALIDATION_MILLIS) {
            return false;
        }
        return true;
    }

    public DateTime getAddDate() {
        return addDate;
    }

    public DateTime getValidateDate() {
        return validateDate;
    }

    public TurboFeedSettings activate(boolean isActive) {
        return new TurboFeedSettings(domain, type, url, isActive, state, addDate, validateDate, businessId, partnerId
                , feedId, status, errorScc, timeScc);
    }

    public TurboFeedSettings withState(TurboFeedState state) {
        return new TurboFeedSettings(domain, type, url, active, state, addDate, validateDate, businessId, partnerId,
                feedId, status, errorScc, timeScc);
    }

    public TurboFeedSettings withValidateDate() {
        return new TurboFeedSettings(domain, type, url, active, state, addDate, DateTime.now(), businessId, partnerId
                , feedId, status, errorScc, timeScc);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        TurboFeedSettings that = (TurboFeedSettings) o;

        if (!domain.equals(that.domain)) {
            return false;
        }
        return url.equals(that.url);
    }

    @Override
    public int hashCode() {
        int result = domain.hashCode();
        result = 31 * result + url.hashCode();
        return result;
    }

}
