package ru.yandex.webmaster3.core.util;

import java.util.NoSuchElementException;
import java.util.Optional;

/**
 * @author avhaliullin
 */
public abstract class Either<L, R> {
    public abstract Optional<L> getLeft();

    public abstract Optional<R> getRight();

    public abstract R getRightUnsafe();

    public abstract L getLeftUnsafe();

    public abstract boolean isLeft();

    public static class Left<L, R> extends Either<L, R> {
        private final L value;

        public Left(L value) {
            this.value = value;
        }

        @Override
        public Optional<L> getLeft() {
            return Optional.of(value);
        }

        @Override
        public Optional<R> getRight() {
            return Optional.empty();
        }

        @Override
        public boolean isLeft() {
            return true;
        }

        @Override
        public R getRightUnsafe() {
            throw new NoSuchElementException("No value present");
        }

        @Override
        public L getLeftUnsafe() {
            return value;
        }

        public L getValue() {
            return value;
        }
    }

    public static class Right<L, R> extends Either<L, R> {
        private final R value;

        public Right(R value) {
            this.value = value;
        }

        @Override
        public Optional<L> getLeft() {
            return Optional.empty();
        }

        @Override
        public Optional<R> getRight() {
            return Optional.of(value);
        }

        @Override
        public boolean isLeft() {
            return false;
        }

        @Override
        public R getRightUnsafe() {
            return value;
        }

        @Override
        public L getLeftUnsafe() {
            throw new NoSuchElementException("No value present");
        }

        public R getValue() {
            return value;
        }
    }

    public static <L, R> Left<L, R> left(L value) {
        return new Left<L, R>(value);
    }

    public static <L, R> Right<L, R> right(R value) {
        return new Right<L, R>(value);
    }
}
