package ru.yandex.webmaster3.core.util;

import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;

import ru.yandex.webmaster3.core.util.http.YandexMimeType;

/**
 * @author Oleg Bazdyrev
 */
public class MimeUtils {

    private static final String UNKNOWN_MIME_TYPE = "unknown";

    private static final Map<YandexMimeType, String> MIME_TYPE_NAMES = Maps.immutableEnumMap(
            ImmutableMap.<YandexMimeType, String>builder()
                    .put(YandexMimeType.UNKNOWN, UNKNOWN_MIME_TYPE)
                    .put(YandexMimeType.MIME_UNKNOWN, "mime_unknown")
                    .put(YandexMimeType.MIME_TEXT, "text/plain")
                    .put(YandexMimeType.MIME_HTML, "text/html")
                    .put(YandexMimeType.MIME_PDF, "application/pdf")
                    .put(YandexMimeType.MIME_RTF, "text/rtf")
                    .put(YandexMimeType.MIME_DOC, "application/msword")
                    .put(YandexMimeType.MIME_MPEG, "video/mpeg")
                    .put(YandexMimeType.MIME_XML, "text/xml")
                    .put(YandexMimeType.MIME_WML, "text/vnd.wap.wml")
                    .put(YandexMimeType.MIME_SWF, "application/x-shockwave-flash")
                    .put(YandexMimeType.MIME_XLS, "application/vnd.ms-excel")
                    .put(YandexMimeType.MIME_PPT, "application/vnd.ms-powerpoint")
                    .put(YandexMimeType.MIME_IMAGE_JPG, "image/jpeg")
                    .put(YandexMimeType.MIME_IMAGE_PJPG, "image/pjpeg")
                    .put(YandexMimeType.MIME_IMAGE_PNG, "image/png")
                    .put(YandexMimeType.MIME_IMAGE_GIF, "image/gif")
                    .put(YandexMimeType.MIME_DOCX, "pplication/msword")
                    .put(YandexMimeType.MIME_ODT, "application/vnd.oasis.opendocument.text")
                    .put(YandexMimeType.MIME_ODP, "application/vnd.oasis.opendocument.presentation")
                    .put(YandexMimeType.MIME_ODS, "application/vnd.oasis.opendocument.spreadsheet")
                    .put(YandexMimeType.MIME_IMAGE_BMP, "image/x-bmp")
                    .put(YandexMimeType.MIME_WAV, "audio/vnd.wave")
                    .put(YandexMimeType.MIME_ARCHIVE, "application/zip")
                    .put(YandexMimeType.MIME_EXE, "application/octet-stream")
                    .put(YandexMimeType.MIME_ODG, "application/vnd.oasis.opendocument.graphics")
                    .put(YandexMimeType.MIME_GZIP, "application/gzip")
                    .put(YandexMimeType.MIME_XLSX, "application/vnd.ms-excel")
                    .put(YandexMimeType.MIME_PPTX, "application/vnd.ms-powerpoint")
                    .put(YandexMimeType.MIME_JAVASCRIPT, "application/javascript")
                    .put(YandexMimeType.MIME_EPUB, "application/epub+zip")
                    .put(YandexMimeType.MIME_TEX, "application/x-tex")
                    .build());

    private static final Map<String, YandexMimeType> MIME_TYPE_BY_NAMES =
            MIME_TYPE_NAMES.entrySet().stream()
                    .collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey, W3Collectors.acceptAnyMerger()));

    public static String toString(YandexMimeType mimeType) {
        return MIME_TYPE_NAMES.getOrDefault(mimeType, UNKNOWN_MIME_TYPE);
    }

    public static YandexMimeType typeByName(String name) {
        var res = YandexMimeType.UNKNOWN;
        if (name != null) {
            res = MIME_TYPE_BY_NAMES.getOrDefault(name, YandexMimeType.UNKNOWN);
        }

        return res;
    }
}
