package ru.yandex.webmaster3.core.util;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Queue;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author avhaliullin
 */
public class ReflectionUtils {
    private static final List<Class<?>> KNOWN_COLLECTIONS_IMPLEMENTATIONS = new ArrayList<>();

    static {
        KNOWN_COLLECTIONS_IMPLEMENTATIONS.add(HashSet.class);
        KNOWN_COLLECTIONS_IMPLEMENTATIONS.add(TreeSet.class);
        KNOWN_COLLECTIONS_IMPLEMENTATIONS.add(ArrayList.class);
    }

    public static Class primitiveToBoxed(Class clazz) {
        if (!clazz.isPrimitive()) {
            throw new IllegalArgumentException("Class " + clazz + " is not primitive");
        }
        if (clazz == Boolean.TYPE) {
            return Boolean.class;
        } else if (clazz == Character.TYPE) {
            return Character.class;
        } else if (clazz == Byte.TYPE) {
            return Byte.class;
        } else if (clazz == Short.TYPE) {
            return Short.class;
        } else if (clazz == Integer.TYPE) {
            return Integer.class;
        } else if (clazz == Long.TYPE) {
            return Long.class;
        } else if (clazz == Float.TYPE) {
            return Float.class;
        } else if (clazz == Double.TYPE) {
            return Double.class;
        } else {
            throw new RuntimeException("Unknown primitive type " + clazz);
        }
    }

    public static Class<?> getClassFromType(Type type) {
        if (type instanceof Class) {
            return (Class) type;
        } else if (type instanceof ParameterizedType) {
            return (Class<?>) ((ParameterizedType) type).getRawType();
        } else {
            throw new RuntimeException("Unknown type kind: " + type);
        }
    }

    private static Method tryGetMethodForName(Class<?> clazz, String name) {
        try {
            return clazz.getMethod(name);
        } catch (NoSuchMethodException e) {
            return null;
        }
    }

    public static Method getGetterForName(Class<?> clazz, String name) {
        String getterSuffix = name.substring(0, 1).toUpperCase() + name.substring(1);
        Method result = tryGetMethodForName(clazz, "get" + getterSuffix);
        if (result == null) {
            result = tryGetMethodForName(clazz, "is" + getterSuffix);
        }
        return result;
    }

    public static String getNameForSetter(Method method) {
        String name = method.getName();
        if (name.startsWith("set")) {
            char firstChar = Character.toLowerCase(name.charAt(3));
            name = firstChar + name.substring(4);
        }

        return name;
    }

    public static boolean isSetter(Method method) {
        return method.getName().startsWith("set") && method.getName().length() > 3 && method.getParameterTypes().length == 1;
    }

    private static Object instantiateWithDefaults(Class<?> clazz, Constructor<?> constructor) {
        int parametersCount = constructor.getParameterTypes().length;
        Object[] args = new Object[parametersCount];
        for (int i = 0; i < parametersCount; i++) {
            Class<?> paramClass = constructor.getParameterTypes()[i];
            if (paramClass.isPrimitive()) {
                if (paramClass == Float.TYPE) {
                    args[i] = 0f;
                } else if (paramClass == Double.TYPE) {
                    args[i] = 0d;
                } else if (paramClass == Long.TYPE) {
                    args[i] = 0L;
                } else if (paramClass == Integer.TYPE) {
                    args[i] = 0;
                } else if (paramClass == Short.TYPE) {
                    args[i] = (short) 0;
                } else if (paramClass == Byte.TYPE) {
                    args[i] = (byte) 0;
                } else if (paramClass == Boolean.TYPE) {
                    args[i] = false;
                } else {
                    throw new RuntimeException("Unknown primitive class: " + paramClass);
                }
            } else if (paramClass.isEnum() && paramClass.getEnumConstants().length > 0) {
                args[i] = paramClass.getEnumConstants()[0];
            }
            // Не примитивные - null
        }
        try {
            return constructor.newInstance(args);
        } catch (InstantiationException | IllegalAccessException | InvocationTargetException e) {
            throw new RuntimeException("Failed to instantiate class " + clazz + " with constructor " + constructor, e);
        }
    }

    public static Object instantiateWithDefaults(Class<?> clazz) {
        Constructor<?>[] constructors = clazz.getConstructors();
        if (constructors.length == 0) {
            throw new RuntimeException("Failed to instantiate class " + clazz + " no public constructors found");
        }
        int maxArgs = -1;
        Constructor<?> maxArgsConstructor = null;
        for (Constructor<?> constructor : constructors) {
            int parametersCount = constructor.getParameterTypes().length;
            if (parametersCount == 0) {
                return instantiateWithDefaults(clazz, constructor);
            }
            if (parametersCount > maxArgs) {
                maxArgs = parametersCount;
                maxArgsConstructor = constructor;
            }
        }
        return instantiateWithDefaults(clazz, maxArgsConstructor);
    }

    public static Collection newCollection(Class<?> clazz) {
        if (!Collection.class.isAssignableFrom(clazz)) {
            throw new IllegalArgumentException("Class " + clazz + " is not a collection");
        }
        Class<?> implementationClass = null;
        if (Modifier.isAbstract(clazz.getModifiers())) {
            for (Class<?> impl : KNOWN_COLLECTIONS_IMPLEMENTATIONS) {
                if (clazz.isAssignableFrom(impl)) {
                    implementationClass = impl;
                }
            }
            if (implementationClass == null) {
                throw new IllegalArgumentException("Cannot find implementation for collection class " + clazz);
            }
        } else {
            implementationClass = clazz;
        }
        return (Collection) instantiateWithDefaults(implementationClass);
    }

    public static List<Class<?>> listParents(Class<?> clazz) {
        Set<Class<?>> used = new HashSet<>();
        Queue<Class<?>> queue = new ArrayDeque<>();
        List<Class<?>> result = new ArrayList<>();
        queue.offer(clazz);
        while (!queue.isEmpty()) {
            Class<?> current = queue.poll();
            result.add(current);
            Class<?> spr = current.getSuperclass();
            if (spr != null && used.add(spr)) {
                queue.offer(spr);
                result.add(spr);
            }
            List<Class<?>> orderedIfaces = Arrays.asList(current.getInterfaces());
            orderedIfaces.sort(Comparator.comparing(Class::getCanonicalName));//хоть как-нибудь детерминируем
            for (Class<?> iface : orderedIfaces) {
                if (used.add(iface)) {
                    queue.offer(iface);
                    result.add(iface);
                }
            }
        }
        return result;
    }

    public static Method getCallerMethod(JavaMethodWitness witness) {
        return witness.getClass().getEnclosingMethod();
    }

    private static final Pattern LAMBDA_NAME_PATTERN = Pattern.compile("^lambda\\$(.*)\\$\\d+$");

    /**
     * Возвращает нормальное имя явно заведенного метода, даже если передана лямбда.
     * Например, вместо lambda$testCurrentMethod$0 - вернет testCurrentMethod
     *
     * @param method
     * @return
     */
    public static String getEnclosingMethodName(Method method) {
        String result = method.getName();
        Matcher m = LAMBDA_NAME_PATTERN.matcher(result);
        if (m.find()) {
            return m.group(1);
        } else {
            return result;
        }
    }
}
