package ru.yandex.webmaster3.core.util;

import lombok.extern.slf4j.Slf4j;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URIBuilder;
import org.jetbrains.annotations.Nullable;
import ru.yandex.wmtools.common.SupportedProtocols;
import ru.yandex.wmtools.common.util.URLUtil;
import ru.yandex.wmtools.common.util.uri.WebmasterUriUtils;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.List;

/**
 * @author aherman
 */
@Slf4j
public class UrlUtils {
    private static final int MAX_URL_LENGTH = 1024;
    private static final int MAX_HOST_LENGTH = 255;

    @Nullable
    // WMC-8224
    public static native String canonizeUrlForRobot(String url);

    public static String urlToUnicode(String urlString) {
        URL url = null;
        try {
            url = SupportedProtocols.getURL(urlString);
        } catch (MalformedURLException | URISyntaxException e) {
            throwInvalidURLException(urlString, e);
        } catch (SupportedProtocols.UnsupportedProtocolException e) {
            throw new IllegalArgumentException("unexpected protocol was found in url: " + urlString);
        }

        String s;
        final String protocol = url.getProtocol();
        final String authority = ((s = url.getAuthority()) != null && s.length() > 0) ? s : "";
        final String path = ((s = url.getPath()) != null) ? s : "";
        final String query = ((s = url.getQuery()) != null) ? '?' + s : "";
        final String ref = ((s = url.getRef()) != null) ? '#' + s : "";

        return protocol + "://" + IdUtils.IDN.toUnicode(authority) + path + query + ref;
    }

    public static URL prepareUrl(String url, boolean usePunycode) throws IllegalArgumentException {
        return doPrepareUrl(url, usePunycode, false);
    }

    private static URL doPrepareUrl(String urlString, boolean usePunycode, boolean isHost) throws IllegalArgumentException {
        return doPrepareUrl(urlString, usePunycode, isHost, true);
    }

    public static URL doPrepareUrl(String urlString, boolean usePunycode, boolean isHost, boolean validate)
            throws IllegalArgumentException {
        //todo check host length
        if (isHost) {
            checkHostLength(urlString);
        } else {
            checkUrlLength(urlString);
        }

        URL url = null;     // make compiler happy
        try {
            url = SupportedProtocols.getURL(urlString);
        } catch (MalformedURLException | URISyntaxException e) {
            throwInvalidURLException(urlString, e);
        } catch (SupportedProtocols.UnsupportedProtocolException e) {
            throw new IllegalArgumentException("unexpected protocol was found in url: " + urlString);
        }

        if (usePunycode) {
            try {
                String encodedHost = IdUtils.IDN.toASCII(url.getHost());
                if (!encodedHost.equals(url.getHost())) {
                    urlString = url.toString().replace(url.getHost(), encodedHost);
                    url = new URL(urlString);
                }
            } catch (IllegalArgumentException e) {
                /*
                 * IdUtils.IDN.toASCII() method performs some checks on initial and encoded urls and throws
                 * IllegalArgumentException in case url is not valid.
                 */
                throwInvalidURLException(url.getHost(), e);
            } catch (MalformedURLException e) {
                throwInvalidURLException(urlString, e);
            }
        }

        if (!validate) {
            return url;
        }

        if (!isValid(url)) {
            try {
                URI uri = new URI(url.getProtocol(), url.getAuthority(), url.getPath(), url.getQuery(), null);
                url = new URL(uri.toASCIIString());
            } catch (MalformedURLException e) {
                throwInvalidURLException(urlString, e);
            } catch (URISyntaxException e) {
                throwInvalidURLException(urlString, e);
            }
        }
        if (!isValid(url)) {
            throwInvalidURLException(url.toString(), null);
        }
        if (url.getHost().matches("\\d+\\.\\d+\\.\\d+\\.\\d+")) {
            throw new IllegalArgumentException("IP address was used instead of hostname: " + url.getHost());
        }

        return url;
    }

    private static void throwInvalidURLException(String url, Exception e) throws IllegalArgumentException {
        throw new IllegalArgumentException("Invalid url: " + url, e);
    }

    /**
     * Checks if url syntax is valid.
     *
     * @param url url which syntax is to be checked
     * @return true, if url syntax is valid
     */
    public static boolean isValid(String url) {
        return URLUtil.isURLValid(url);
    }

    public static String removeParameter(String url, String parameter) {
        try {
            URIBuilder builder = new URIBuilder(WebmasterUriUtils.toOldUri(url));
            List<NameValuePair> queryParams = builder.getQueryParams();
            queryParams.removeIf(pair -> pair.getName().equals(parameter));
            builder.setParameters(queryParams);
            String result = builder.toString();
            if (result.endsWith("?")) {
                result = result.substring(0, result.length() - 1);
            }
            return result;
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * Checks if url syntax is valid.
     *
     * @param url url which syntax is to be checked
     * @return true, if url syntax is valid
     */
    public static boolean isValid(URL url) {
        return isValid(url.toString());
    }

    private static void checkHostLength(String hostname) throws IllegalArgumentException {
        if (hostname.length() > MAX_HOST_LENGTH) {
            throw new IllegalArgumentException("Hostname too long: " + hostname);
        }
    }

    private static void checkUrlLength(String url) throws IllegalArgumentException {
        if (url.length() > MAX_URL_LENGTH) {
            throw new IllegalArgumentException("Url too long: " + url);
        }
    }

    // Чтобы можно было мокать при тестировании
    public static class CanonizeUrlForRobotWrapper {
        public String canonizeUrlForRobot(String url) {
            return UrlUtils.canonizeUrlForRobot(url);
        }
    }

}
