package ru.yandex.webmaster3.core.util;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.wmtools.common.Constants;

import java.util.ArrayList;
import java.util.List;

import static ru.yandex.webmaster3.core.data.WebmasterHostId.Schema.HTTP;
import static ru.yandex.webmaster3.core.data.WebmasterHostId.Schema.HTTPS;

/**
 * @author avhaliullin
 */
public class WwwUtil {

    public static final String WWW_PREFIX = "www.";
    public static final String M_PREFIX = "m.";

    public static String switchWWW(String hostName) {
        if (hostName.contains(Constants.SCHEME_DELIMITER)) {
            if (hostName.toLowerCase().contains(Constants.SCHEME_DELIMITER + WWW_PREFIX)) {
                return hostName.replace(Constants.SCHEME_DELIMITER + WWW_PREFIX, Constants.SCHEME_DELIMITER);
            } else {
                return hostName.replace(Constants.SCHEME_DELIMITER, Constants.SCHEME_DELIMITER + WWW_PREFIX);
            }
        } else {
            if (hostName.toLowerCase().startsWith(WWW_PREFIX)) {
                return hostName.substring(WWW_PREFIX.length());
            } else {
                return WWW_PREFIX + hostName;
            }
        }
    }

    public static boolean isWWW(String hostName) {
        int lastDotIndex = hostName.lastIndexOf('.');
        if (lastDotIndex <= WWW_PREFIX.length()) {
            return false;
        }
        hostName = hostName.toLowerCase();
        return hostName.startsWith(WWW_PREFIX) || hostName.contains(Constants.SCHEME_DELIMITER + WWW_PREFIX);
    }

    public static boolean isItRight(String hostName, MainMirrorStateEnum state) {
        return isWWW(hostName) ^ MainMirrorStateEnum.WITHOUT_WWW.equals(state);
    }

    public static String getHostName(String hostName, MainMirrorStateEnum state) {
        switch (state) {
            case WITH_WWW: {
                return isWWW(hostName) ? hostName : switchWWW(hostName);
            }
            case WITHOUT_WWW: {
                return isWWW(hostName) ? switchWWW(hostName) : hostName;
            }
            default: {
                return hostName;
            }
        }
    }

    public static boolean equalsIgnoreWww(String host1, String host2) {
        return host1.equalsIgnoreCase(host2) || host1.equalsIgnoreCase(switchWWW(host2));
    }

    public static String cutWww(String domain) {
        int lastDotIndex = domain.lastIndexOf('.');
        if (lastDotIndex > WWW_PREFIX.length() && domain.startsWith(WWW_PREFIX)) {
            return domain.substring(WWW_PREFIX.length());
        }

        return domain;
    }

    public static String cutWWWAndM(String domain) {
        domain = domain.toLowerCase();
        int lastDotIndex = domain.lastIndexOf('.');
        if (lastDotIndex > WWW_PREFIX.length() && domain.startsWith(WWW_PREFIX)) {
            return domain.substring(WWW_PREFIX.length());
        }
        if (lastDotIndex > M_PREFIX.length() && domain.startsWith(M_PREFIX)) {
            return domain.substring(M_PREFIX.length());
        }
        return domain;
    }

    public static String cutWWWAndM(WebmasterHostId hostId) {
        return cutWWWAndM(hostId.getPunycodeHostname());
    }

    /**
     * Генерирует все возможные host_id для домена
     *
     * @param domains
     * @return
     */
    public static List<WebmasterHostId> domainToHostIds(String domain) {
        List<WebmasterHostId> result = new ArrayList<>();
        result.add(new WebmasterHostId(HTTP, domain, HTTP.getDefaultPort()));
        result.add(new WebmasterHostId(HTTPS, domain, HTTPS.getDefaultPort()));
        result.add(new WebmasterHostId(HTTP, "www." + domain, HTTP.getDefaultPort()));
        result.add(new WebmasterHostId(HTTPS, "www." + domain, HTTPS.getDefaultPort()));
        result.add(new WebmasterHostId(HTTP, "m." + domain, HTTP.getDefaultPort()));
        result.add(new WebmasterHostId(HTTPS, "m." + domain, HTTPS.getDefaultPort()));
        return result;
    }

}
