package ru.yandex.webmaster3.core.util.concurrent;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;

/**
 * Аналог ArrayBlockingQueue, но без поддержки стандартных java-интерфейсов,
 * зато со встроенной поддержкой счетчика producer'ов. Метод isExhausted() возвращает true,
 * только если в очереди больше нет элементов и все писатели объявили, что писать больше не будут.
 * ArrayBlockingQueue переиспользовать не удалось, поскольку когда все писатели заканчивают, нужно уведомить всех
 * заблокированных читателей о том, что данных больше не будет - а для этого нужен доступ к соответствующему condition'у
 *
 * @author avhaliullin
 */
public class ExhaustableQueue<T> {
    private final Object[] items;

    private int takeIndex;
    private int putIndex;
    private int count;
    private int leaveWriters;
    private boolean terminated;

    private final ReentrantLock lock;
    private final Condition notEmpty;
    private final Condition notFull;

    public ExhaustableQueue(int size, int writers) {
        this.items = new Object[size];
        this.leaveWriters = writers;
        this.lock = new ReentrantLock();
        this.notEmpty = lock.newCondition();
        this.notFull = lock.newCondition();
        this.terminated = false;
    }

    public void put(T e) throws InterruptedException {
        if (e == null) {
            throw new NullPointerException();
        }
        ReentrantLock lock = this.lock;
        lock.lockInterruptibly();
        try {
            while (!terminated && count == items.length) {
                notFull.await();
            }
            if (terminated) {
                throw new InterruptedException("Queue was terminated");
            }
            Object[] items = this.items;
            items[putIndex] = e;
            if (++putIndex == items.length) {
                putIndex = 0;
            }
            count++;
            notEmpty.signal();
        } finally {
            lock.unlock();
        }
    }

    public T pullBlocking() throws InterruptedException {
        ReentrantLock lock = this.lock;
        lock.lockInterruptibly();
        try {
            while (count == 0 && leaveWriters > 0 && !terminated) {
                notEmpty.await();
            }
            if (terminated) {
                throw new InterruptedException("Queue was terminated");
            }
            if (count > 0) {
                Object[] items = this.items;
                T e = (T) items[takeIndex];
                items[takeIndex] = null;
                if (++takeIndex == items.length) {
                    takeIndex = 0;
                }
                count--;
                notFull.signal();
                return e;
            } else {
                return null;
            }
        } finally {
            lock.unlock();
        }
    }

    public void pullBatch(Consumer<T> consumer, int limit) throws InterruptedException {
        ReentrantLock lock = this.lock;
        lock.lockInterruptibly();
        try {
            while (count == 0 && leaveWriters > 0 && !terminated) {
                notEmpty.await();
            }
            if (terminated){
                throw new InterruptedException("Queue was terminated");
            }
            int sent = 0;
            while (count > 0 && sent < limit) {
                Object[] items = this.items;
                T e = (T) items[takeIndex];
                items[takeIndex] = null;
                if (++takeIndex == items.length) {
                    takeIndex = 0;
                }
                count--;
                consumer.accept(e);
                sent++;
            }
            notFull.signal();
        } finally {
            lock.unlock();
        }
    }

    public void terminate(){
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            terminated = true;
            count = 0;
            putIndex = 0;
            takeIndex = 0;
            leaveWriters = 0;
            notEmpty.signalAll();
            notFull.signalAll();
        } finally {
            lock.unlock();
        }
    }

    public void doneWriting() {
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            if (!terminated) {
                leaveWriters--;
                if (leaveWriters <= 0) {
                    notEmpty.signalAll();
                }
            }
        } finally {
            lock.unlock();
        }
    }

    public boolean isExhausted() {
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            return leaveWriters <= 0 && count <= 0;
        } finally {
            lock.unlock();
        }
    }

    public int size() {
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            return count;
        } finally {
            lock.unlock();
        }
    }

    public int capacity() {
        return items.length;
    }
}
