package ru.yandex.webmaster3.core.util.conductor;

import java.io.IOException;
import java.net.URI;
import java.util.List;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.config.SocketConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.web.util.UriComponentsBuilder;
import ru.yandex.webmaster3.core.http.HttpConstants;

/**
 * @author aherman
 */
public class ConductorClient {
    private final Logger log = LoggerFactory.getLogger(ConductorClient.class);

    private String condutorApiBaseUrl;
    private int connectTimeoutMs = HttpConstants.DEFAULT_CONNECT_TIMEOUT;
    private int socketTimeoutMs = 1_000;
    private int connectionPoolTimeoutMs = HttpConstants.DEFAULT_CONNECTION_REQUEST_TIMEOUT;

    private CloseableHttpClient httpClient;

    private static final ObjectMapper OM = new ObjectMapper();

    public void init() {
        SocketConfig socketConfig = SocketConfig.custom()
                .setSoTimeout(socketTimeoutMs)
                .build();

        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(connectTimeoutMs)
                .setSocketTimeout(socketTimeoutMs)
                .setConnectionRequestTimeout(connectionPoolTimeoutMs)
                .build();

        httpClient = HttpClientBuilder.create()
                .setDefaultSocketConfig(socketConfig)
                .setMaxConnPerRoute(16)
                .setMaxConnTotal(16)
                .setDefaultRequestConfig(requestConfig)
                .build();
    }

    public void destroy() {
        IOUtils.closeQuietly(httpClient);
    }

    public List<ConductorHostInfo> listHostsInGroup(String groupName) throws ConductorException {
        URI requestUri = UriComponentsBuilder.fromUriString(condutorApiBaseUrl)
                .pathSegment("groups2hosts", groupName)
                .queryParam("fields", "fqdn,datacenter_name,root_datacenter_name")
                .queryParam("format", "json")
                .build().toUri();
        log.debug("Load group: " + requestUri);
        HttpGet httpGet = new HttpGet(requestUri);
        String content;
        try (CloseableHttpResponse httpResponse = httpClient.execute(httpGet)) {
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode != HttpStatus.SC_OK) {
                throw new ConductorException("Unable to get group data: " + groupName + " httpStatus=" + statusCode);
            }

            HttpEntity entity = httpResponse.getEntity();
            if (entity == null) {
                throw new ConductorException("Unable to get group data: " + groupName + " response empty");
            }

            content = EntityUtils.toString(entity);
        } catch (IOException e) {
            throw new ConductorException("Unable to get group data: " + groupName, e);
        }

        try {
            List<ConductorHostInfo> conductorHostInfos =
                    OM.readValue(content, new TypeReference<List<ConductorHostInfo>>() {});
            log.debug("Hosts found: {}", conductorHostInfos.size());
            return conductorHostInfos;
        } catch (IOException e) {
            throw new ConductorException("Unable to read conductor response", e);
        }
    }

    public ConductorHostInfo getHostInfo(String hostFqdn) {
        return null;
    }

    @Required
    public void setCondutorApiBaseUrl(String condutorApiBaseUrl) {
        this.condutorApiBaseUrl = condutorApiBaseUrl;
    }

    public void setConnectTimeoutMs(int connectTimeoutMs) {
        this.connectTimeoutMs = connectTimeoutMs;
    }

    public void setSocketTimeoutMs(int socketTimeoutMs) {
        this.socketTimeoutMs = socketTimeoutMs;
    }

    public void setConnectionPoolTimeoutMs(int connectionPoolTimeoutMs) {
        this.connectionPoolTimeoutMs = connectionPoolTimeoutMs;
    }
}
