package ru.yandex.webmaster3.core.util.joda;

import org.joda.time.Duration;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author avhaliullin
 */
public class DurationSerializationUtil {
    private static final String SUFFIX_MILLIS = "ms";
    private static final String SUFFIX_SECONDS = "s";
    private static final String SUFFIX_MINUTES = "m";
    private static final String SUFFIX_HOURS = "h";
    private static final String SUFFIX_DAYS = "d";

    private static final Pattern PATTERN = Pattern.compile("^(\\d+)(" + SUFFIX_MILLIS + "|" + SUFFIX_SECONDS + "|" + SUFFIX_MINUTES + "|" + SUFFIX_HOURS + "|" + SUFFIX_DAYS + ")$");

    public static String serializeDuration(Duration d) {
        long millis = d.getMillis();
        if (millis % 1000L != 0L) {
            return millis + SUFFIX_MILLIS;
        } else {
            long seconds = millis / 1000L;
            if (seconds % 60L != 0L) {
                return seconds + SUFFIX_SECONDS;
            } else {
                long minutes = seconds / 60L;
                if (minutes % 60L != 0L) {
                    return minutes + SUFFIX_MINUTES;
                } else {
                    long hours = minutes / 60L;
                    if (hours % 24L != 0) {
                        return hours + SUFFIX_HOURS;
                    } else {
                        long days = hours / 24L;
                        return days + SUFFIX_DAYS;
                    }
                }
            }
        }
    }

    private static Duration deserializeFromMatcher(Matcher m) {
        long amount = Long.parseLong(m.group(1));
        String unit = m.group(2);
        switch (unit) {
            case SUFFIX_MILLIS:
                return Duration.millis(amount);
            case SUFFIX_SECONDS:
                return Duration.standardSeconds(amount);
            case SUFFIX_MINUTES:
                return Duration.standardMinutes(amount);
            case SUFFIX_HOURS:
                return Duration.standardHours(amount);
            case SUFFIX_DAYS:
                return Duration.standardDays(amount);
            default:
                // не IllegalArgumentException, потому что это баг либы.
                // предполагается, что к этому моменту неизвестные unit'ы уже срезаны регуляркой
                throw new RuntimeException("Unknown duration unit " + unit);
        }
    }

    public static Duration deserializeCompatible(String s) {
        Matcher m = PATTERN.matcher(s);
        if (m.find()) {
            return deserializeFromMatcher(m);
        } else {
            return Duration.parse(s);
        }
    }

    public static Duration deserializeDuration(String s) {
        Matcher m = PATTERN.matcher(s);
        if (!m.find()) {
            throw new IllegalArgumentException("Cannot parse duration from string \"" + s + "\"");
        }
        return deserializeFromMatcher(m);
    }
}
