package ru.yandex.webmaster3.core.util.json;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.BeanProperty;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.deser.ContextualDeserializer;
import com.fasterxml.jackson.databind.deser.std.StdScalarDeserializer;
import org.jetbrains.annotations.Nullable;
import ru.yandex.webmaster3.core.util.enums.IntEnum;
import ru.yandex.webmaster3.core.util.enums.IntEnumResolver;

import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;

/**
 * @author aherman
 */
public class IntEnumJsonDeserializer<T extends Enum<T> & IntEnum> extends StdScalarDeserializer<T> implements ContextualDeserializer {
    @Nullable
    private final IntEnumResolver<T> resolver;
    @Nullable
    private final T unknownValue;

    public IntEnumJsonDeserializer() {
        super((Class<?>) null);
        resolver = null;
        unknownValue = null;
    }

    public IntEnumJsonDeserializer(Class<T> vc) {
        super(vc);
        this.resolver = getResolver(vc);
        T unknown = null;
        try {
            unknown = resolver.getUnknownValue();
        } catch (IllegalArgumentException ign) { // к слову о том, как не надо делать api
        }
        this.unknownValue = unknown;
    }

    public IntEnumJsonDeserializer(Class<T> vc, @Nullable T unknownValue) {
        super(vc);
        this.unknownValue = unknownValue;
        resolver = getResolver(vc);
    }


    @Override
    @SuppressWarnings("unchecked")
    public T deserialize(JsonParser jp, DeserializationContext ctxt) throws IOException {
        if (resolver == null) {
            throw new RuntimeException("Deserialized wasn't initialized");
        }
        if (jp.currentToken() == JsonToken.VALUE_STRING) {
            String value = jp.getValueAsString();
            return resolver.valueOfOrUnknown(value);
        }
        int value = _parseIntPrimitive(jp, ctxt);

        if (resolver != null) {
            return resolver.fromValueOrUnknown(value);
        } else {
            T[] enumConstants = (T[]) handledType().getEnumConstants();
            for (T c : enumConstants) {
                if (c.value() == value) {
                    return c;
                }
            }
        }
        return unknownValue;
    }

    @Override
    public JsonDeserializer<?> createContextual(DeserializationContext ctxt, BeanProperty property) {
        return new IntEnumJsonDeserializer(property.getType().getRawClass());
    }

    @SuppressWarnings("unchecked")
    static <T extends Enum<T> & IntEnum> IntEnumResolver<T> getResolver(Class<T> vc) {
        IntEnumResolver<T> r;
        try {
            Field f = vc.getDeclaredField("R");
            if (!Modifier.isStatic(f.getModifiers())) {
                r = null;
            } else if (!Modifier.isPublic(f.getModifiers())) {
                r = null;
            } else if (!IntEnumResolver.class.isAssignableFrom(f.getType())) {
                r = null;
            } else {
                r = (IntEnumResolver<T>) f.get(null);
            }
        } catch (NoSuchFieldException | IllegalAccessException e) {
            r = null;
        }
        return r;
    }
}
