package ru.yandex.webmaster3.core.util.json;

import java.io.IOException;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WebmasterJsonModule;
import ru.yandex.webmaster3.core.util.joda.jackson.WebmasterDurationModule;

/**
 * ishalaru
 * 27.11.2019
 **/
public class JsonMapping {
    public static final TypeReference<List<String>> STRING_LIST_REFERENCE = new TypeReference<>() {};
    public static final TypeReference<List<Integer>> INTEGER_LIST_REFERENCE = new TypeReference<>() {};
    public static final TypeReference<List<Long>> LONG_LIST_REFERENCE = new TypeReference<>() {};

    public static final ObjectMapper OM = new ObjectMapper()
            .registerModule(new WebmasterJsonModule(false))
            .registerModule(new JodaModule())
            .registerModule(new ParameterNamesModule())
            .registerModule(new WebmasterDurationModule(true))
            .registerModule(new Jdk8Module())
            .setSerializationInclusion(JsonInclude.Include.ALWAYS)
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
            .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS)
            .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS)
            .enable(DeserializationFeature.READ_UNKNOWN_ENUM_VALUES_USING_DEFAULT_VALUE);

    public static final ObjectMapper OM_ASCII = OM.copy()
            .configure(JsonGenerator.Feature.ESCAPE_NON_ASCII, true);

    public static final ObjectMapper OM_PRETTY = OM.copy()
            .enable(SerializationFeature.INDENT_OUTPUT);

    public static final String ERROR_MESSAGE = "Json mapping error";


    private JsonMapping() {
    }

    public static <T> String writeValueAsString(T object) {
        try {
            return OM.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            throw new WebmasterException(ERROR_MESSAGE,
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(JsonMapping.class, ERROR_MESSAGE), e);
        }
    }

    public static <T> String writeValueAsPrettyString(T object) {
        try {
            return OM_PRETTY.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            throw new WebmasterException(ERROR_MESSAGE,
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(JsonMapping.class, ERROR_MESSAGE), e);
        }
    }

    public static <T> String writeValueAsStringAscii(T object) {
        try {
            return OM_ASCII.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            throw new WebmasterException(ERROR_MESSAGE,
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(JsonMapping.class, ERROR_MESSAGE), e);
        }
    }

    public static <T> byte[] writeValueAsBytes(T object) {
        try {
            return OM.writeValueAsBytes(object);
        } catch (JsonProcessingException e) {
            throw new WebmasterException(ERROR_MESSAGE,
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(JsonMapping.class, ERROR_MESSAGE), e);
        }
    }

    public static <T> T readValue(String s, Class<T> clazz) {
        try {
            return OM.readValue(s, clazz);
        } catch (IOException e) {
            throw new WebmasterException(ERROR_MESSAGE,
                    new WebmasterErrorResponse.UnableToReadJsonRequestResponse(JsonMapping.class, e), e);
        }
    }

    public static <T> T readValue(byte[] b, Class<T> clazz) {
        try {
            return OM.readValue(b, clazz);
        } catch (IOException e) {
            throw new WebmasterException(ERROR_MESSAGE,
                    new WebmasterErrorResponse.UnableToReadJsonRequestResponse(JsonMapping.class, e), e);
        }
    }

    public static <T> T readValue(String s, TypeReference<T> typeReference) {
        try {
            return OM.readValue(s, typeReference);
        } catch (IOException e) {
            throw new WebmasterException(ERROR_MESSAGE,
                    new WebmasterErrorResponse.UnableToReadJsonRequestResponse(JsonMapping.class, e), e);
        }
    }

    public static <T> T readValue(JsonParser s, Class<T> clazz) {
        try {
            return OM.readValue(s, clazz);
        } catch (IOException e) {
            throw new WebmasterException(ERROR_MESSAGE,
                    new WebmasterErrorResponse.UnableToReadJsonRequestResponse(JsonMapping.class, e), e);
        }
    }

    public static <T> T readValue(JsonParser s, TypeReference<T> typeReference) {
        try {
            return OM.readValue(s, typeReference);
        } catch (IOException e) {
            throw new WebmasterException(ERROR_MESSAGE,
                    new WebmasterErrorResponse.UnableToReadJsonRequestResponse(JsonMapping.class, e), e);
        }
    }

    public static JsonNode readTree(String s) {
        try {
            return OM.readTree(s);
        } catch (IOException e) {
            throw new WebmasterException(ERROR_MESSAGE,
                    new WebmasterErrorResponse.UnableToReadJsonRequestResponse(JsonMapping.class, e), e);
        }
    }
}
