package ru.yandex.webmaster3.core.util.sitemap;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;

import java.util.List;
import java.util.UUID;
import java.util.stream.Stream;

/**
 * Created by ifilippov5 on 02.09.17.
 */
public class SitemapRecrawlQuotaUtil {

    /**
     * Возвращает количество дат из списка >= boundDate с точностью до дня
     */
    private static int countDaysLaterOrSameBoundDate(Stream<DateTime> dates, DateTime boundDate) {
        return dates.mapToInt(date -> (!date.withTimeAtStartOfDay().isBefore(boundDate.withTimeAtStartOfDay())) ? 1 : 0)
                .sum();
    }

    private static int countDaysLaterOrSameBoundDate(List<DateTime> dates, DateTime boundDate) {
        return countDaysLaterOrSameBoundDate(dates.stream(), boundDate);
    }

    public static int getQuotaUsed(List<DateTime> dates, int maxDaysLookup) {
        DateTime today = DateTime.now().withTimeAtStartOfDay();
        DateTime dateStart = today.minusDays(maxDaysLookup - 1);
        return countDaysLaterOrSameBoundDate(dates, dateStart);
    }

    /**
     * Возвращает месячную квоту, сколько потрачено, ближайший день, когда можно делать новые запросы
     */
    public static SitemapRecrawlQuotaUtil.QuotaUsage computeQuotaInfo(List<DateTime> dates, int monthlyQuota, int maxDaysLookup) {
        DateTime today = DateTime.now().withTimeAtStartOfDay();
        DateTime dateStart = today.minusDays(maxDaysLookup - 1);

        int count = countDaysLaterOrSameBoundDate(dates, dateStart);
        if (count < monthlyQuota) {
            return new SitemapRecrawlQuotaUtil.QuotaUsage(monthlyQuota, count, today);
        }

        int left = 1;
        int right = maxDaysLookup;
        while (left <= right) {
            int mid = (left + right) >> 1;
            int cnt = countDaysLaterOrSameBoundDate(dates, dateStart.plusDays(mid));
            if (cnt < monthlyQuota) {
                right = mid - 1;
            } else {
                left = mid + 1;
            }
        }
        return new SitemapRecrawlQuotaUtil.QuotaUsage(monthlyQuota, count, today.plusDays(left));
    }

    public static boolean quotaIsRemain(List<Pair<UUID, DateTime>> requests, int monthlyQuota, int maxDaysLookup) {
        DateTime today = DateTime.now().withTimeAtStartOfDay();
        DateTime dateStart = today.minusDays(maxDaysLookup - 1);
        return countDaysLaterOrSameBoundDate(requests.stream().map(Pair::getRight), dateStart) < monthlyQuota;
    }

    public static class QuotaUsage {
        private final int monthlyQuota;
        private final int quotaUsed;
        private final DateTime nearestAllowedDay;

        public QuotaUsage(int monthlyQuota, int quotaUsed, DateTime nearestAllowedDay) {
            this.monthlyQuota = monthlyQuota;
            this.quotaUsed = quotaUsed;
            this.nearestAllowedDay = nearestAllowedDay;
        }

        public int getMonthlyQuota() {
            return monthlyQuota;
        }

        public int getQuotaUsed() {
            return quotaUsed;
        }

        public DateTime getNearestAllowedDay() {
            return nearestAllowedDay;
        }
    }
}
