package ru.yandex.webmaster3.core.util.trie;

import org.apache.commons.lang3.ArrayUtils;
import ru.yandex.webmaster3.core.data.WebmasterHostId;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by ifilippov5 on 17.10.17.
 */
public class EncodeHostUtil {
    public static final String WWW = "www.";
    public static final String XN = "xn--";

    public static byte[] hostToByteArray(WebmasterHostId hostId) {
        String name = hostId.getPunycodeHostname();
        boolean www = name.startsWith(WWW);
        if (www) {
            name = name.substring(WWW.length());
        }
        boolean xn = name.startsWith(XN);
        if (xn) {
            name = name.substring(XN.length());
        }
        byte defaultPortsMask = encodePortAsBitMask(hostId.getPort());

        //сразу выделим на массив столько памяти, сколько надо, тогда по пути не случится assureSize
        byte[] key = new byte[name.length() + 1 + (defaultPortsMask == 0 ? 4 : 0)];
        System.arraycopy(name.getBytes(StandardCharsets.US_ASCII), 0, key, 0, name.length());
        ArrayUtils.reverse(key, 0, name.length());
        if (defaultPortsMask == 0) {
            encodePortAsInt(key, name.length(), hostId.getPort());
        }
        key[key.length - 1] = (byte) ((hostId.getSchema() == WebmasterHostId.Schema.HTTP ? nthBit(0) : 0) | (www ? nthBit(1) : 0)
                | (xn ? nthBit(2) : 0));
        key[key.length - 1] |= defaultPortsMask;
        return key;
    }

    /***
     * Можно обойтись одним битом - хранить в нем WebmasterHostId.isDefaultPort(). Сейчас используется два бита, так как это
     * не увеличивает расход памяти, но зато учтены случаи, когда порт является одним из дефолтных, но не совпадает с дефолтным для протокола.
     * (http и 443 порт, https и 80 порт)
     */
    private static byte encodePortAsBitMask(int port) {
        return (byte)((port == WebmasterHostId.DEFAULT_HTTP_PORT ? nthBit(3) : 0) | (port == WebmasterHostId.DEFAULT_HTTPS_PORT ? nthBit(4) : 0));
    }

    private static void encodePortAsInt(byte[] buffer, int offset, int port) {
        TrieBufferWriter writer = new TrieBufferWriter();
        writer.setBuffer(buffer);
        writer.writeInt(offset, port);
    }

    public static WebmasterHostId fromByteArray(byte[] data, int len) {
        byte lastByte = data[len - 1];
        byte defaultPortMask = (byte) (lastByte & (nthBit(3) | nthBit(4)));
        byte[] portData;
        int port;
        if (defaultPortMask == 0) {
            portData = ArrayUtils.subarray(data, len - 5, len - 1);
            data = ArrayUtils.subarray(data, 0, len - 5);
            TrieBufferReader reader = new TrieBufferReader(portData);
            port = reader.readInt();
        } else {
            port = (nthBitIsOne(defaultPortMask, 3) ? WebmasterHostId.DEFAULT_HTTP_PORT : WebmasterHostId.DEFAULT_HTTPS_PORT);
            data = ArrayUtils.subarray(data, 0, len - 1);
        }
        byte hostPrefix = (byte) (lastByte & (nthBit(0) | nthBit(1) | nthBit(2)));

        ArrayUtils.reverse(data, 0, len);
        WebmasterHostId.Schema schema = nthBitIsOne(hostPrefix, 0) ? WebmasterHostId.Schema.HTTP : WebmasterHostId.Schema.HTTPS;
        String punycodeHostname = nthBitIsOne(hostPrefix, 1) ? WWW : "";
        punycodeHostname += nthBitIsOne(hostPrefix, 2) ? XN : "";
        punycodeHostname += new String(data);
        return new WebmasterHostId(schema, punycodeHostname, port);
    }

    public static int nthBit(int n) {
        return (1 << n);
    }

    public static boolean nthBitIsOne(byte value, int n) {
        return (value & (1 << n)) > 0;
    }
}
