package ru.yandex.webmaster3.core.util.xml;

import org.apache.xerces.util.SecurityManager;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.Attributes;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.ext.LexicalHandler;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.*;
import java.io.IOException;
import java.io.InputStream;
import java.util.Stack;

/**
 * Copy-paste from https://stackoverflow.com/questions/4915422/get-line-number-from-xml-node-java
 */
public class PositionalXMLReader {
    final static String LINE_NUMBER_KEY_NAME = "lineNumber";

    public static Document readXML(final InputStream is) throws IOException, SAXException {
        final Document doc;
        SAXParser parser;
        try {
            final SAXParserFactory factory = SAXParserFactory.newInstance();
            parser = factory.newSAXParser();
            final DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
            final DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
            doc = docBuilder.newDocument();
        } catch (final ParserConfigurationException e) {
            throw new RuntimeException("Can't create SAX parser / DOM builder.", e);
        }

        final Stack<Element> elementStack = new Stack<Element>();
        final StringBuilder textBuffer = new StringBuilder();
        final DefaultHandler handler = new MyDefaultHandler(doc, elementStack, textBuffer);
        parser.setProperty("http://xml.org/sax/properties/lexical-handler", handler);
        XMLReader reader = parser.getXMLReader();
        reader.setFeature("http://apache.org/xml/features/validation/schema/augment-psvi", false);
        reader.setFeature("http://apache.org/xml/features/nonvalidating/load-dtd-grammar", false);
        reader.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);
        reader.setFeature("http://xml.org/sax/features/external-parameter-entities", false);
        reader.setFeature("http://xml.org/sax/features/external-general-entities", false);
        reader.setFeature("http://xml.org/sax/features/validation", false);
        reader.setProperty("http://apache.org/xml/properties/security-manager", new SecurityManager());
        parser.parse(is, handler);

        return doc;
    }

    private static class MyDefaultHandler extends DefaultHandler implements LexicalHandler {
        private final Document doc;
        private final Stack<Element> elementStack;
        private final StringBuilder textBuffer;
        private Locator locator;

        public MyDefaultHandler(Document doc, Stack<Element> elementStack, StringBuilder textBuffer) {
            this.doc = doc;
            this.elementStack = elementStack;
            this.textBuffer = textBuffer;
        }

        @Override
        public void setDocumentLocator(final Locator locator) {
            this.locator = locator; // Save the locator, so that it can be used later for line tracking when traversing nodes.
        }

        @Override
        public void startElement(final String uri, final String localName, final String qName, final Attributes attributes)
                throws SAXException {
            addTextIfNeeded();
            final Element el = doc.createElement(qName);
            for (int i = 0; i < attributes.getLength(); i++) {
                el.setAttribute(attributes.getQName(i), attributes.getValue(i));
            }
            el.setUserData(LINE_NUMBER_KEY_NAME, String.valueOf(this.locator.getLineNumber()), null);
            elementStack.push(el);
        }

        @Override
        public void endElement(final String uri, final String localName, final String qName) {
            addTextIfNeeded();
            final Element closedEl = elementStack.pop();
            if (elementStack.isEmpty()) { // Is this the root element?
                doc.appendChild(closedEl);
            } else {
                final Element parentEl = elementStack.peek();
                parentEl.appendChild(closedEl);
            }
        }

        @Override
        public void characters(final char ch[], final int start, final int length) throws SAXException {
            textBuffer.append(ch, start, length);
        }

        // Outputs text accumulated under the current node
        private void addTextIfNeeded() {
            if (textBuffer.length() > 0) {
                final Element el = elementStack.peek();
                final Node textNode = doc.createTextNode(textBuffer.toString());
                el.appendChild(textNode);
                textBuffer.delete(0, textBuffer.length());
            }
        }

        @Override
        public void startDTD(String name, String publicId, String systemId) throws SAXException {
            // ignore
        }

        @Override
        public void endDTD() throws SAXException {
            // ignore
        }

        @Override
        public void startEntity(String name) throws SAXException {
            // ignore
        }

        @Override
        public void endEntity(String name) throws SAXException {
            // ignore
        }

        @Override
        public void startCDATA() throws SAXException {
            addTextIfNeeded();
        }

        @Override
        public void endCDATA() throws SAXException {
            if (textBuffer.length() > 0) {
                final Element el = elementStack.peek();
                final Node textNode = doc.createCDATASection(textBuffer.toString());
                el.appendChild(textNode);
                textBuffer.delete(0, textBuffer.length());
            }
        }

        @Override
        public void comment(char[] ch, int start, int length) throws SAXException {
            // ignore
        }
    }
}
