package ru.yandex.webmaster3.core.util.yson;

import com.google.protobuf.CodedInputStream;
import ru.yandex.webmaster3.core.util.yson.tokens.IYsonToken;
import ru.yandex.webmaster3.core.util.yson.tokens.YsonBooleanToken;
import ru.yandex.webmaster3.core.util.yson.tokens.YsonControlToken;
import ru.yandex.webmaster3.core.util.yson.tokens.YsonToken;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.util.NoSuchElementException;

/**
 * @author avhaliullin
 */
public class YsonBinaryStreamReader implements Closeable {
    private final InputStream rawIn;
    private final CodedInputStream in;
    private IYsonToken next;
    private boolean finished;

    public YsonBinaryStreamReader(InputStream in) {
        this.rawIn = in;
        this.in = CodedInputStream.newInstance(in);
        this.in.setSizeLimit(Integer.MAX_VALUE);
    }

    private int readTag() throws IOException {
        in.resetSizeCounter();
        int result = in.readRawByte();
        while (result == ' ' || result == '\t' || result == '\n') {
            result = in.readRawByte();
        }
        return result;
    }

    public boolean hasNext() throws IOException {
        if (next != null) {
            return true;
        }
        if (finished) {
            return false;
        }
        try {
            if (in.isAtEnd()) {
                finished = true;
                return false;
            }
            int tagInt = readTag();

            YsonNodeTypeTag tag = YsonNodeTypeTag.getByTag(tagInt);
            if (tag == null) {
                YsonControlToken controlToken = YsonToken.createControlToken(tagInt);
                if (controlToken == null) {
                    throw new RuntimeException("Expected tag or control char, but found " + tag);
                }
                next = controlToken;
                return true;
            }
            switch (tag) {
                case BOOL_FALSE:
                    next = YsonBooleanToken.FALSE_NODE;
                    return true;
                case BOOL_TRUE:
                    next = YsonBooleanToken.TRUE_NODE;
                    return true;
                case DOUBLE:
                    double value = in.readDouble();
                    next = YsonToken.createDouble(value);
                    return true;
                case INT64:
                    long longValue = in.readSInt64();
                    next = YsonToken.createSigned(longValue);
                    return true;
                case UINT64:
                    longValue = in.readUInt64();
                    next = YsonToken.createUnsigned(longValue);
                    return true;
                case STRING:
                    int len = in.readSInt32();
                    byte[] data = in.readRawBytes(len);
                    next = YsonToken.createString(data);
                    return true;
                default:
                    throw new RuntimeException("Unknown tag " + tag);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public IYsonToken next() throws IOException {
        if (!hasNext()) {
            throw new NoSuchElementException();
        }
        IYsonToken res = next;
        next = null;
        return res;
    }

    @Override
    public void close() throws IOException {
        finished = true;
        next = null;
        rawIn.close();
    }
}
