package ru.yandex.webmaster3.core.util.yson;

import com.google.protobuf.CodedOutputStream;

import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;

/**
 * @author avhaliullin
 */
public class YsonBinaryStreamWriter implements Closeable {
    private final OutputStream out;
    private final CodedOutputStream protoOut;

    public YsonBinaryStreamWriter(OutputStream out) {
        this.out = out;
        this.protoOut = CodedOutputStream.newInstance(out);
    }

    public void writeMapStart() throws IOException {
        protoOut.writeRawByte('{');
    }

    public void writeMapEnd() throws IOException {
        protoOut.writeRawByte('}');
        writeDelimiter();
    }

    public void writeListStart() throws IOException {
        protoOut.writeRawByte('[');
    }

    public void writeListEnd() throws IOException {
        protoOut.writeRawByte(']');
        writeDelimiter();
    }

    public void writeMapKey(String key) throws IOException {
        byte[] data = key.getBytes(StandardCharsets.US_ASCII);
        writeBinary(data, 0, data.length);
        protoOut.writeRawByte('=');
    }

    public void writeEntity() throws IOException {
        protoOut.writeRawByte('#');
        writeDelimiter();
    }

    public void writeStringValue(String value) throws IOException {
        writeBinaryStringValue(value.getBytes(StandardCharsets.UTF_8));
    }

    public void writeBinaryStringValue(byte[] value) throws IOException {
        writeBinaryStringValue(value, 0, value.length);
    }

    public void writeBinaryStringValue(byte[] value, int offset, int len) throws IOException {
        writeBinary(value, offset, len);
        writeDelimiter();
    }

    public void writeSInt64Value(long value) throws IOException {
        writeTag(YsonNodeTypeTag.INT64);
        writeSInt64(value);
        writeDelimiter();
    }

    public void writeUInt64Value(long value) throws IOException {
        writeTag(YsonNodeTypeTag.UINT64);
        protoOut.writeUInt64NoTag(value);
        writeDelimiter();
    }

    public void writeDoubleValue(double value) throws IOException {
        writeTag(YsonNodeTypeTag.DOUBLE);
        protoOut.writeDoubleNoTag(value);
        writeDelimiter();
    }

    public void writeBooleanValue(boolean value) throws IOException {
        writeTag(value ? YsonNodeTypeTag.BOOL_TRUE : YsonNodeTypeTag.BOOL_FALSE);
        writeDelimiter();
    }

    private void writeBinary(byte[] value, int offset, int len) throws IOException {
        writeTag(YsonNodeTypeTag.STRING);
        writeSInt64(len);
        protoOut.writeRawBytes(value, offset, len);
    }


    private void writeSInt64(long value) throws IOException {
        protoOut.writeSInt64NoTag(value);
    }

    private void writeDelimiter() throws IOException {
        protoOut.writeRawByte(';');
    }

    private void writeTag(YsonNodeTypeTag tag) throws IOException {
        protoOut.writeRawByte(tag.value);
    }

    public void flush() throws IOException {
        protoOut.flush();
    }

    @Override
    public void close() throws IOException {
        flush();
        out.close();
    }

}
