package ru.yandex.webmaster3.core.util.yson;

import com.fasterxml.jackson.core.Base64Variant;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.core.ObjectCodec;
import com.fasterxml.jackson.core.base.ParserBase;
import com.fasterxml.jackson.core.io.IOContext;
import com.google.common.primitives.UnsignedLong;
import ru.yandex.webmaster3.core.util.yson.tokens.IYsonToken;
import ru.yandex.webmaster3.core.util.yson.tokens.YsonBooleanToken;
import ru.yandex.webmaster3.core.util.yson.tokens.YsonControlToken;
import ru.yandex.webmaster3.core.util.yson.tokens.YsonDoubleToken;
import ru.yandex.webmaster3.core.util.yson.tokens.YsonInt64Token;
import ru.yandex.webmaster3.core.util.yson.tokens.YsonStringToken;

import java.io.IOException;
import java.nio.charset.StandardCharsets;

/**
 * @author avhaliullin
 */
public class YsonToJsonParser extends ParserBase {
    private YsonBinaryStreamReader reader;

    private ObjectCodec objectCodec;
    private IYsonToken curYsonToken;
    private boolean nextCanBeKey = false;
    private String currentString;


    public YsonToJsonParser(IOContext ctxt, int features, YsonBinaryStreamReader reader) {
        super(ctxt, features);
        this.reader = reader;
    }

    @Override
    protected void _closeInput() throws IOException {
        if (reader != null) {
            if (_ioContext.isResourceManaged() || isEnabled(Feature.AUTO_CLOSE_SOURCE)) {
                reader.close();
            }
            reader = null;
        }
    }

    @Override
    public JsonToken nextToken() throws IOException {
        curYsonToken = null;
        if (!reader.hasNext()) {
            return null;
        }
        curYsonToken = reader.next();
        boolean canBeKey = nextCanBeKey;
        nextCanBeKey = false;
        currentString = null;
        _numTypesValid = NR_UNKNOWN;
        if (curYsonToken instanceof YsonControlToken) {
            switch ((YsonControlToken) curYsonToken) {
                case DELIMITER:
                    if (_parsingContext.inObject()) {
                        nextCanBeKey = true;
                    }
                    return nextToken();
                case EQ:
                    return nextToken();
                case MAP_START:
                    _parsingContext = _parsingContext.createChildObjectContext(_tokenInputRow, _tokenInputCol);
                    nextCanBeKey = true;
                    return (_currToken = JsonToken.START_OBJECT);
                case MAP_END:
                    if (!_parsingContext.inObject()) {
                        _reportMismatchedEndMarker('}', ']');
                    }
                    _parsingContext = _parsingContext.clearAndGetParent();
                    return (_currToken = JsonToken.END_OBJECT);
                case LIST_START:
                    _parsingContext = _parsingContext.createChildArrayContext(_tokenInputRow, _tokenInputCol);
                    return (_currToken = JsonToken.START_ARRAY);
                case LIST_END:
                    if (!_parsingContext.inArray()) {
                        _reportMismatchedEndMarker(']', '}');
                    }
                    _parsingContext = _parsingContext.clearAndGetParent();
                    return (_currToken = JsonToken.END_ARRAY);
                case ENTITY:
                    return (_currToken = JsonToken.VALUE_NULL);
                default:
                    throw new RuntimeException("Unknown control token " + curYsonToken);
            }
        } else if (curYsonToken instanceof YsonBooleanToken) {
            return (_currToken = ((YsonBooleanToken) curYsonToken).getValue() ? JsonToken.VALUE_TRUE : JsonToken.VALUE_FALSE);
        } else if (curYsonToken instanceof YsonStringToken) {
            if (canBeKey) {
                _parsingContext.setCurrentName(getText());
                return (_currToken = JsonToken.FIELD_NAME);
            } else {
                return (_currToken = JsonToken.VALUE_STRING);
            }
        } else if (curYsonToken instanceof YsonInt64Token) {
            return (_currToken = JsonToken.VALUE_NUMBER_INT);
        } else if (curYsonToken instanceof YsonDoubleToken) {
            return (_currToken = JsonToken.VALUE_NUMBER_FLOAT);
        } else {
            throw new RuntimeException("Unknown token " + curYsonToken);
        }
    }

    @Override
    public String getText() throws IOException {
        if (currentString == null) {
            if (curYsonToken instanceof YsonStringToken) {
                YsonStringToken stringToken = (YsonStringToken) curYsonToken;
                currentString = new String(stringToken.getData(), StandardCharsets.UTF_8);
            } else {
                //TODO: поддержать приведение из других типов
                throw _constructError("Expected string node, but found " + curYsonToken);
            }
        }
        return currentString;
    }

    @Override
    public byte[] getBinaryValue(Base64Variant b64variant) throws IOException {
        if (curYsonToken instanceof YsonStringToken) {
            return ((YsonStringToken) curYsonToken).getData();
        } else {
            throw _constructError("Expected string node, but found " + curYsonToken);
        }
    }

    @Override
    protected void _parseNumericValue(int expType) throws IOException {
        if (curYsonToken instanceof YsonDoubleToken) {
            _numTypesValid = NR_DOUBLE;
            _numberDouble = ((YsonDoubleToken) curYsonToken).getValue();
        } else if (curYsonToken instanceof YsonInt64Token) {
            YsonInt64Token int64Token = (YsonInt64Token) curYsonToken;
            if (int64Token.isSigned()) {
                _numTypesValid = NR_LONG;
                _numberLong = int64Token.getValue();
            } else {
                _numTypesValid = NR_BIGINT;
                _numberBigInt = UnsignedLong.fromLongBits(int64Token.getValue()).bigIntegerValue();
            }
        } else {
            _reportError("Current token (" + curYsonToken + ") not numeric, can not use numeric value accessors");
        }
    }

    @Override
    protected int _parseIntValue() throws IOException {
        _parseNumericValue(NR_INT);
        if ((_numTypesValid & NR_INT) == 0) {
            convertNumberToInt();
        }
        return _numberInt;
    }

    @Override
    public char[] getTextCharacters() throws IOException {
        return getText().toCharArray();
    }

    @Override
    public int getTextLength() throws IOException {
        return getText().length();
    }

    @Override
    public int getTextOffset() throws IOException {
        return 0;
    }

    @Override
    public ObjectCodec getCodec() {
        return objectCodec;
    }

    @Override
    public void setCodec(ObjectCodec c) {
        this.objectCodec = c;
    }
}
