package ru.yandex.webmaster3.core.validator;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.InputStreamBody;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.FileEntity;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.InputStreamBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.web.util.UriComponentsBuilder;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.ActionStatus;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.validator.model.ErrorDescription;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * Клиент для проекта webmaster3-validotr
 * Created by Oleg Bazdyrev on 02/08/2017.
 */
public class XmlValidatorService {

    private static final Logger log = LoggerFactory.getLogger(XmlValidatorService.class);
    private static final int socketTimeoutMs = 60000;
    private static final int connectTimeoutMs = HttpConstants.DEFAULT_CONNECT_TIMEOUT;
    private static final ObjectMapper OM = new ObjectMapper().registerModule(new ParameterNamesModule());

    private URI serviceUrl;
    private CloseableHttpClient httpClient;

    public void init() {
        RequestConfig requestConfig = RequestConfig.custom()
                .setSocketTimeout(socketTimeoutMs)
                .setConnectTimeout(connectTimeoutMs)
                .build();

        httpClient = HttpClients.custom()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionTimeToLive(30, TimeUnit.SECONDS)
                .build();
    }

    public List<ErrorDescription> validateXml(String xml, XmlValidatorSourceType xmlType,
                                              String xsd, XmlValidatorSourceType xsdType, InputStream xmlFile) {
        URI requestUri = UriComponentsBuilder.fromUri(serviceUrl).path("/validate/xml.json")
                .queryParam("xml", xml).queryParam("xmlType", xmlType.name())
                .queryParam("xsd", xsd).queryParam("xsdType", xsdType.name())
                .queryParam("outputFormat", "JSON").build().toUri();

        HttpPost httpPost = new HttpPost(requestUri);
        httpPost.setEntity(MultipartEntityBuilder.create()
                .addPart("xmlFile", new InputStreamBody(xmlFile, "xmlFile")).build());

        try (CloseableHttpResponse response = httpClient.execute(httpPost);
             InputStream content = response.getEntity().getContent()) {
            JsonNode rootNode = OM.readTree(content);
            if (rootNode.get("status").asText().equals(ActionStatus.SUCCESS.name())) {
                String data = rootNode.get("data").get("output").asText();
                if (data.equals("\"ok\"")) {
                    // no errors
                    return new ArrayList<>();
                } else {
                    return OM.readValue(data, ErrorDescriptions.class).errors;
                }
            } else {
                JsonNode errorNode = rootNode.get("errors").get(0);
                String errorMessage = errorNode.get("message").asText();
                log.info("Error when executing ValidateXmlAction: {}", errorMessage);
                throw new WebmasterException(errorMessage,
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), errorMessage));
            }
        } catch (ClientProtocolException e) {
            log.error("ClientProtocolException", e);
            throw new WebmasterException("ClientProtocolException",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "ClientProtocolException"), e);
        } catch (IOException e) {
            String msg = "Error when executing ValidateXmlAction";
            log.error(msg, e);
            throw new WebmasterException(msg, new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), msg), e);
        }
    }

    @Required
    public void setServiceUrl(URI serviceUrl) {
        this.serviceUrl = serviceUrl;
    }

    public static class ErrorDescriptions {

        private final int count;
        private final List<ErrorDescription> errors;

        @JsonCreator
        public ErrorDescriptions(int count, List<ErrorDescription> errors) {
            this.count = count;
            this.errors = errors;
        }

        public int getCount() {
            return count;
        }

        public List<ErrorDescription> getErrors() {
            return errors;
        }
    }
}
