package ru.yandex.webmaster3.core.worker.task;

import java.util.UUID;

import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.verification.VerificationType;

/**
 * @author avhaliullin
 */
@Slf4j
public class VerifyHostTaskData extends WorkerTaskData {
    private final UUID verificationRecordId;
    private final long userId;
    private final VerificationType verificationType;
    private final DateTime nextAttempt;

    public VerifyHostTaskData(
            @JsonProperty("taskId") UUID taskId,
            @JsonProperty("hostId") WebmasterHostId hostId,
            @JsonProperty("verificationRecordId") UUID verificationRecordId,
            @JsonProperty("userId") long userId,
            @JsonProperty("verificationType") VerificationType verificationType,
            @JsonProperty("taskPriority") WorkerTaskPriority taskPriority,
            @JsonProperty("nextAttempt") DateTime nextAttempt) {
        super(taskId, hostId, taskPriority, true);
        this.verificationRecordId = verificationRecordId;
        this.userId = userId;
        this.verificationType = verificationType;
        this.nextAttempt = nextAttempt;
    }

    private VerifyHostTaskData(WebmasterHostId hostId, long userId, UUID verificationRecordId, VerificationType verificationType, WorkerTaskPriority taskPriority, DateTime nextAttempt) {
        super(hostId, taskPriority, true);
        this.verificationRecordId = verificationRecordId;
        this.userId = userId;
        this.verificationType = verificationType;
        this.nextAttempt = nextAttempt;
    }

    public long getUserId() {
        return userId;
    }

    public UUID getVerificationRecordId() {
        return verificationRecordId;
    }

    public VerificationType getVerificationType() {
        return verificationType;
    }

    public DateTime getNextAttempt() {
        return nextAttempt;
    }

    @Override
    public WorkerTaskType getTaskType() {
        return WorkerTaskType.VERIFY_HOST;
    }

    @Override
    public String getShortDescription() {
        return "Verifies host for user";
    }

    public static WorkerTaskData createTaskData(WebmasterHostId hostId, long userId, UUID verificationRecordId, VerificationType verificationType, WorkerTaskPriority taskPriority, DateTime nextAttempt) {
        log.info("create Task for verification record_id - {}", verificationRecordId);
        log.info("create Task for verification host_id - {}, user_id - {}", hostId, userId);
        if (verificationType == VerificationType.WHOIS) {
            return new VerifyWithWhoisTaskData(hostId, userId, verificationRecordId);
        } else {
            return new VerifyHostTaskData(hostId, userId, verificationRecordId, verificationType, taskPriority, nextAttempt);
        }
    }

    public static WorkerTaskData createTaskData(WebmasterHostId hostId, long userId, UUID verificationRecordId, VerificationType verificationType, WorkerTaskPriority taskPriority) {
        log.info("create Task for verification record_id - {}, st - {}", verificationRecordId, Thread.currentThread().getStackTrace());
        log.info("create Task for verification host_id - {}, user_id - {}", hostId, userId);
        if (verificationType == VerificationType.WHOIS) {
            return new VerifyWithWhoisTaskData(hostId, userId, verificationRecordId);
        } else {
            return new VerifyHostTaskData(hostId, userId, verificationRecordId, verificationType, taskPriority, DateTime.now());
        }
    }
}
