package ru.yandex.webmaster3.core.worker.task;

import java.util.UUID;

import com.datastax.driver.core.utils.UUIDs;
import com.fasterxml.jackson.annotation.JsonIgnore;

import ru.yandex.webmaster3.core.data.WebmasterHostId;

/**
 * @author aherman
 */
public abstract class WorkerTaskData {
    private final UUID taskId;
    private final WebmasterHostId hostId;
    private WorkerTaskPriority taskPriority = WorkerTaskPriority.NORMAL;
    private boolean dcAware;

    protected WorkerTaskData(UUID taskId, WebmasterHostId hostId, WorkerTaskPriority taskPriority, boolean dcAware) {
        this.taskId = taskId;
        this.hostId = hostId;
        this.taskPriority = taskPriority;
        this.dcAware = dcAware;
    }

    protected WorkerTaskData(UUID taskId, WebmasterHostId hostId, WorkerTaskPriority taskPriority) {
        this(taskId, hostId, taskPriority, false);
    }

    protected WorkerTaskData(UUID taskId, WebmasterHostId hostId) {
        this(taskId, hostId, WorkerTaskPriority.NORMAL);
    }

    protected WorkerTaskData(WebmasterHostId hostId) {
        this(UUIDs.timeBased(), hostId);
    }

    protected WorkerTaskData(WebmasterHostId hostId, WorkerTaskPriority taskPriority, boolean dcAware) {
        this(UUIDs.timeBased(), hostId, taskPriority, dcAware);
    }

    protected WorkerTaskData(WebmasterHostId hostId, WorkerTaskPriority taskPriority) {
        this(UUIDs.timeBased(), hostId, taskPriority);
    }

    protected WorkerTaskData() {
        this(UUIDs.timeBased(), null);
    }

    public UUID getTaskId() {
        return taskId;
    }

    public WebmasterHostId getHostId() {
        return hostId;
    }

    public WorkerTaskPriority getTaskPriority() {
        return taskPriority;
    }

    public boolean shouldRunLocally() {
        return isDcAware() && taskPriority.isHigher(WorkerTaskPriority.NORMAL);
    }

    public abstract WorkerTaskType getTaskType();

    public int getRetryCount() {
        return 1;
    }

    @JsonIgnore
    protected boolean isDcAware() {
        return dcAware;
    }

    @JsonIgnore
    public abstract String getShortDescription();

    @Override
    public String toString() {
        return getClass().getSimpleName() + "{id=" + getTaskId() + ", type=" + getTaskType() + "  " +
                getShortDescription() + "} ";
    }
}
