package ru.yandex.webmaster3.core.xcelite.writer;

import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.reflections.ReflectionUtils;
import ru.yandex.webmaster3.core.xcelite.column.Col;
import ru.yandex.webmaster3.core.xcelite.column.ColumnsExtractor;
import ru.yandex.webmaster3.core.xcelite.sheet.XceliteSheet;
import ru.yandex.webmaster3.core.xcelite.styles.CellStyles;

import java.lang.reflect.Field;
import java.util.*;

import static org.reflections.ReflectionUtils.withName;

/**
 * @author leonidrom
 */
public class StreamingBeanSheetWriter<T> extends SheetWriterAbs<T> {
    private final LinkedHashSet<Col> columns;
    private org.apache.poi.ss.usermodel.Row headerRow;
    private int rowIndex = 0;
    private final CellStyles cellStyles;

    public StreamingBeanSheetWriter(XceliteSheet sheet, Class<T> type) {
        super(sheet, true);
        ColumnsExtractor extractor = new ColumnsExtractor(type);
        extractor.extract();
        columns = extractor.getColumns();
        this.cellStyles = new CellStyles(sheet.getNativeSheet().getWorkbook());
    }

    @Override
    /*
     * Данная реализация интерфейса позволяет писать как все данные целиком,
     * так и построчно, передавая каждый раз коллекцию из одного элемента.
     */
    public void write(Collection<T> data) {
        if (writeHeader) {
            writeHeader();
        }

        writeData(data);
    }

    @SuppressWarnings("unchecked")
    private void writeData(Collection<T> data) {
        try {
            for (T t : data) {
                org.apache.poi.ss.usermodel.Row row = sheet.getNativeSheet().createRow(rowIndex);
                int i = 0;
                for (Col col : columns) {
                    Set<Field> fields = ReflectionUtils.getAllFields(t.getClass(), withName(col.getFieldName()));
                    Field field = fields.iterator().next();
                    field.setAccessible(true);
                    Object fieldValueObj = field.get(t);
                    Cell cell = row.createCell(i);
                    writeToCell(cell, col, fieldValueObj);
                    i++;
                }
                rowIndex++;
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @SuppressWarnings("unchecked")
    private void writeToCell(Cell cell, Col col, Object fieldValueObj) {
        if (fieldValueObj == null) {
            cell.setCellValue((String) null);
            return;
        }

        if (col.getDataFormat() != null) {
            cell.setCellStyle(cellStyles.getCustomDataFormatStyle(col.getDataFormat()));
        }

        if (col.getType() == Date.class) {
            if (col.getDataFormat() == null) {
                cell.setCellStyle(cellStyles.getDateStyle());
            }
        }

        writeToCell(cell, fieldValueObj, col.getType());
    }

    private void writeHeader() {
        headerRow = sheet.getNativeSheet().createRow(rowIndex);
        rowIndex++;
        addHeaderColumns();

        writeHeader = false;
    }

    private void addHeaderColumns() {
        if (headerRow == null) {
            return;
        }

        int i = headerRow.getLastCellNum() == -1 ? 0 : headerRow.getLastCellNum();
        for (Col column : columns) {
            Cell cell = headerRow.createCell(i);
            cell.setCellType(CellType.STRING);
            cell.setCellStyle(cellStyles.getBoldStyle());
            cell.setCellValue(column.getName());
            i++;
        }
    }
}
