package ru.yandex.webmaster3.core.zora;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import com.fasterxml.jackson.databind.node.TreeTraversingParser;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.jetbrains.annotations.NotNull;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.security.tvm.TVMTokenService;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;
import ru.yandex.webmaster3.core.zora.data.request.ZoraPDFetchRequest;
import ru.yandex.webmaster3.core.zora.data.request.ZoraRequest;
import ru.yandex.webmaster3.core.zora.data.request.ZoraUrlFetchRequest;
import ru.yandex.webmaster3.core.zora.data.response.ZoraRawPDFetchResponse;
import ru.yandex.webmaster3.core.zora.data.response.ZoraRawUrlFetchResponse;
import ru.yandex.webmaster3.core.zora.data.response.ZoraResponseWithRaw;

/**
 * @author avhaliullin
 * Хождение в http json api зоры для получения информации об урле
 */
@Slf4j
public class OfflineZoraService extends AbstractExternalAPIService {
    private static final String PATH = "/request/json?format_output=1&escape=b64&enum=name";
    public static final ObjectMapper OM = new ObjectMapper()
            .registerModule(new ParameterNamesModule())
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
            .setSerializationInclusion(JsonInclude.Include.NON_NULL);

    private static final String TVM_TOKEN_FIELD = "TvmServiceTicket";

    static {
        OM.disable(SerializationFeature.WRITE_NULL_MAP_VALUES);
    }

    private CloseableHttpClient httpClient;
    // клиент который обслуживает realtime запросы из проверки ответа
    private CloseableHttpClient realtimeHttpClient;
    @Setter
    private String zoraUrl;
    @Setter
    private String zoraOnlineUrl; // ZORAOPS-285
    @Setter
    private TVMTokenService tvmTokenService;

    public void init() {
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(1)
                .setConnectTimeout(100)
                .setSocketTimeout(5 * 60_000)
                .build();

        httpClient = HttpClients.custom()
                .disableCookieManagement()
                .setMaxConnPerRoute(150)
                .setMaxConnTotal(150)
                .setConnectionTimeToLive(130, TimeUnit.SECONDS)
                .setDefaultRequestConfig(requestConfig)
                .build();
        realtimeHttpClient = HttpClients.custom()
                .disableCookieManagement()
                .setMaxConnPerRoute(150)
                .setMaxConnTotal(150)
                .setConnectionTimeToLive(25, TimeUnit.SECONDS)
                .setDefaultRequestConfig(requestConfig)
                .build();
    }

    @NotNull
    private <T> ZoraResponseWithRaw<T> executeRequest(ZoraRequest request, Class<T> respClass,
                                                      boolean isBatchQuery, CloseableHttpClient httpClient) {
        String baseUrl = zoraOnlineUrl;
        if (isBatchQuery) {
            baseUrl = zoraUrl;
        }
        HttpPost httpReq = new HttpPost(baseUrl + PATH);
        String reqString;
        ObjectNode reqJson;
        JsonNode responseJson = null;
        try {
            reqJson = OM.valueToTree(request);
            log.info("Zora request: {}, url: {}", OM.writeValueAsString(reqJson), baseUrl); // without tvm
            reqJson.set(TVM_TOKEN_FIELD, new TextNode(tvmTokenService.getToken())); // аутентификация
            reqString = OM.writeValueAsString(reqJson);
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Failed to serialize request", e);
        }
        HttpEntity entity = new StringEntity(reqString, ContentType.APPLICATION_JSON);
        httpReq.setEntity(entity);
        try (CloseableHttpResponse response = httpClient.execute(httpReq)) {
            HttpEntity responseEntity = response.getEntity();
            if (responseEntity == null) {
                log.error("Zora responded with no body");
            } else if (responseEntity.getContentType() == null || !"application/json".equals(responseEntity.getContentType().getValue())) {
                log.error("Zora responded with non-json body: Content-Type=" + responseEntity.getContentType() + " " +
                        "body=" + IOUtils.toString(responseEntity.getContent(), StandardCharsets.UTF_8));
            } else {
                responseJson = OM.readTree(responseEntity.getContent());
//                String bodyString = OM.writeValueAsString(responseJson);
//                log.info("Zora response: {}", bodyString);
            }
            if (response.getStatusLine().getStatusCode() != 200) {
                throw buildException("Zora udf request finished with status " + response.getStatusLine().getStatusCode(), null);
            }
            if (responseJson == null) {
                throw buildException("Failed to parse json response", null);
            }
            return new ZoraResponseWithRaw(OM.readValue(new TreeTraversingParser(responseJson), respClass), reqJson,
                    responseJson);
        } catch (IOException e) {
            throw buildException("Zora udf request failed for url - " + request.getUrl() , e);
        }
    }

    @ExternalDependencyMethod("urlfetch")
    public ZoraResponseWithRaw<ZoraRawUrlFetchResponse> fetchUrl(ZoraUrlFetchRequest request) {
        return trackQuery(new JavaMethodWitness() {
                          }, ALL_ERRORS_INTERNAL, () -> executeRequest(request, ZoraRawUrlFetchResponse.class,
                request.isOnline(), httpClient)
        );
    }

    @ExternalDependencyMethod("pdfetch")
    public ZoraResponseWithRaw<ZoraRawPDFetchResponse> fetchUrl(ZoraPDFetchRequest request) {
        return trackQuery(new JavaMethodWitness() {
                          }, ALL_ERRORS_INTERNAL, () ->
                        executeRequest(request, ZoraRawPDFetchResponse.class, request.isOnline(), httpClient)
        );
    }
    @ExternalDependencyMethod("pdfetch")
    public ZoraResponseWithRaw<ZoraRawPDFetchResponse> realtimeFetchUrl(ZoraPDFetchRequest request) {
        return trackQuery(new JavaMethodWitness() {
                          }, ALL_ERRORS_INTERNAL, () ->
                        executeRequest(request, ZoraRawPDFetchResponse.class, request.isOnline(), realtimeHttpClient)
        );
    }


    private static WebmasterException buildException(String message, Throwable cause) {
        return new WebmasterException(message,
                new WebmasterErrorResponse.InternalUnknownErrorResponse(OfflineZoraService.class, message), cause);
    }

}
