package ru.yandex.webmaster3.core.zora.data.request;

import java.net.URI;
import java.net.URL;
import java.util.Collections;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;
import org.joda.time.Duration;
import org.joda.time.ReadableInstant;

import ru.yandex.webmaster3.core.zora.ZoraSourceEnum;
import ru.yandex.webmaster3.core.zora.data.common.ZoraDuration;
import ru.yandex.webmaster3.core.zora.data.common.ZoraInstant;
import ru.yandex.webmaster3.core.zora.data.fetch.ZoraFetchOpts;
import ru.yandex.webmaster3.core.zora.data.fetch.ZoraHttpHeaders;
import ru.yandex.webmaster3.core.zora.data.fetch.ZoraPeerVerifyMode;
import ru.yandex.webmaster3.core.zora.data.fetch.ZoraSslCertPolicy;
import ru.yandex.webmaster3.core.zora.data.pdfetch.ZoraPDFetchOpts;
import ru.yandex.webmaster3.core.zora.data.redir.ZoraRedirMode;
import ru.yandex.webmaster3.core.zora.data.redir.ZoraRedirSettings;
import ru.yandex.wmtools.common.util.uri.WebmasterUriUtils;

/**
 * @author avhaliullin
 */
public class AbstractZoraRequest implements ZoraUrlFetchRequest, ZoraPDFetchRequest {
    @JsonProperty("SourceName")
    private final String sourceName;
    @JsonProperty("Timeout")
    private final ZoraDuration timeout;
    @JsonProperty("Fetch")
    private final ZoraFetchOpts fetch;
    @JsonProperty("Calc")
    private final CalcOpts calc;
    @JsonProperty("PDFetch")
    private final ZoraPDFetchOpts pdFetchOpts;
    private final AbstractBuilder<?> builder;

    private AbstractZoraRequest(String sourceName, ZoraDuration timeout, ZoraFetchOpts fetch, CalcOpts calc, ZoraPDFetchOpts pdFetchOpts, AbstractBuilder<?> builder) {
        this.sourceName = sourceName;
        this.timeout = timeout;
        this.fetch = fetch;
        this.calc = calc;
        this.pdFetchOpts = pdFetchOpts;
        this.builder = builder;
    }

    @JsonIgnore
    public AbstractBuilder<?> getBuilder() {
        return builder;
    }

    @Override
    public String getUrl() {
        return fetch.getUrl();
    }

    @Override
    @JsonIgnore
    public ZoraSourceEnum getSourceEnum() {
        return ZoraSourceEnum.getBySourceName(builder.source);
    }

    public String getSourceName() {
        return sourceName;
    }

    public ZoraDuration getTimeout() {
        return timeout;
    }

    public ZoraFetchOpts getFetch() {
        return fetch;
    }

    public CalcOpts getCalc() {
        return calc;
    }

    public ZoraPDFetchOpts getPdFetchOpts() {
        return pdFetchOpts;
    }

    @Override
    @JsonIgnore
    public boolean isOnline() {
        return fetch.getRequestType() == ZoraFetchOpts.RequestType.ONLINE;
    }

    static class CalcOpts {
        @JsonProperty("ProcId")
        private final String procId;
        @JsonProperty("ParamsFormat")
        private final ZoraParamsFormat paramsFormat;

        @JsonCreator
        public CalcOpts(String procId, ZoraParamsFormat paramsFormat) {
            this.procId = procId;
            this.paramsFormat = paramsFormat;
        }

        public String getProcId() {
            return procId;
        }

        public ZoraParamsFormat getParamsFormat() {
            return paramsFormat;
        }
    }

    abstract static class AbstractBuilder<Self extends AbstractBuilder<Self>> implements Cloneable {
        private Duration timeout = Duration.standardSeconds(40L);
        private String source = "webmaster";
        private Duration freshness = Duration.ZERO;
        private URI url;
        //        private boolean ignoreRobotsTxt = false;
        private int redirDepth = 0;
        private ReadableInstant ifModifiedSince;
        protected String pdFetchProcId;
        protected boolean includeDoc = true;
        protected boolean includeRobots = false;
        private ZoraSslCertPolicy sslCertPolicy;
        private ZoraFetchOpts.RequestType requestType = ZoraFetchOpts.RequestType.ONLINE;
        private Long priority;

        protected abstract Self getSelf();

        public Self timeout(Duration timeout) {
            this.timeout = timeout;
            return getSelf();
        }

        public Self source(ZoraSourceEnum source) {
            this.source = source.getSourceName();
            return getSelf();
        }

        public Self priority(Long priority) {
            this.priority = priority;
            return getSelf();
        }

        public Self freshness(Duration freshness) {
            this.freshness = freshness;
            return getSelf();
        }

        public Self requestType(ZoraFetchOpts.RequestType requestType) {
            this.requestType = requestType;
            return getSelf();
        }
//        public Builder ignoreRobotsTxt(boolean ignoreRobotsTxt) {
//            this.ignoreRobotsTxt = ignoreRobotsTxt;
//            return getSelf();
//        }

        public Self redirDepth(int redirDepth) {
            this.redirDepth = redirDepth;
            return getSelf();
        }

        public Self sslCertPolicy(ZoraPeerVerifyMode peerVerifyMode) {
            sslCertPolicy = new ZoraSslCertPolicy(peerVerifyMode);
            return getSelf();
        }

        public URI getUrl() {
            return url;
        }

        public Self url(URI url) {
            this.url = url;
            return getSelf();
        }

        public Self url(URL url) {
            return url(url.toString());
        }

        public Self url(String url) {
            try {
                return url(WebmasterUriUtils.toOldUri(url));
            } catch (Exception e) {
                throw new RuntimeException("Unable to convert host URL to URI", e);
            }
        }

        public Self ifModifiedSince(ReadableInstant date) {
            this.ifModifiedSince = date;
            return getSelf();
        }

        protected AbstractZoraRequest _build() {
            ZoraPDFetchOpts pdFetchOpts = null;
            if (pdFetchProcId != null) {
                pdFetchOpts = new ZoraPDFetchOpts(
                        pdFetchProcId,
                        ZoraParamsFormat.EDF_JSON,
                        new ZoraPDFetchOpts.Params(includeDoc, includeRobots)
                );
            }
            try {
                return new AbstractZoraRequest(
                        Preconditions.checkNotNull(source),
                        ZoraDuration.fromDuration(Preconditions.checkNotNull(timeout)),
                        new ZoraFetchOpts(
                                requestType,
                                Preconditions.checkNotNull(url).toASCIIString(),
                                Preconditions.checkNotNull(ZoraDuration.fromDuration(freshness)),
                                //                            ignoreRobotsTxt,
                                redirDepth > 0 ? new ZoraRedirSettings(ZoraRedirMode.Last, redirDepth) : null,
                                sslCertPolicy,
                                new ZoraHttpHeaders(Collections.emptyList(), ZoraInstant.fromInstant(ifModifiedSince)),
                                priority
                        ),
                        new CalcOpts("robot", ZoraParamsFormat.EDF_JSON),
                        pdFetchOpts,
                        (AbstractBuilder<?>) clone()
                );
            } catch (CloneNotSupportedException e) {
                throw new RuntimeException(e);
            }
        }
    }
}
