package ru.yandex.webmaster3.core.util;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

/**
 * @author aherman
 */
public class CityHash102Test {
    private static final long k0 = 0xc3a5c85c97cb3127L;
    private static final long kSeed0 = 1234567;
    private static final long kSeed1 = k0;
    private static final int kDataSize = 1 << 20;
    private static final int kTestSize = 300;

    private byte data[];

    @Before
    public void setUp() throws Exception {
        data = new byte[kDataSize];

        long a = 9;
        long b = 777;
        for (int i = 0; i < kDataSize; i++) {
            a = (a ^ (a >>> 41)) * k0 + b;
            b = (b ^ (b >>> 41)) * k0 + i;
            data[i] = (byte) (b >>> 37);
        }
    }

    @Test
    public void testHashes() throws Exception {
        for (int i = 0; i < kTestSize - 1; i++) {
            test(testdata[i], i * i, i);
        }
        test(testdata[kTestSize - 1], 0, kDataSize);
    }

    private void test(long[] expected, int offset, int len) {
        System.out.println("Test " + offset + ", " + len);
        long ch64 = CityHash102.cityHash64(data, offset, len);
        long ch64Seed = CityHash102.cityHash64WithSeed(data, offset, len, kSeed0);
        long ch64Seed2 = CityHash102.cityHash64WithSeed(data, offset, len, kSeed0, kSeed1);
        long[] u = CityHash102.cityHash128(data, offset, len);
        long[] v = CityHash102.cityHash128WithSeed(data, offset, len, kSeed0, kSeed1);

        System.out.println("CH64 C(" + Long.toHexString(ch64) + ")");
        Assert.assertEquals(expected[0], ch64);
        Assert.assertEquals(expected[1], ch64Seed);
        Assert.assertEquals(expected[2], ch64Seed2);
        Assert.assertEquals(expected[3], u[0]);
        Assert.assertEquals(expected[4], u[1]);
        Assert.assertEquals(expected[5], v[0]);
        Assert.assertEquals(expected[6], v[1]);
    }

    static final long testdata[][] = new long[][] {
            {0x9ae16a3b2f90404fL, 0x75106db890237a4aL, 0x3feac5f636039766L, 0x3df09dfc64c09a2bL, 0x3cb540c392e51e29L, 0x6b56343feac0663L, 0x5b7bc50fd8e8ad92L},
            {0x75e9dee28ded761dL, 0x931992c1b14334c5L, 0x245eeb25ba2c172eL, 0x1290f0e8a5caa74dL, 0xca4c6bf7583f5cdaL, 0xe1d60d51632c536dL, 0xcbc54a1db641910aL},
            {0x75de892fdc5ba914L, 0xf89832e71f764c86L, 0x39a82df1f278a297L, 0xb4af8ae673acb930L, 0x992b7acb203d8885L, 0x57b533f3f8b94d50L, 0xbbb69298a5dcf1a1L},
            {0x69cfe9fca1cc683aL, 0xe65f2a81e19b8067L, 0x20575ea6370a9d14L, 0x8f52532fc6f005b7L, 0x4ebe60df371ec129L, 0xc6ef8a7f8deb8116L, 0x83df17e3c9bb9a67L},
            {0x675b04c582a34966L, 0x53624b5ef8cd4f45L, 0xc412e0931ac8c9b1L, 0x798637e677c65a3L, 0x83e3b06adc4cd3ffL, 0xf3e76e8a7135852fL, 0x111e66cfbb05366dL},
            {0x46fa817397ea8b68L, 0xcc960c1c15ce2d20L, 0xe5f9f947bafb9e79L, 0xb342cdf0d7ac4b2aL, 0x66914d44b373b232L, 0x261194e76cb43966L, 0x45a0010190365048L},
            {0x406e959cdffadec7L, 0xe80dc125dca28ed1L, 0xe5beb146d4b79a21L, 0xe66d5c1bb441541aL, 0xd14961bc1fd265a2L, 0xe4cc669d4fc0577fL, 0xabf4a51e36da2702L},
            {0x46663908b4169b95L, 0x4e7e90b5c426bf1dL, 0xdc660b58daaf8b2cL, 0xb298265ebd1bd55fL, 0x4a5f6838b55c0b08L, 0xfc003c97aa05d397L, 0x2fb5adad3380c3bcL},
            {0xf214b86cffeab596L, 0x5fccb0b132da564fL, 0x86e7aa8b4154b883L, 0x763529c8d4189ea8L, 0x860d77e7fef74ca3L, 0x3b1ba41191219b6bL, 0x722b25dfa6d0a04bL},
            {0xeba670441d1a4f7dL, 0xeb6b272502d975faL, 0x69f8d424d50c083eL, 0x313d49cb51b8cd2cL, 0x6e982d8b4658654aL, 0xdd59629a17e5492dL, 0x81cb23bdab95e30eL},
            {0x172c17ff21dbf88dL, 0x1f5104e320f0c815L, 0x1e34e9f1fa63bcefL, 0x3506ae8fae368d2aL, 0x59fa2b2de5306203L, 0x67d1119dcfa6007eL, 0x1f7190c648ad9aefL},
            {0x5a0838df8a019b8cL, 0x73fc859b4952923L, 0x45e39daf153491bdL, 0xa9b91459a5fada46L, 0xde0fbf8800a2da3L, 0x21800e4b5af9dedbL, 0x517c3726ae0dbae7L},
            {0x8f42b1fbb2fc0302L, 0x5ae31626076ab6caL, 0xb87f0cb67cb75d28L, 0x2498586ac2e1fab2L, 0xe683f9cbea22809aL, 0xa9728d0b2bbe377cL, 0x46baf5cae53dc39aL},
            {0x72085e82d70dcea9L, 0x32f502c43349ba16L, 0x5ebc98c3645a018fL, 0xc7fa762238fd90acL, 0x8d03b5652d615677L, 0xa3f5226e51d42217L, 0x46d5010a7cae8c1eL},
            {0x32b75fc2223b5032L, 0x246fff80eb230868L, 0xa6fdbc82c9aeecc0L, 0xc089498074167021L, 0xab094a9f9ab81c23L, 0x4facf3d9466bcb03L, 0x57aa9c67938cf3ebL},
            {0xe1dd010487d2d647L, 0x12352858295d2167L, 0xacc5e9b6f6b02dbbL, 0x1c66ceea473413dfL, 0xdc3f70a124b25a40L, 0x66a6dfe54c441cd8L, 0xb436dabdaaa37121L},
            {0x2994f9245194a7e2L, 0xb7cd7249d6db6c0cL, 0x2170a7d119c5c6c3L, 0x8505c996b70ee9fcL, 0xb92bba6b5d778eb7L, 0x4db4c57f3a7a4aeeL, 0x3cfd441cb222d06fL},
            {0x32e2ed6fa03e5b22L, 0x58baf09d7c71c62bL, 0xa9c599f3f8f50b5bL, 0x1660a2c4972d0fa1L, 0x1a1538d6b50a57cL, 0x8a5362485bbc9363L, 0xe8eec3c84fd9f2f8L},
            {0x37a72b6e89410c9fL, 0x139fec53b78cee23L, 0x4fccd8f0da7575c3L, 0x3a5f04166518ac75L, 0xf49afe05a44fc090L, 0xcb01b4713cfda4bdL, 0x9027bd37ffc0a5deL},
            {0x10836563cb8ff3a1L, 0xd36f67e2dfc085f7L, 0xedc1bb6a3dcba8dfL, 0xbd4f3a0566df3bedL, 0x81fc8230c163dcbeL, 0x4168bc8417a8281bL, 0x7100c9459827c6a6L},
            {0x4dabcb5c1d382e5cL, 0x9a868c608088b7a4L, 0x7b2b6c389b943be5L, 0xc914b925ab69fda0L, 0x6bafe864647c94d7L, 0x7a48682dd4afa22L, 0x40fe01210176ba10L},
            {0x296afb509046d945L, 0xc38fe9eb796bd4beL, 0xd7b17535df110279L, 0xdd2482b87d1ade07L, 0x662785d2e3e78ddfL, 0xeae39994375181bbL, 0x9994500c077ee1dbL},
            {0xf7c0257efde772eaL, 0xaf6af9977ecf7bffL, 0x1cdff4bd07e8d973L, 0xfab1f4acd2cd4ab4L, 0xb4e19ba52b566bdL, 0x7f1db45725fe2881L, 0x70276ff8763f8396L},
            {0x61e021c8da344ba1L, 0xcf9c720676244755L, 0x354ffa8e9d3601f6L, 0x44e40a03093fbd92L, 0xbda9481cc5b93caeL, 0x986b589cbc0cf617L, 0x210f59f074044831L},
            {0xc0a86ed83908560bL, 0x440c8b6f97bd1749L, 0xa99bf2891726ea93L, 0xac0c0b84df66df9dL, 0x3ee2337b437eb264L, 0x8a341daed9a25f98L, 0xcc665499aa38c78cL},
            {0x35c9cf87e4accbf3L, 0x2267eb4d2191b2a3L, 0x80217695666b2c9L, 0xcd43a24abbaae6dL, 0xa88abf0ea1b2a8ffL, 0xe297ff01427e2a9dL, 0x935d545695b2b41dL},
            {0xe74c366b3091e275L, 0x522e657c5da94b06L, 0xca9afa806f1a54acL, 0xb545042f67929471L, 0x90d10e75ed0e75d8L, 0x3ea60f8f158df77eL, 0x8863eff3c2d670b7L},
            {0xa3f2ca45089ad1a6L, 0x13f6270fe56fbce4L, 0x1f93a534bf03e705L, 0xaaea14288ae2d90cL, 0x1be3cd51ef0f15e8L, 0xe8b47c84d5a4aac1L, 0x297d27d55b766782L},
            {0xe5181466d8e60e26L, 0xcf31f3a2d582c4f3L, 0xd9cee87cb71f75b2L, 0x4750ca6050a2d726L, 0xd6e6dd8940256849L, 0xf3b3749fdab75b0L, 0xc55d8a0f85ba0ccfL},
            {0xfb528a8dd1e48ad7L, 0x98c4fd149c8a63ddL, 0x4abd8fc3377ae1fL, 0xd7a9304abbb47cc5L, 0x7f2b9a27aa57f99L, 0x353ab332d4ef9f18L, 0x47d56b8d6c8cf578L},
            {0xda6d2b7ea9d5f9b6L, 0x57b11153ee3b4cc8L, 0x7d3bd1256037142fL, 0x90b16ff331b719b5L, 0xfc294e7ad39e01e6L, 0xd2145386bab41623L, 0x7045a63d44d76011L},
            {0x61d95225bc2293eL, 0xf6c52cb6be9889a8L, 0x91a0667a7ed6a113L, 0x441133d221486a3dL, 0xfb9c5a40e19515bL, 0x6c967b6c69367c2dL, 0x145bd9ef258c4099L},
            {0x81247c01ab6a9cc1L, 0xfbccea953e810636L, 0xae18965000c31be0L, 0x15bb46383daec2a5L, 0x716294063b4ba089L, 0xf3bd691ce02c3014L, 0x14ccaad685a20764L},
            {0xc17f3ebd3257cb8bL, 0xe9e68c939c118c8dL, 0x72a5572be35bfc1bL, 0xf6916c341cb31f2aL, 0x591da1353ee5f31cL, 0xf1313c98a836b407L, 0xe0b8473eada48cd1L},
            {0x9802438969c3043bL, 0x6cd07575c948dd82L, 0x83e26b6830ea8640L, 0xd52f1fa190576961L, 0x11d182e4f0d419ccL, 0x5d9ccf1b56617424L, 0xc8a16debb585e452L},
            {0x3dd8ed248a03d754L, 0xd8c1fcf001cb62e0L, 0x87a822141ed64927L, 0x4bfaf6fd26271f47L, 0xaefeae8222ad3c77L, 0xcfb7b24351a60585L, 0x8678904e9e890b8fL},
            {0xc5bf48d7d3e9a5a3L, 0x8f0249b5c5996341L, 0xc6d2c8a606f45125L, 0xfd1779db740e2c48L, 0x1950ef50fefab3f8L, 0xe4536426a6196809L, 0x699556c502a01a6aL},
            {0xbc4a21d00cf52288L, 0x28df3eb5a533fa87L, 0x6081bbc2a18dd0dL, 0x8eed355d219e58b9L, 0x2d7b9f1a3d645165L, 0x5758d1aa8d85f7b2L, 0x9c90c65920041dffL},
            {0x172c8674913ff413L, 0x1815a22400e832bfL, 0x7e011f9467a06650L, 0x161be43353a31dd0L, 0x79a8afddb0642ac3L, 0xdf43af54e3e16709L, 0x6e12553a75b43f07L},
            {0x17a361dbdaaa7294L, 0xc67d368223a3b83cL, 0xf49cf8d51ab583d2L, 0x666eb21e2eaa596L, 0x778f3e1b6650d56L, 0x3f6be451a668fe2dL, 0x5452892b0b101388L},
            {0x5cc268bac4bd55fL, 0x232717a35d5b2f1L, 0x38da1393365c961dL, 0x2d187f89c16f7b62L, 0x4eb504204fa1be8L, 0x222bd53d2efe5faL, 0xa4dcd6d721ddb187L},
            {0xdb04969cc06547f1L, 0xfcacc8a75332f120L, 0x967ccec4ed0c977eL, 0xac5d1087e454b6cdL, 0xc1f8b2e284d28f6cL, 0xcc3994f4a9312cfaL, 0x8d61606dbc4e060dL},
            {0x25bd8d3ca1b375b2L, 0x4ad34c2c865816f9L, 0x9be30ad32f8f28aaL, 0x7755ea02dbccad6aL, 0xcb8aaf8886247a4aL, 0x8f6966ce7ea1b6e6L, 0x3f2863090fa45a70L},
            {0x166c11fbcbc89fd8L, 0xcce1af56c48a48aaL, 0x78908959b8ede084L, 0x19032925ba2c951aL, 0xa53ed6e81b67943aL, 0xedc871a9e8ef4bdfL, 0xae66cf46a8371abaL},
            {0x3565bcc4ca4ce807L, 0xec35bfbe575819d5L, 0x6a1f690d886e0270L, 0x1ab8c584625f6a04L, 0xccfcdafb81b572c4L, 0x53b04ba39fef5af9L, 0x64ce81828eefeed4L},
            {0xb7897fd2f274307dL, 0x6d43a9e5dd95616dL, 0x31a2218e64d8fce0L, 0x664e581fc1cf769bL, 0x415110942fc97022L, 0x7a5d38fee0bfa763L, 0xdc87ddb4d7495b6cL},
            {0xaba98113ab0e4a16L, 0x287f883aede0274dL, 0x3ecd2a607193ba3bL, 0xe131f6cc9e885c28L, 0xb399f98d827e4958L, 0x6eb90c8ed6c9090cL, 0xec89b378612a2b86L},
            {0x17f7796e0d4b636cL, 0xddba5551d716137bL, 0x65f9735375df1adaL, 0xa39e946d02e14ec2L, 0x1c88cc1d3822a193L, 0x663f8074a5172bb4L, 0x8ad2934942e4cb9cL},
            {0x33c0128e62122440L, 0xb23a588c8c37ec2bL, 0xf2608199ca14c26aL, 0xacab0139dc4f36dfL, 0x9502b1605ca1345aL, 0x32174ef1e06a5e9cL, 0xd824b7869258192bL},
            {0x988bc5d290b97aefL, 0x6754bb647eb47666L, 0x44b5cf8b5b8106a8L, 0xa1c5ba961937f723L, 0x32d6bc7214dfcb9bL, 0x6863397e0f4c6758L, 0xe644bcb87e3eef70L},
            {0x23c8c25c2ab72381L, 0xd6bc672da4175fbaL, 0x6aef5e6eb4a4eb10L, 0x3df880c945e68aedL, 0x5e08a75e956d456fL, 0xf984f088d1a322d7L, 0x7d44a1b597b7a05eL},
            {0x450fe4acc4ad3749L, 0x3111b29565e4f852L, 0xdb570fc2abaf13a9L, 0x35107d593ba38b22L, 0xfd8212a125073d88L, 0x72805d6e015bfacfL, 0x6b22ae1a29c4b853L},
            {0x48e1eff032d90c50L, 0xdee0fe333d962b62L, 0xc845776990c96775L, 0x8ea71758346b71c9L, 0xd84258cab79431fdL, 0xaf566b4975cce10aL, 0x5c5c7e70a91221d2L},
            {0xc048604ba8b6c753L, 0x21ea6d24b417fdb6L, 0x4e40a127ad2d6834L, 0x5234231bf173c51L, 0x62319525583eaf29L, 0x87632efa9144cc04L, 0x1749de70c8189067L},
            {0x67ff1cbe469ebf84L, 0x3a828ac9e5040eb0L, 0x85bf1ad6b363a14bL, 0x2fc6c0783390d035L, 0xef78307f5be5524eL, 0xa46925b7a1a77905L, 0xfea37470f9a51514L},
            {0xb45c7536bd7a5416L, 0xe2d17c16c4300d3cL, 0xb70b641138765ff5L, 0xa5a859ab7d0ddcfcL, 0x8730164a0b671151L, 0xaf93810c10348dd0L, 0x7256010c74f5d573L},
            {0x215c2eaacdb48f6fL, 0x33b09acf1bfa2880L, 0x78c4e94ba9f28bfL, 0x981b7219224443d1L, 0x1f476fc4344d7bbaL, 0xabad36e07283d3a5L, 0x831bf61190eaaeadL},
            {0x241baf16d80e0fe8L, 0xb6b3c5b53a3ce1dL, 0x6ae6b36209eecd70L, 0xa560b6a4aa3743a4L, 0xb3e04f202b7a99bL, 0x3b3b1573f4c97d9fL, 0xccad8715a65af186L},
            {0xd10a9743b5b1c4d1L, 0xf16e0e147ff9ccd6L, 0xfbd20a91b6085ed3L, 0x43d309eb00b771d5L, 0xa6d1f26105c0f61bL, 0xd37ad62406e5c37eL, 0x75d9b28c717c8cf7L},
            {0x919ef9e209f2edd1L, 0x684c33fb726a720aL, 0x540353f94e8033L, 0x26da1a143e7d4ec4L, 0x55095eae445aacf4L, 0x31efad866d075938L, 0xf9b580cff4445f94L},
            {0xb5f9519b6c9280bL, 0x7823a2fe2e103803L, 0xd379a205a3bd4660L, 0x466ec55ee4b4302aL, 0x714f1b9985deeaf0L, 0x728595f26e633cf7L, 0x25ecd0738e1bee2bL},
            {0x77a75e89679e6757L, 0x25d31fee616b5dd0L, 0xd81f2dfd08890060L, 0x7598df8911dd40a4L, 0x3b6dda517509b41bL, 0x7dae29d248dfffaeL, 0x6697c427733135fL},
            {0x9d709e1b086aabe2L, 0x4d6d6a6c543e3fecL, 0xdf73b01acd416e84L, 0xd54f613658e35418L, 0xfcc88fd0567afe77L, 0xd18f2380980db355L, 0xec3896137dfbfa8bL},
            {0x91c89971b3c20a8aL, 0x87b82b1d55780b5L, 0xbc47bb80dfdaefcdL, 0x87e11c0f44454863L, 0x2df1aedb5871cc4bL, 0xba72fd91536382c8L, 0x52cebef9e6ea865dL},
            {0x16468c55a1b3f2b4L, 0x40b1e8d6c63c9ff4L, 0x143adc6fee592576L, 0x4caf4deeda66a6eeL, 0x264720f6f35f7840L, 0x71c3aef9e59e4452L, 0x97886ca1cb073c55L},
            {0x8015f298161f861eL, 0x3b4a12bf2e24a16L, 0x37b223562c48b473L, 0xd82489179f16d4e8L, 0xa3c59f65e2913cc5L, 0x36cbaecdc3532b3bL, 0xf1b454616cfeca41L},
            {0x71e244d7e2843a41L, 0x2132bd3f394ac99dL, 0x1791b7a5b93ad1f9L, 0xfd7feb3d2956875eL, 0xd7192a886b8b01b6L, 0x16e71dba55f5b85aL, 0x93dabd3ff22ff144L},
            {0x5d3cb0d2c7ccf11fL, 0x1215f183d5a24092L, 0xea833d94dca4809aL, 0xb5b472960ece11ecL, 0x13935c99b9abbf53L, 0x3e80d95687f0432cL, 0x3516ab536053be5L},
            {0xd6cffe6c223aba65L, 0xb19224aad3a69ef1L, 0x67268f8829a9f99dL, 0x62e33ba258712d51L, 0xfa085c15d779c0eL, 0x2c15d9142308c5adL, 0xfeb517011f27be9eL},
            {0x8a17c5054e85e2beL, 0x15e35d5a33726681L, 0x9b345fa359c4e8e3L, 0xe4041579de57c879L, 0xbbf513cb7bab5553L, 0x66ad0373099d5fa0L, 0x44bb6b21b87f3407L},
            {0x77d112a0b7084c6aL, 0x2f869c2d79d95e45L, 0xa3c8f877e8ebc840L, 0x16fde90d02a1343bL, 0xad14e0ed6e165185L, 0x8df6e0b2f24085ddL, 0xcaa8a47292d50263L},
            {0x708f2a6e8bd57583L, 0x688e0faea5f15272L, 0xd28955c99ed63d38L, 0x8459801016414808L, 0x6fbf75735353c2d1L, 0x6e69aaf2d93ed647L, 0x85bb5b90167cce5eL},
            {0x50bc8f76b62c8de9L, 0x88b4d8ebe13cbd79L, 0xda08ee1bf528e82eL, 0xaad20d70e231582bL, 0xeab92d70d9a22e54L, 0xcc5ab266375580c0L, 0x85091463e3630dceL},
            {0x8b15a656b553641aL, 0x611c74d4137bf21bL, 0xa051cbbf796013c1L, 0x38a42e0db55a4275L, 0x585971da56bb56d6L, 0xcd957009adc1482eL, 0xd6a96021e427567dL},
            {0x6ba74ccf722a52beL, 0x75e2d5362c0050b1L, 0x32e95f14d29a1c01L, 0x3526d9b950a1d910L, 0xa58ba01135bca7c0L, 0xcbad32e86d60a87cL, 0xadde1962aad3d730L},
            {0xfb317bb7533454d0L, 0x15f9898fd61f0209L, 0x593d179631ddf22cL, 0x7c909e8cd5261727L, 0xc5acb3d5fbdc832eL, 0x54eff5c782ad3cddL, 0x9d54397f3caf5bfaL},
            {0x8eec643f62c90feaL, 0xdf17fc55b1d4a915L, 0x62fa77ed321e937fL, 0x479f936b6d496dcaL, 0xdc2dc93d63739d4aL, 0x27e4151c3870498cL, 0x3a3a22ba512d13baL},
            {0x81ce6becdf10dff2L, 0x4182c78d3d609461L, 0xdb5cb16e44cb1e37L, 0x464f1adf4c68577L, 0xacf3961e1c9d897fL, 0x985b01ab89b41fe1L, 0x6972d6237390aac0L},
            {0x549c669fb0049f69L, 0xa01f16549b0a628eL, 0x675a9a86499cd4e4L, 0x8af42343888843cL, 0x191433ffcbab7800L, 0x7eb45fc94f88a71L, 0x31bc5418ffb88fa8L},
            {0x2b6a3433940bbf2dL, 0xdda5e942a8098f8bL, 0x812bcb2a17f1f652L, 0xdc46069eec17bfdfL, 0xcacb63fe65d9e3eL, 0x362fb57287d530c6L, 0x5854a4fbe1762d9L},
            {0xd80b7a3c691401b7L, 0xe205b8266ea761cbL, 0x8e44beb4b7cde31bL, 0x69437142dae5a255L, 0xf2980cc4816965acL, 0xdbbe76ba1d9adfcfL, 0x49c18025c0a8b0b5L},
            {0xab3bf6b494f66ef3L, 0x530b0467dcaf3c4bL, 0x383cc50df33afc6fL, 0x5e351e20f30377bfL, 0x91b3805daf12972cL, 0x94417fa6452a265eL, 0xbfa301a26765a7cL},
            {0x83f7b824a3911d44L, 0x921947a8a2668a44L, 0x13c001ebba408aaaL, 0xfd39b7642cecf78fL, 0x104f1af4e9201df5L, 0xab1a3cc7eaeab609L, 0xcee3363f210a3d8bL},
            {0x3fb8d482d0d9d03fL, 0xd911bf94d3017ee1L, 0x96ebbf1ceac7b4cbL, 0x18865ff87619fd8fL, 0xdec5293e665663d8L, 0xea07c345872d3201L, 0x6fce64da038a17abL},
            {0xad346a1f100b3944L, 0x3934eb0f8d35a797L, 0x77664abec282db4cL, 0xb5f630ac75a8ce03L, 0x7cf71ae74fa8566aL, 0xe068f2b4618df5dL, 0x369df952ad3fd0b8L},
            {0xdb210eb547a3dbc5L, 0xe1013615221cb0d7L, 0x4ca87abbb73194d8L, 0x1b0118c5c60a99c7L, 0x6ae919ef932301b8L, 0xcde25defa089c2fcL, 0xc2a3776e3a7716c4L},
            {0xe55fab4f920abdc0L, 0x7fa81600f789f5a6L, 0x6f67cf7344c18fceL, 0x2a5e555fd35627dbL, 0x55d5da439c42f3b8L, 0xa758e451732a1c6fL, 0x18caa6b46664b484L},
            {0x3b530fff7e848c5eL, 0x152e3fec5a21ed68L, 0x4340e5798860241aL, 0x1944ec723253392bL, 0x7ea6aa6a2f278ea5L, 0x5ff786af8113b3d5L, 0x194832eb9b0b8d0fL},
            {0xbde3379279d1cae1L, 0xf3596e48364bdaacL, 0x9f070e7509abc6bfL, 0x81d90ddff0d00fdbL, 0x2c8c7ce1173b5c77L, 0x18c6b6c8d3f91dfbL, 0x415d5cbbf7d9f717L},
            {0x4008062bc7755b37L, 0x8873f772dc7d1ea2L, 0x4502cd3133c94d7dL, 0x381068e0f65f708bL, 0xb4f3762e451b12a6L, 0x6d61ed2f6d4e741L, 0x8b3b9df537b91a2cL},
            {0x76a66ce0ee8094d1L, 0x99dc87d517229612L, 0x83b12c2aff5dd46eL, 0x6e8e8ff107799274L, 0x24a2ef180891b531L, 0xc0eaf33a074bcb9dL, 0x1fa399a82974e17eL},
            {0x2bc3dfb3b1756918L, 0x3e0269476ab76f14L, 0x52c60b61184e08deL, 0xf5f8b21ec30bd3a0L, 0x80a442fd5c6482a8L, 0x4fde11e5ccde5169L, 0x55671451f661a885L},
            {0xd060dc1e8ca204eeL, 0xce494f4b2198e36fL, 0x1f120ffb0524d537L, 0xcaac64f5865d87f3L, 0x771b9fdbd3aa4bd2L, 0x88446393c3606c2dL, 0xbc3d3dcd5b7d6d7fL},
            {0xc8ec4fc839254a74L, 0x4d8b8b116ea60b09L, 0xd6a77d7a8c6d11f4L, 0x401a0581221957e2L, 0xfc04e99ae3a283ceL, 0xfe895303ab2d1e3eL, 0x35ab7c498403975bL},
            {0x7cdf98a07b1315b0L, 0x5b7132d0a9ee6608L, 0xd2480e4e97602adL, 0x330b7e93663affbdL, 0x3c59913fcf0d603fL, 0xe207e6572672fd0aL, 0x8a5dc17019c8a667L},
            {0x78284cb5c0143ed8L, 0xac6af8e6f7820e82L, 0x71d171a63a6187b5L, 0x77fbb70409d316e2L, 0xc864432c5208e583L, 0xd3f593922668c184L, 0x23307562648bdb54L},
            {0x5c2c485bdc8e3317L, 0x7bfe5915c5e0fa2dL, 0x6b433526b05fc4d8L, 0x20085827a39ff749L, 0x42e6c504df174606L, 0x839da16331fea7acL, 0x7fd768552b10ffc6L},
            {0x6e38acb798627f75L, 0x55ac9c4d9d32fed7L, 0x766ef46cf807f655L, 0xd0884af223fd056bL, 0xbb33aafc7b80b3e4L, 0x36b722fea81a4c88L, 0x6e72e3022c0ed97L},
            {0xc5fb48f0939b4878L, 0x7b773f9bcd0ec27aL, 0x6d36a844bb3f3360L, 0x984cf3f611546e28L, 0xd7d9c9c4e7efb5d7L, 0xb3152c389532b329L, 0x1c168b512ec5f659L},
            {0x292da6390260110L, 0x7608d31cc4c96e48L, 0xf843ecb8366f0809L, 0x24940a3adac420b8L, 0x5adf73051c52bce0L, 0x1aa5030247ed3d32L, 0xe1ae74ab6804c08bL},
            {0x1e0ee26b7044741bL, 0x1b7f67a75b435af5L, 0xb24891afcb0faa49L, 0x2a55645640911e27L, 0x4fac2eefbd36e26fL, 0x79ad798fb4c5835cL, 0x359aa2faec050131L},
            {0x69b8f7e762db77ecL, 0xd845fd95e4f669e0L, 0xb1e8e3f0f5c9037eL, 0x10a7228693eb083eL, 0x1054fb19cbacf01cL, 0xa8f389d24587ebd8L, 0xafcb783a39926dbaL},
            {0x9b321366d6585031L, 0x8ffcf9094b4ed2e2L, 0x3b7321189816fdccL, 0x39756960441fbe2fL, 0xfb68e5fedbe3d874L, 0x3ff380fbdd27b8eL, 0xf48832fdda648998L},
            {0x9375c89169bf70cfL, 0x45d697d09989365fL, 0x3ab599efd811ae97L, 0xcba4c10e65410ba0L, 0x3c250c8b2d72c1b6L, 0x177e82f415595117L, 0x8c8dcfb9e73d3f6L},
            {0xa8db1643cc52d94dL, 0x47d3bfec129f7eddL, 0x925b29c3dbfea463L, 0x951f2078aa4b8099L, 0xe68b7fefa1cfd190L, 0x41525a4990ba6d4aL, 0xc373552ef4b51712L},
            {0xcf7a9ea6a7a30deeL, 0x9573ae0f07cb7c2eL, 0x6793c6e1fad303ddL, 0xb57ec44bc7101b96L, 0x6cb710e77767a25aL, 0x2f446152d5e3a6d0L, 0xcd69172f94543ce3L},
            {0x42c2e9f84dc7f129L, 0x7b482774b391095cL, 0x216a0d505d49b80L, 0xed094f47671e359dL, 0xd9ebdb047d57611aL, 0x1c620e4d301037a3L, 0xdf6f401c172f68e8L},
            {0x394c2c1cca4e9271L, 0x2573fb79ecb7111fL, 0x5113e80f2555b54cL, 0x5d765af4e88f3277L, 0xd2abe1c63ad4d103L, 0x342a8ce0bc7af6e4L, 0x31bfda956f3e5058L},
            {0xd38df9e9740cb16cL, 0x79be3445c5491402L, 0xa15ead26a317837eL, 0xa6814d3dc578b9dfL, 0x3372111a3292b691L, 0xe97589c81d92b513L, 0x74edd943d1b9b5bfL},
            {0xec12466d1379cfdfL, 0x84aae38bd5b56932L, 0x1407e7cad8d977dfL, 0x63672de7951e1853L, 0x3ca0c763273b99dbL, 0x29e04fa994cccb98L, 0xb02587d792be5ee8L},
            {0x9050986d9ced6a2eL, 0x8bc353d8f72e4f9cL, 0xb16a21f3ae8ddaf4L, 0xa16cd2e8b445a3fdL, 0xf0d4f9fb613c38efL, 0xeee7755d444d8f2fL, 0xb530591eb67ae30dL},
            {0xc7362967930e8a48L, 0xa61695f6772f5336L, 0x96e9b973fe114561L, 0x5386ef0b438d0330L, 0xd39e03c686f8a2daL, 0x9555249bb9073d78L, 0x8c0b3623fdf0b156L},
            {0x47bd8137d464eab3L, 0x236db8fed274d4d7L, 0x499063daa6e4eae3L, 0xe381f24ee1d9a97dL, 0x7c5d95b2a3af2e08L, 0xca714acc461cdc93L, 0x1a8ee94bc847aa3eL},
            {0xcff30d9303db2dfeL, 0x1afb5899ab9c8653L, 0xb2d9cc739ab9f148L, 0x4cbef49086e62678L, 0xd77dfecc2819ef19L, 0xc327e4deaf4c7e72L, 0xb4d58c73a262a32dL},
            {0x8d086fc30b6694b2L, 0x90533a6a1124ec0bL, 0xf24a7ec2f48b6809L, 0xbecb065dc12d8b4eL, 0xebee135492a2018L, 0xd3f07e65bcd9e13aL, 0x85c933e85382e9f9L},
            {0xb7d681356bdd9e4fL, 0x9e8e19b5cdbfb229L, 0xe8f5fbafde7bea61L, 0xbc944c1b5ba2184dL, 0xab3d57e5e60e9714L, 0x5d8d27e7dd0a365aL, 0x4dd809e11740af1aL},
            {0x5bb01fcb2e6ad355L, 0x895c355e71191ef4L, 0x1f7a98978f1bf049L, 0xaaa144fbe3e6fda2L, 0x52a9291d1e212bc5L, 0x2b4c68291f26b570L, 0x45351ab332855267L},
            {0xcd2ff001a80d1b11L, 0xc0f8d9d7d08c74b3L, 0x5df56e499e9ca980L, 0xb8c18d66154ac51L, 0x5807350371ad7388L, 0x81f783f4f5ab2b8L, 0xfa4e659f90744de7L},
            {0x8bfbf611401100cdL, 0x599edd1f5154a546L, 0x56b61ed81d29796L, 0xb744f5056e74ca86L, 0x88aa27b96f3d84a5L, 0xb4b1ee0470ac3826L, 0xaeb46264f4e15d4fL},
            {0xec9ae0cf9290d012L, 0x14368811a4a1621dL, 0xde04cc7d2c562fcfL, 0x4323852cc57e4af3L, 0x1f5f638bbf9d2e5bL, 0x578fb6ac89a31d9L, 0x7792536d9ac4bf12L},
            {0x4ac2a5e9dc03176dL, 0x78df6aca1dd90b2bL, 0xe5fcbc1dfe65f7dbL, 0x766b71bff7d6f461L, 0xb004f2c910a6659eL, 0x4c0eb3848e1a7c8L, 0x3f90439d05c3563bL},
            {0x5fd51f635bc557a8L, 0x7e810b3d0b3db6dcL, 0x603d74dd65a3baf6L, 0xd178444a236c1f2dL, 0x5576deee27f3f103L, 0x943611bb5b1b0736L, 0xa0fde17cb5c2316dL},
            {0xec3521e8efdb1779L, 0x422da247696eedcL, 0xa372b6a2f501313cL, 0x7a265e37da616168L, 0x6a1f06c34bafa27L, 0xfbae175e7ed22a9cL, 0xb144e84f6f33c098L},
            {0xa9147f0fb2e38bb1L, 0xd2da27a1045272e7L, 0x7bfb951842162784L, 0x9fd4d9362494cbbcL, 0xe562bc615befb1b9L, 0x8096808d8646cfdeL, 0xc4084a587b9776ecL},
            {0xa080e609751f2e81L, 0x2df3057a20f24b2eL, 0xb9a0f8b45d79c318L, 0xc84bb7b3881ab070L, 0x36fe6c51023fbda0L, 0xd62838514bb87ea4L, 0x9eeb5e7934373d86L},
            {0x3bc578f69905fa2dL, 0xbb2c316b7f3eb8c2L, 0x18646edbbf14b0edL, 0x4473c8e2a3458ee0L, 0x258053945ab4a39aL, 0xf8d745ca41962817L, 0x7afb6d40df9b8f71L},
            {0x9e6a5e0641d1c0d9L, 0xba3905a52abd5805L, 0x73c0bcd5a7366e15L, 0xb5f52041a698da7L, 0x29864874b5f1936dL, 0x49b3a0c6d78f98daL, 0x93a1a8c7d90de296L},
            {0x83b0cdb3c934c679L, 0x88e72fd2e9c85618L, 0x25cbcd575a21c446L, 0x664ec3fad8521859L, 0x406f082beb9ca29aL, 0xb6b0fb3a7981c7c8L, 0x3ebd280b598a9721L},
            {0xf174161497c5fa97L, 0xdb3717129871fa98L, 0x5eb7a9d5a724daedL, 0x5414e385f5677a6dL, 0x41ef105f8a682a28L, 0x94ae9f66d82d71f0L, 0x6b698643f5924cc6L},
            {0xd7262cb2f2755e70L, 0xf87e6123e1e56dc7L, 0x76a474a551662078L, 0xd4bd358fed3e6aa5L, 0x8a1ba396356197d9L, 0x5bcf3e13a8184f6dL, 0x5f17183906e40929L},
            {0x1444ce264e8784b7L, 0x72dd36167600c3c4L, 0x7e52afec3d9d166eL, 0xdb0c32f76f5b7fc1L, 0x5e41b711f0abd1a0L, 0x41f387462b60ec17L, 0x20cd7feaf6b0d5eaL},
            {0x532e6b5c95a2e229L, 0x2e74b7cb427c4e8aL, 0x90febf2e15deaaa5L, 0x32d3a29cf49e2dc9L, 0x3079c0b0c2269bd0L, 0x906379f72fdcdff1L, 0xea076563ae6ed4ceL},
            {0x183d112159f539ebL, 0xe0a272f5325ccfceL, 0x82051bab809a3bf3L, 0xa54eaa5d7f3a7227L, 0x9d26922965d54727L, 0x858c68ea2e46ec2eL, 0xe157ea542fd04d72L},
            {0x8f18272400b3ace9L, 0xf936fba4e6c7f6f5L, 0xa395a3524727f255L, 0xea37f61c0c2f6d53L, 0x9b0c2174f14a01f5L, 0x631f4fbb52f7b4e1L, 0x882e8ea542c5a526L},
            {0x43761e6a5f6f2fd6L, 0x49ff8647ef7de2cdL, 0x9db86e928d9c50d8L, 0xa32c192f6e3c3f66L, 0x8f10077b8a902d00L, 0x85dcc88c159d5666L, 0x2277bf43094b85beL},
            {0x44f615fcd096fbfeL, 0xf91b1ef287c2b469L, 0xb0153a7a31076599L, 0xa82a7bb790678fc9L, 0xd197682c421e4373L, 0x92183ad53c7546c2L, 0x80e483f6e47eb386L},
            {0x27613f9db818cf78L, 0xffe86d35e2b6af5cL, 0xbf8519db808bc3faL, 0x33322363b5f45216L, 0x7e83f1fe4189e843L, 0xe9fbbdca5076a660L, 0xa025a44ad19f89a6L},
            {0x3f6984c7afaebd0bL, 0xce1a85c17d6eaa95L, 0x5b4fdfaf084a67cbL, 0x33c6690937582317L, 0xfe6d61a77985d7bbL, 0xd33afeed3012068cL, 0x9769fde4cc576926L},
            {0x8fc511284f47c772L, 0xb4c0724c9dd30a7aL, 0xd04d61891a700659L, 0xf3d41b3d4717eb83L, 0x2670d457dde68842L, 0x8af262ab62293cL, 0xc4bbf2c513a0d201L},
            {0x15ae5f12f88592e2L, 0xaeec43559dd10188L, 0x5bd91d548ffefd6eL, 0xdd70e407984cfa80L, 0x66996d6066db6e1aL, 0xccdf7065727283f8L, 0x11e3ac9d9fa63e16L},
            {0x905f995bddf92cb7L, 0xf7847475e7e1e12L, 0x4dc5bb06cd995ceeL, 0x9432536dd9f65229L, 0x192dc54522da3e3dL, 0x5eecf6aff92e88d4L, 0x2d450eaa62f5f8cfL},
            {0xa23ac6bef8905fecL, 0xf675b3d658210f05L, 0xb363b9add66e1684L, 0xb9aa5bead3352801L, 0x8a6d9e02a19a4229L, 0x481ba75a9a2f85cdL, 0x17f5920e749edd3cL},
            {0x403b94a75160a06bL, 0xc6eb4d9351f81e3aL, 0xaa0739cef99075f8L, 0x8d8e7c63385df78eL, 0x16d55add72a5e25eL, 0x5fc4a1f7b4b489f1L, 0xb6bb01a378d6ff12L},
            {0x14d1ee05672fc19bL, 0x703c1a85efd79788L, 0xd065eb50cfb5c451L, 0xce218d5b44f7825aL, 0x2ae0c64765800d3aL, 0x176c27079a2958b8L, 0x9c886360f29d425dL},
            {0xf59376c617951a2aL, 0x32082c76de4710ffL, 0x5d291a873d8d9f9aL, 0x1682f54521c291e2L, 0x17ca7ab8d97ba0d9L, 0xd8d6d62c4e314ae1L, 0x269df5967d606f43L},
            {0x63982fdc37a9dc5L, 0x8f7025774c4faea9L, 0xe7d5ccc2c570c989L, 0x8868a216a8f9946L, 0x136b5ce2ede03238L, 0xc19f0f6f73f2e30eL, 0x9219f59f493aa238L},
            {0xeb480334ed838b48L, 0x87f74500f9dd7becL, 0x39dfa7182359c93dL, 0x6b7d8d04ae2db2b2L, 0x703b85004b504bd6L, 0x6b08bbdf158d17ebL, 0xaa84994c75c80c1bL},
            {0xd0b9004efa0a1164L, 0x9f68c1718dbde89fL, 0xb5d9a7ab23808f7L, 0x9463491a8439eb54L, 0x2bd049dd3b9307ddL, 0x6b8387ee8561f97eL, 0x2db67c3e7c0097f2L},
            {0xb31f2b6cc2a15506L, 0x28ad14cab53f717aL, 0x341f7d6f1212356bL, 0xeee5254624f99323L, 0x50f935ad0b11b484L, 0x56944e9d03e9b415L, 0x886a205391ac0526L},
            {0x4f9da8a709bec12fL, 0x69347f992ba0b479L, 0xd82b32d7831a57aaL, 0x5e1aaa3d321c7c55L, 0xd808df942eb2b92bL, 0x6f303048a4f26df9L, 0xcaea0391727bdafdL},
            {0x5504000602e6f8cfL, 0x17b80b6681b62db0L, 0xd83dda14a4423630L, 0x7926e11179d2b882L, 0xe94fcaa22d091d09L, 0xea51d34158cd60a8L, 0xb6d56df73b12baccL},
            {0x2d022d82f513a883L, 0xda474eedd3ed528L, 0xb339f8f80f556971L, 0xbba2b0a0f1b18055L, 0x1e72c71e6c54cb2eL, 0xe43bfe7158c31dc3L, 0x416e04cd551cf777L},
            {0xa87268205997eddbL, 0xe58cd6fb05cb965L, 0xc69eac6e1256a4d9L, 0xe1dbe62ee1aec1a2L, 0xdb97b9879cfc5bb0L, 0xc1a96637db293163L, 0xcdacf3525efecb1eL},
            {0xfde5f0a803c3affcL, 0x9db3e206a3deda39L, 0x6a678f40a8058ab0L, 0xdd00bf5b13f1b349L, 0x9441065c7bddf30aL, 0xe6005534fbe17f8eL, 0x32346eb79b5ac530L},
            {0xfa46e0e215c1aabdL, 0x6b795034438df194L, 0x14571d59c36d29eaL, 0xfa36d64aa42643b7L, 0xd2c8ac46f42c7edL, 0xf3d36aa6e975be3fL, 0xe7f022266f7e0f3eL},
            {0x7e1f98b2c16f8b2bL, 0xc492c4d36704af4cL, 0xd23b89c36294385aL, 0x244cf0d778a05ec5L, 0x8380a5edd5a19005L, 0x29a9617055f5a9d9L, 0x7005e59f480fa82L},
            {0x65a58d22d8665e60L, 0x41d64e7bbfe54f80L, 0x14ce593a63d5a590L, 0x97f748c8a8d4f327L, 0x3445886827d5e08cL, 0x634b32b334de0fa5L, 0x5ff94df6197c1bc1L},
            {0xb781b9a55e7d6ab9L, 0xed16ad5400bc87eL, 0xb2ae79ab4b601becL, 0x1cfc0839227804c3L, 0xaf51c631d33d0e65L, 0xdd36799ebc883ef5L, 0x548b458a566154a8L},
            {0xa88c857b1aeb0835L, 0x1b2cdaa3a001518cL, 0xb03b42ed85390beaL, 0x47a8abfd4b36b416L, 0xc28d691c93c456aeL, 0x7b0890fd2c0a3e45L, 0x757b11be8b925e40L},
            {0x2a25994979124643L, 0x3a3e0c52948d116aL, 0x8667e3420bb8d50L, 0xee4ae633d60287f3L, 0xbaf02bf6e52c4b9eL, 0x31a0d0505fb5b745L, 0xbed2029003e3afe5L},
            {0x17236ed61e669c6fL, 0xb049f89783cd2c3fL, 0x28fe56765c1b7edL, 0xe8e11d96868fd11eL, 0xd7323d276887576aL, 0xb6a62449ac769727L, 0xdfcf0d8773df717aL},
            {0x304f56359ac375a8L, 0x9b303c940f78c5b5L, 0xd52df3445a1a027cL, 0xfa406ca71ad32716L, 0x103ee9820a95f912L, 0x4d4cc990a9c5f759L, 0x7a70381414b621cfL},
            {0x2e236ded6ce34194L, 0x55cc633269f53f75L, 0xb0b18d56850de1a5L, 0xeb648960aa937c06L, 0xac4d4ac05a66c83eL, 0xb56ebc918ee8e99cL, 0xacf505efb66a8d78L},
            {0x837ecb10d69f9bb9L, 0xa51533872cb4f5b1L, 0x620aaae18ac52da4L, 0x94728cfb8d580043L, 0x98dc3ea6890f66e4L, 0x3373ffcf756e1a31L, 0x5742e8e69476ef0eL},
            {0xc94bc80993d726f2L, 0x13d4514a68dca054L, 0x4b2b9d78cda610a0L, 0xc6a9ed722f850d2L, 0xff91629f0e2bda83L, 0x637d74f5a36dc3d3L, 0x788b506986bfbabaL},
            {0x463b54729349357aL, 0x381b0b67f1b0e6cfL, 0xab16e09e606fd74fL, 0x783cc746f3e0ad94L, 0x4cb37580460063a4L, 0xbc693f29e8c8e068L, 0xd8d02963f0137472L},
            {0x52e298a69bc61248L, 0xf0c389219e5b280dL, 0x45d0e8cbf8999a16L, 0xabf9fdcb39df4494L, 0xe5f29602c7c58f65L, 0x41b808023b066957L, 0xe2158bf17bd49509L},
            {0xf31bde28294be223L, 0x6fcf217a39638e31L, 0xb57c20d0886767ccL, 0x9ff176e872b59750L, 0xb79b82d5e03a36a7L, 0xa05cd10a650a945cL, 0xbf270f3283985d7dL},
            {0xd1d98f3bbaf26f1eL, 0x6e1bb3126ee93a70L, 0xfdb28b9faa7455cbL, 0x77bccd28f0502dc2L, 0x4704cff4be86c032L, 0x2d7083cbb8dabf9L, 0x40f403385f91540fL},
            {0x77969267e761a5e2L, 0xef907093519fd3ffL, 0x6a50d507180b9dd8L, 0xf7671fba16e03cddL, 0x1d6da71263c3c627L, 0x1f3fd89b2729c1a7L, 0x823b6d2b1df14ad9L},
            {0x763f1101a3d8e5d6L, 0x6de2bec8b2a9f0d9L, 0xbdc2cec3b0c72126L, 0xc684b15ed14c4849L, 0x25370610ff0d1b07L, 0x7b933b223c95a22fL, 0x3a44eb381671ea69L},
            {0xb6ffcab942c26180L, 0x1261443adbb2a65bL, 0x8bc905dfc7b85595L, 0xada824a1b0baddeeL, 0xdadce601b47afe19L, 0x46c3679d5436da89L, 0xc6251e4bebbd57daL},
            {0x65a85965268277a5L, 0xf29a5b2c9a51c439L, 0xaabf0324ece8814L, 0x6521fe578234c086L, 0x88294a1221e85ad5L, 0x685d8b1b46910a3aL, 0x600a02f24903a0a2L},
            {0x6579248c4cabcf91L, 0x21b4fdd6da56df76L, 0x727c8a9161e8cc9eL, 0xf5ef59c60d0513ffL, 0xa2c01a296e191385L, 0x8a7be6557ac395e3L, 0xfa772f9eee991c2eL},
            {0xfcea6deb6fbc95deL, 0xfa083e475037dd9bL, 0xa402e8b120895dcfL, 0x128da31d726a4316L, 0x6b12ccb52a7d436bL, 0x86a73e06caadae8cL, 0xf0106fd320ce8874L},
            {0xa5afb4dac88f15f0L, 0xfa2d11898daaa5e1L, 0xc118493d086eb030L, 0x932d16df5d691fb1L, 0x5dde659f15fe9b64L, 0x6a55cbf2cd4cf502L, 0x2c69b73b509c23e4L},
            {0x35f437b7acbfd454L, 0x23e56c12e72a5eeeL, 0xb29d3d3f9fea4832L, 0xbd6947ba8f8af57L, 0xe2c925a1262dbc65L, 0xcec8c16603d2616eL, 0x4acfe590ee1f5942L},
            {0x8f45f63a2f2d77d5L, 0x2f251880d8b83227L, 0x7b457c2c096a4fefL, 0x60386ba5baeb0a28L, 0x4c8db172ac835d05L, 0xc9f3245a03b60ffaL, 0x1e40258aeb689328L},
            {0x62258e6fe64ea749L, 0xe61da1411606c50dL, 0x1e383042228a6524L, 0xed27416d233bcc8cL, 0xc36902f1d5236598L, 0x72cc7a789bdc0df8L, 0x16e1c593759df8baL},
            {0xfc109f4192ba2587L, 0xda1eb26735719b82L, 0xda0616341a86759bL, 0xbae2015903c43013L, 0xded1d087f02e51dfL, 0x4865349424eee47dL, 0xcfab56ebb783561bL},
            {0x5364968136715e44L, 0x4d6df79cca67503cL, 0x1b04a476a5af6319L, 0x38769eab2c8333caL, 0x3babeb25b54f33e4L, 0x70e9850467e4d7c4L, 0x4b23b70a8351c781L},
            {0xdd84538848e07acbL, 0x7e8b2c2ea4b31867L, 0x4a1b7795ab30febcL, 0x6264b317e74f1956L, 0xe7baa170c2d01227L, 0x1d6e44e71bfb4f79L, 0x96c3307376e7264aL},
            {0x397d78f9c2fb2a8aL, 0xa35668e29dd4493eL, 0x4b5d814365f300c3L, 0xb47c59e2352ab2deL, 0xfee916bf69dca4f2L, 0x31720599f9b05204L, 0xcefd056a9a8cdb6dL},
            {0xa3a22aed573f4128L, 0x6f441959a6a745a4L, 0x61b303cec0f02d33L, 0x7b245212b57b6416L, 0x483eb1c2247030caL, 0x9e73ba2d0cf42603L, 0x7caaf3b868c1fa6dL},
            {0x94bcd5be64b0caf0L, 0x3eee5a2a3b25c93dL, 0x465dd18f325b992dL, 0xfd45ffd2387a5d25L, 0x35a79122b1ac38d2L, 0x45c691735698b5cL, 0x87dfa15417044ee0L},
            {0x81d9fe1f35fe8dcL, 0x2c5e1e21c57e9a0fL, 0x9cee67a7a5c2455L, 0xf92e222a71b413feL, 0xec8f28b36b4b2d43L, 0x73416a88b65b673bL, 0xcf89cf0d3d4148aeL},
            {0xaa21f88e4310c4aaL, 0xfd678eaf1a2ef449L, 0x9eb788a336487a2fL, 0x6c43d6ff49cff14bL, 0x88c3f8f93676751cL, 0x2322a939c0b8bef3L, 0x4adf71d19b5e462bL},
            {0x88e65c8bd8fd0dc3L, 0x67506bfcc1350acL, 0x8e35bd6404cd4a19L, 0x10f30b6a6ec67afeL, 0xaeb8adee4a38102cL, 0xc7c31205261346baL, 0xed3a7eb75f0bd93cL},
            {0xee7c287c7a74eaf6L, 0xa4716c9b01821875L, 0x15b4a29d0e8621fdL, 0xde6e60dce8b3c61L, 0x8c97da4b7dca945cL, 0x315027b9093d6a5L, 0xc60067ac2204a60L},
            {0x59492bfd26df7a46L, 0x88ef9253d909b4e7L, 0x2dabd903fca564b6L, 0xd154dee359c629ddL, 0x61da6808b6b00605L, 0x1c862110cb1fa740L, 0x877bc417d7060346L},
            {0x79471e68a2e7b4c3L, 0xe2ecc811cd228f6cL, 0x1c0c9618d88b7cf9L, 0xc0eff8b2276af95fL, 0x3ec1a9c95964b183L, 0x9b480684e56c25ddL, 0x8b849da7c70dbd8fL},
            {0xf806f8b0f54bbbf4L, 0xc55ef4c3c37c3e8L, 0xc57fbf8500049609L, 0x48180ec114a019d9L, 0x5dde07c8bebb3f74L, 0xf3c14ac2fd15afc7L, 0x18ba219f867fa279L},
            {0xaf0a9fa8d197fc2aL, 0x83821f8be6e51915L, 0x297afa7cc277321aL, 0x5006b740dd5776eeL, 0xf139123a5edbaacaL, 0x40b1400056322a19L, 0xb8d5d524f5b2d708L},
            {0xa93491c935028bfdL, 0x2c36d238aac64cd6L, 0xfc5dcf1134041bb3L, 0xd813172f90f9ff47L, 0xc65afeef14c7926fL, 0xe0b91c18356984ddL, 0x5b8bcaba09e98375L},
            {0x35fb344f57414e7eL, 0xb4b10fbd03ba41bdL, 0xf084e04093c69f21L, 0x1f4862ff7005835bL, 0xe3281b9e914c7ab1L, 0xb551d8d1d75dfe2dL, 0xcc64942fb0af0ab5L},
            {0x650c588ae7997006L, 0x7b6516105738f40bL, 0xfd225f2b4d05e97bL, 0x76d74223d7c381f4L, 0x84445fec43ac4ba9L, 0xe457a2d9de2f0cbL, 0x4f80e43844d5a721L},
            {0x8e83c18ec4fac9b2L, 0xdd53e5013eea5434L, 0x15a4f84e1fbb081fL, 0x2494508d174f0a6aL, 0xf8f421c15e7812a6L, 0x15d14a45d852a3c1L, 0xa12e6ceaa7b91862L},
            {0x35422c6582e3fa2eL, 0xbabd15f9e3a1362dL, 0xaac1901ece0f6dbeL, 0xa1d206f25bcb2e04L, 0x630172f8ae1fbfa7L, 0x4df60c162fd13077L, 0x24fcca03cbfd0d36L},
            {0xfc0cb7f55d516f4eL, 0xa92f59467a97a76eL, 0xf058bdc360787c2eL, 0x2a428170995dc60eL, 0x90bf382200c6050dL, 0x522302fa40bdb9faL, 0x7d84077cabfda564L},
            {0xe6245e6273cd7da4L, 0xbb6b62af69cce44fL, 0xdef05217f2ba3b7eL, 0x40bd603028fabdfeL, 0x897f7885daba93L, 0xd343910442a51554L, 0xfe448c8b278337b5L},
            {0xbfb40261b25b0146L, 0xfd74edf7a1e0a088L, 0xd9de3386702c1efaL, 0xac61387d34b8eba3L, 0xf658eceff68e4f98L, 0x77eb7ea3faf6af6fL, 0xf2ae3f731cc6c3d1L},
            {0x298876b240a1f937L, 0x81c826252bc3fd08L, 0xf4ec81df6387b639L, 0xa87f848ac3739101L, 0x50a679588482b1c6L, 0x5ac0460a499bdd23L, 0x49cf4b738f9b6eb1L},
            {0xbf26833d8f21542eL, 0x62d95f72f9dc8b6aL, 0x978d2ce692fc2cd8L, 0xb2fc06ae50b5bdc4L, 0xb5ebdc6f7e689092L, 0x1324ae1b8c471019L, 0x235375036a20b675L},
            {0xff85120bd8fa3cd4L, 0xf768d9c178d8e1a3L, 0xa7a136668d023fecL, 0x227a859ccfb575f7L, 0x8fc9e6749312034bL, 0xfe1358976e8d1d7fL, 0x47e95007f7949411L},
            {0xa37277b9eb9b16fcL, 0x2b780c1cd1dad364L, 0xdd4a2f266cd6aa0fL, 0xcce760ed2d17daf7L, 0x2c5e908016d4ff3cL, 0xc68cab0c465584c1L, 0x67a3eafc44e4d677L},
            {0xb95c558eb132482fL, 0x86deccc1c79e3da4L, 0x8b43da48c6f5fd49L, 0xd35063f44069c518L, 0xbfa680a6399cae70L, 0x14af385219857c8bL, 0x1689871cfa0f2813L},
            {0xeb2a51b23ea2f82dL, 0x7d3802b6045fbd2bL, 0xa8670a39b06b41ceL, 0x851073e9d44dc921L, 0xb0aa6451924617c8L, 0x7b3ca65185907aadL, 0x4c1e01ba84283e44L},
            {0xc85dcc13ce7d29c0L, 0x3a520e93ccdeb592L, 0xea19bd7ee74ed003L, 0x3628daac258a4a48L, 0xafb35f7b660a441L, 0x5cb7c82d5974037aL, 0xf285ff762ef24753L},
            {0x8a8707d80cb54c7aL, 0xa4135e3dcd129a6cL, 0x411cbfa152de1998L, 0x674142656124b5c7L, 0x1280123d0a21a062L, 0x1afc0b67eb7c5bf7L, 0x6a666c0d97040f27L},
            {0x12c7ffecff1800baL, 0x1172a3993fff1057L, 0x44bd867e67fc8ae4L, 0x81b2a78e5319365cL, 0xb3aa90916d02335fL, 0xd36241c3a0745572L, 0x5c6e74ae8dd3d88bL},
            {0xcb16c5c1e342e34dL, 0x9087c1dc80a18b68L, 0xd48a476057f23dc8L, 0x41ff65495500e2e4L, 0xb8fab1ad95574a61L, 0xdbfd0326a7599b91L, 0xae5eb38b876d3fa9L},
            {0x27fddd06bd368c50L, 0xf7efaa7ef3bed090L, 0xb505f7b0690e3f70L, 0xed194c89f81522b9L, 0x272a0528540527e1L, 0x57e1e98c484b9f28L, 0xe0d5a808989c1b7L},
            {0x5e6c6ee85cec7703L, 0xdc33500e0dea1513L, 0xb0d9d93b584d752fL, 0x596dad7ffc69035L, 0x1a31664d3d509c10L, 0x8dcf0b12245cbae5L, 0x6a01673fd8b513dL},
            {0x2117190446b50f9dL, 0x8b48030b85d083abL, 0x68a79e0698163241L, 0x716bb879de7fba1dL, 0xa388cdfa3c61ab73L, 0x411fada7499faf84L, 0x14ef89ce95bee4a3L},
            {0xf3f12b62f51a9b55L, 0x2866854cc1edef6cL, 0x1476fb5f05a37391L, 0xc40dceb74330b7b7L, 0xb84db83f7ccc5f9L, 0x9d0117552050ed7fL, 0x25b93de89e85456cL},
            {0x2ee01b9e2a7692a6L, 0x2a9920e8a8923beaL, 0xc6cfbcd09cc47583L, 0x80e53ddb1a7abd17L, 0xd585cba8c327e538L, 0xf4d313c03777336cL, 0x7d136bd0afbaf9dcL},
            {0x53ca5e2da19191b7L, 0xab8fba8b2c21655L, 0xede0ddbd23d66d73L, 0xce7dca3935c2fdc1L, 0xbd19a32205fcc165L, 0xa0a12ad442bce1f6L, 0x15eae05c9c6a0e03L},
            {0xce6d0917744faa2fL, 0x42d66b23a164a48aL, 0xecece7b7f05004fL, 0xe08c6a73a5559556L, 0x889313a96c9a1323L, 0x3253f1f2a3a66b1aL, 0xa2767544b53fee4eL},
            {0xf9b8ca6b46052208L, 0xd693bb3a5a89c965L, 0x6e975d1339c4774aL, 0xd72e8efd275e3db8L, 0xc471d5f3d0c9b1b6L, 0xc62c1d0fc7800ec2L, 0xacb7f73f9b5a1109L},
            {0xfb1cb91d94d6cddbL, 0xdfb1e3d6b795c8acL, 0xec4df9c3e90f1e24L, 0x6c487cdea2c507deL, 0x82d42d569ff4c955L, 0x7c3767d754e62f5fL, 0x3916c993f8cdca8cL},
            {0xa39e2eab5f174f15L, 0x9f2a05a63d015c7bL, 0xc2e46128224f249bL, 0x3d2b520d3144119bL, 0xc8ade99b9d8d5092L, 0x65882e9c99a0ed12L, 0x42a89a2be41263d9L},
            {0xe9bfc7e088623326L, 0xa79c980a7458736fL, 0xe7a09424c5bd6f77L, 0xd623ef8d9e4750ddL, 0xb329a5d0ce2c4320L, 0x724fc6ee18c04a2fL, 0x6f288c76ecde63bbL},
            {0x24d3561ce4eda075L, 0x74fdf369a4ba7bd8L, 0xc979f4ef12661fbdL, 0xb2e2bf501c9bd4eeL, 0xf66a2607c4d22a24L, 0xb9709df0c8fa8889L, 0xf70db2a5a9e6f385L},
            {0x3edb299037e41adcL, 0x8e3327b45d22677fL, 0x85d3af0877d1b233L, 0x41a0a96292eebd12L, 0xd331d1a9960dd15eL, 0x45c06e443e3580efL, 0x8c9a4b60297b5822L},
            {0x4ccafed99120c34cL, 0xc061a2298aacd9eeL, 0xd8a2a419bbd61dd4L, 0xee848fe0fa5feec3L, 0xa1c2bbe4bea46de1L, 0xf72fee59825eaba4L, 0x1c6cbfcae94f761aL},
            {0x811039d76b0f5c10L, 0x81c01b119d95abfbL, 0xb9f230a525dd1a79L, 0x82a0d5833ef0fe08L, 0xbcab840f326aa717L, 0xed7f80003ad9c7a8L, 0xe1a7e9e27bfbb5ceL},
            {0xf26eca16e4f6b311L, 0x32720d4ea0a72e4L, 0x7e13d7dbee27de4eL, 0x5c37936e56cf7e46L, 0x295f982a83b30c99L, 0xb0241eb8061d0f95L, 0xe4dc70591f41cea4L},
            {0x8ce51e30cf1501bbL, 0x35e452a0a514fbf6L, 0xe12df99407eac10bL, 0x2a4a1228a520332aL, 0x7746e7c2193f936eL, 0x814a4661f92c5f06L, 0xdf8cbc1191bb982bL},
            {0x80d0fa7707773de4L, 0x4be9c7b015a574a9L, 0xd4cb1cb66a739318L, 0xdf8dc7766f988303L, 0x8188a46bb7a98536L, 0xd9fcbdd211e305cbL, 0x2c798285814ddf2eL},
            {0x698d6cc716818773L, 0x845fe2403582149L, 0x9502bc1422758522L, 0x3d59e8cf894ec921L, 0x5310828b8dbedfdbL, 0x238dce16320651ddL, 0x7b38b1f93ce8749bL},
            {0xcaaa5ff55032cbcfL, 0x538e9005d8665c92L, 0xe174f0f93d30f0bcL, 0x9ad16d0b0a5892beL, 0xa2ce93130b6539ebL, 0xe50402009848b944L, 0x5bf398fdd39286d5L},
            {0x3333d53faadbec42L, 0x3f8857090ee7798bL, 0x5c95401451994dacL, 0xca985ee7a329cd7eL, 0x76ecafcc948c9562L, 0x268ce4e1a2a5a074L, 0x21d353fba6630d78L},
            {0x10882aac3dd3587L, 0xe0963a96a791586fL, 0x2d2e1c962520b6deL, 0xb2a78656df8faaa1L, 0xddfae1420e3e858dL, 0x1912b4f86123a4d6L, 0x986e18713086add0L},
            {0xb11fde1059b22334L, 0xefc2d98538f4ecfcL, 0x36af0ce3f9940bdfL, 0xd7a26a8e9c020084L, 0xf5177c6dc6d8a5L, 0x6926948892e970c8L, 0x840b22073cf60998L},
            {0x8977ae72ed603d45L, 0x60f4ffd92231c25bL, 0xe2b1b66758d158fcL, 0x70caf8189b6e929bL, 0x1b80d6fcc87b4d5eL, 0x77ae1691bcc4bbeaL, 0x5c619855527e1200L},
            {0xf65b17f58e2f82f6L, 0x7b5e65ee7c85e9b9L, 0x5e28d4218467b771L, 0xc0a7d673c0ae5225L, 0x31b05cd187dce5fcL, 0x3bdaf6c1c3992de9L, 0xaf7d1ee6c8d8e3aeL},
            {0x63689bb426fad75L, 0xe65b123bfc973da6L, 0xdb08275d11847a43L, 0x92dc01e5daa6f8cfL, 0xe2c4e337f7e3c4bfL, 0x35339b7ca3a1be91L, 0x80a30021da4c2964L},
            {0xf09d687ab01da414L, 0x89e97db87314fab1L, 0x2c48cf28ff80b17fL, 0x9443d8392ae194e1L, 0x929e316b17083568L, 0x117dea737f1df80bL, 0x1fc14678b5adf5ffL},
            {0xf9946308ce8bcec0L, 0xcdfe313f59a7c805L, 0x43f83dac819e8271L, 0xf7ee9f4f36e9cfcbL, 0x16c3965ae72f209eL, 0x56f7bd99b0d467f3L, 0xc7bc0fdcc7a4f542L},
            {0x5f2a932916c5c63fL, 0xf4787134f14a7108L, 0xfba7efcc1e2629e6L, 0x9b0287c30033872dL, 0x7fe7cd37d8292591L, 0xa402891b4a428cbeL, 0x473f085727dea256L},
            {0x3a7933b10ff2e831L, 0x22ff143fefbbd3b9L, 0x2e552c66fb8678d2L, 0xf427ee42d5ee8003L, 0x1eebdcf751988c45L, 0xa4262cebab700e78L, 0xf3b14f368783f74L},
            {0x41f45d562a6689bL, 0xe23f0e34570f037cL, 0x990e39e880dc1533L, 0xbd8a072257c813e4L, 0x547537b7deeece82L, 0xf0a854abf63d7f2cL, 0x6ad10c54bbf5f37eL},
            {0xbcec7d59b5858e63L, 0xce27686675aca1b9L, 0x6dbc1f5cd79cec30L, 0x35792372c0a1f9ecL, 0xb5c79c04405b7d56L, 0x78be8b169f1d27ecL, 0xee96813ea6366da7L},
            {0x82ea92d6830c37adL, 0x8839b9de78d0ead5L, 0x15f84a79fe513c18L, 0xccd8ee9c81f0fd31L, 0xc052c7f03a00caf1L, 0xe685a7c5e2dfefd8L, 0x5d1e5b7cad442fc8L},
            {0x27cc4624e3a8fd6cL, 0xc2c8aae889151aa7L, 0xac3dce81ee301ebdL, 0x27d10cde7611dbf4L, 0xb1df219237f18451L, 0xfdd47fef8b61284bL, 0x529b0e44e4875fc5L},
            {0xbfa129745aeb3923L, 0x490ee8b72fb3248eL, 0x5daefa90bb67c95eL, 0xd08e3a551657f581L, 0x174c60b071a111d9L, 0xe42f0ffcf7a8a264L, 0x2dcd1114c8457f34L},
            {0x9b19fb3f08515329L, 0x85dea11f9278c39eL, 0x5d8dd543eb53b5c5L, 0x62792122b242fbb0L, 0x1339a529c030fb61L, 0xd95fd1afa65a5dedL, 0x8e6bffc81a339dd0L},
            {0xb944c2c819b2038dL, 0xfc4cf4ef53de6f83L, 0xbec7b4166eb32958L, 0x98560aae6f6c1e35L, 0xeea46b496f45722bL, 0x74a0e05c8ef8afb6L, 0x63efa8fb5359a688L},
            {0x6e8d2803df3b267aL, 0xb94287ee66ec3f05L, 0x4b19fa3db0bb8ae1L, 0xc72b486a73ddfdb2L, 0xf87aad46e1a788daL, 0x5fae4d0974a5384eL, 0xd7864668291c713dL},
            {0xa5ed64048af45d9dL, 0x7b48feba418052fbL, 0x7a4a22451f57afc2L, 0x6ca5a10bb6dde0cfL, 0x916d9f2c62b33970L, 0xa7005fe34edfbc94L, 0x14ed78a60e348f2L},
            {0x6d56acb61a9abe8eL, 0xfdb4477c368483a0L, 0x352075394f788b74L, 0x4f626288a601b303L, 0xfa445e36c5fc1bd8L, 0x487f76509190057eL, 0xc0c2ec27c850d93fL},
            {0x4f03f6750128b16fL, 0xc2307541c9970984L, 0xeba0e38bdfb2f415L, 0x844490b6c94ff01dL, 0x97d2f6acf11431c5L, 0x31e7de47b6d4d6adL, 0xfcab3de1e8f50d67L},
            {0x6e717510c8e732c4L, 0xd3e508a9e3855fabL, 0x18e0ba1d43b19fe8L, 0x848ad83b6e0d60fL, 0xc09d282f51da855dL, 0x82e7f74688a014edL, 0xd3fb00a2bfb9f821L},
            {0x6167f57448c6559bL, 0x1bdac82270ba5dafL, 0x99414798c789a18bL, 0x5ca3088d1d613904L, 0xb514e80fa4bd6173L, 0xc7f333680ad450aL, 0xba41a35ce36b4fddL},
            {0x4c445bb3cc5dc033L, 0x2f468b3f6e9ccf43L, 0x60e78440ab5f7a3fL, 0xc0a78fbbb4d9c7feL, 0x5d5372b2750b6a97L, 0x9fc2a31931008d5aL, 0xd236ea6530b29183L},
            {0x3d63ec327c84a0bfL, 0x40dd9339cd2e68ffL, 0xf0ba798fa143e548L, 0x3ad5fe46fea96c61L, 0x5b78b66fef8dea6fL, 0x686332310340452dL, 0xa826cb8d2394f95fL},
            {0xeab5f4a8d3ec6334L, 0xa7e1065573315d35L, 0x3381e6aeaa8906cdL, 0xbb2fca5617f2c8c8L, 0xa68ae975813669dL, 0x21eb53b81a1608d7L, 0x96a0a8cef0ab1adcL},
            {0x1ffad87ddc8ca76aL, 0xe6b143d6ed7f42a0L, 0x51fc65a5f15337a2L, 0x281b10815ee6b36cL, 0x131460a7e307fb49L, 0x323ac05bb6f260fL, 0x86504e553eeeb51bL},
            {0xfcc3b1db7bb174a0L, 0x61bea0ed7dc160e8L, 0x7882e4ab6c8cb280L, 0xab914bed4e97e8d2L, 0x6071c5a779cc97fcL, 0xe5472aa9a23a7d31L, 0x628dea5a3164c608L},
            {0xcffe79062bb4e7cdL, 0x9e01b50f95301ea2L, 0x2a616a3eb9110b32L, 0x325894413570e9b0L, 0x1ba9bdb939ee6d9bL, 0x6a7c5f758b0f8a22L, 0x5613c8af1381df60L},
            {0xa21717e2b3d282eeL, 0x1ac0595d4f40cda4L, 0x429a8a47cea11c02L, 0xe680b930b66396edL, 0xd017b03635aece79L, 0xb5547e06d64d2394L, 0xcd8ee2c6d0f48658L},
            {0x7e4143da4d878be5L, 0xde837a44b83df910L, 0xc5096fd7847216cL, 0xd1ffc6a0e63da251L, 0xfc0b63ceaef917c0L, 0xac5020cdbb7db31aL, 0xf600187306f3f361L},
            {0x23b80b8bc4e75405L, 0x29de38e234555045L, 0xb58684b753996de8L, 0x7be7ad6fb131552L, 0x4c4e822573890072L, 0x1b1bffc34b326bfeL, 0xd4ca629a7c07f772L},
            {0xa6ae749a1ed10838L, 0xa9c706bcdcfa891eL, 0xde2661c018abc48cL, 0x97ad0eeccf7beed9L, 0xbf70a97987134ce1L, 0xb0988e5caa387cb2L, 0xc762a4b2a2126a63L},
            {0xd4b4a81be36638f2L, 0x92aabc9931541a5aL, 0x43ab0147e976e855L, 0xc825b61dceb4e636L, 0xe0809d70e918ada7L, 0xc35b1dca85adbeaL, 0xf77b1cd8381a85a9L},
            {0x5bab2890f354896dL, 0xa9d1f89bd9868ddL, 0x3dcc900485630f0fL, 0x4f22b5392f0b094eL, 0xb0f6c85f71e717a6L, 0x46957a3b2d65a038L, 0xb083716110d971b4L},
            {0x4c0a184632b0499aL, 0xe9c2e26d5bd7346L, 0x3e4fd6dfe99c67d2L, 0x84adabbc4885d2cL, 0x7f13d1c57c1436eeL, 0x816789354e143b64L, 0x3df2247f878cc4a9L},
            {0xb45a39714746ec86L, 0x96f58143107477acL, 0x2dd11909380bb2cdL, 0xd91b8bb8672fd8f5L, 0x740fccac7b4f751bL, 0x30dece8f93a98d22L, 0x4dfc62e32800ede8L},
            {0xc4b90839e91abfb2L, 0xe82891efce710c00L, 0x4a7ed592a3a82dd4L, 0x7e607fe600517cd0L, 0xe532f493827b0237L, 0x395cda8e4fe45809L, 0x54c07a612f99b802L},
            {0xe81d35c8ed7827feL, 0x1a262c26a7b07276L, 0x207dc323cb840325L, 0xc248f06ca75157a0L, 0xd89f50212f3ce653L, 0xbe6f8171f28a86d0L, 0x429a914e8bcd778eL},
            {0x587c5ee43e034ebdL, 0x17d007f9bc666c2dL, 0xaf9c82c94dfcda1fL, 0x2ca7857ac0ec7867L, 0x4c7405c3f345264cL, 0x7a5fcd4b620e0939L, 0xfac9f4b7677b447fL},
            {0xb1ec87f8823040acL, 0xa7b6d2e2223d8bb0L, 0xd41677026942ade4L, 0xac5aadb9c48b988dL, 0x850ad5a0d3650159L, 0xe6f67795d6a04567L, 0x5f9ba2bbfa36e575L},
            {0x7677dff12f92fbd9L, 0xa1c9bc7d32f35ca1L, 0xedcd974aa7488258L, 0xe38fa487026a5a0bL, 0x3b9b7540bf4802a5L, 0xf3118d8cf9507c02L, 0x6e8147b6eab1fe87L},
            {0xb69cea6e5a0e28fdL, 0x3e24ba0592afe2eL, 0x57dedae1b68ddd05L, 0x6e5f6aded4efd69fL, 0x41aaf253fd433093L, 0x811156ade1688bfbL, 0xf6be0584b63c47f3L},
            {0xf7180ae2e0f325e5L, 0xd853c886b1187cb8L, 0x500adca11f8e94deL, 0xf3c31d687579578fL, 0x2c3f467e63c5225fL, 0x5b92432e0e17d7aaL, 0xac7174bf58f98decL},
            {0xa08d214869e84ccfL, 0x12011850607c0dc0L, 0x6ca21154a2193f1eL, 0x7941a5bb3a6fb8e0L, 0x31785fae4ea5cbd3L, 0x1614b3e8ba5368d0L, 0x7f23439c6eced206L},
            {0xcfff666740e2f99fL, 0x434be89d9bd1b14L, 0xd12ba6e50904b61fL, 0xeaac07a345101168L, 0x3ab8690ec91d6ed0L, 0xa3e4855a45efe602L, 0x34c0bbd016958641L},
            {0x2fc743551c71634eL, 0x5a39844593357e83L, 0x5ae234ee018ecf0bL, 0x4b0b4fda75dafd7aL, 0x1dbc42dfe207f246L, 0x5a0d608f0148d695L, 0x3fea6c290b1dd217L},
            {0x9bf4d77b464c9435L, 0x3b3e99dd8dbe902dL, 0xbad6a2d23cb69f3fL, 0x694a09a072fb55d0L, 0x554f4bef9ec86462L, 0x62b67efddc71a65eL, 0x742a639df5c8e97fL},
            {0x5e6b758083214c84L, 0xcb50a5ebbc51fe29L, 0x4a8d4e03e37c386aL, 0xc39c9c99c57f6ca5L, 0xb42e5c0f6facbc5dL, 0xb61e56112182b0bdL, 0x1e47e5eae9d39cc1L},
            {0x40548138ef68aa78L, 0x5ba2ddd6b6414839L, 0x9b08bb741da55929L, 0xd1a6d32fe1544ce7L, 0x2116dc3d7c295fd2L, 0xda9bf65c101dfb2bL, 0x28e657eb1e7ec91bL},
            {0x7c6b73ef50249070L, 0x253fa259c87d5773L, 0x6d3fd907de23f5eeL, 0x48293bc212b01988L, 0xbc97363f6593b1c6L, 0xeb58e37ccf58a370L, 0x43958d7b7ea6e3eaL},
            {0x462a1dc5b9cb1b3bL, 0x7904069a5fa90c8bL, 0x6b34023aa308a14dL, 0x30ca8d4223d71e5cL, 0x265806e3aa04ad6eL, 0x99003da804259b22L, 0x852d586b40d5fdc9L},
            {0xb8b156aa6c884b21L, 0x6320356e9c8220ebL, 0xdab9fa981067044fL, 0xd87813c08592d18cL, 0xb382d881f2c03851L, 0x984c34fa745f8617L, 0xe83b77ea9b8cb55L},
            {0xc7afcc722488f9e6L, 0xad6f0985312a64a6L, 0x97c923f4604fdcf4L, 0x9ea08f9e3a9dcd7aL, 0xc43489710d913809L, 0xd93c98c6519cbb12L, 0x440d5b0518ebbba7L},
            {0x7a45b5b10dc24dbcL, 0x954a638cab780f0aL, 0x7f807ade405b7144L, 0xa06a1ff81f995ca6L, 0x3f86a498bc53f3fbL, 0x5754b5a8eaa5f9aeL, 0xc9525aa3857aeac2L},
            {0xefe499d7a567391dL, 0xcc2e34ce91112f84L, 0xdb5b75c40a8d6871L, 0x9baddec72034af7cL, 0x76093c1151a9c334L, 0x87e40485b73a54daL, 0x60016735dade9c79L},
            {0xb60d26b461d05e25L, 0x85e44ec2d4c15b5fL, 0xa33941259a71f695L, 0xdb65a2f99f10b462L, 0x3a8a521b95aa4061L, 0x926cb95f25d21bf0L, 0xbd55cebbdeb0995bL},
            {0xc15d366b98d92986L, 0x55bf21d6d14af29cL, 0x5dc43a61cfcdab12L, 0xa0f0e3426fd64509L, 0xc83035f88a0b77efL, 0x2763e28e263f5dffL, 0xc3ec793cc63dd03bL},
            {0x9addb551a523df05L, 0xe60921a3a4aceffdL, 0x80019fdf97a7e7afL, 0x45b80999148290a8L, 0xa28692320b415d8eL, 0x2bb4caa23d8fc335L, 0x886ba29327b7d888L},
            {0xbd0a37a2ad2465b9L, 0x27f0829d19cd8cf2L, 0xf13fdd1a783dea5bL, 0xd865088b9158ab0dL, 0x8dce24a60fd399dL, 0x3c1c449046cf6093L, 0xc868ae36645aa748L},
            {0xe7a7162d930c5056L, 0xcca8d5eac07d1880L, 0x98d7152acf6aead9L, 0xd120fb01533ea3dbL, 0x87ba57b866606483L, 0xd1ab19f981be7f77L, 0x575d2739539d89daL},
            {0xb9982c5395b09406L, 0xae9457448497d448L, 0xccb79281b0518ea2L, 0x750ad5690f9e2ea4L, 0x480ea42b71c98703L, 0x183a69ca58d0194bL, 0x38d9e1c3079391c3L},
            {0xe41766d004eef8fdL, 0xaadcffc4079be65L, 0x1762a40c971c5256L, 0x6c0dca72f7dfd702L, 0xb72cd78f733b3838L, 0x98fd59dc49501bb7L, 0xd64b0dfa56080086L},
            {0xa3074a96c88c47deL, 0x9dbfa7ee06c6d629L, 0x7a3010c27a54bf5bL, 0x1a646f82ee865034L, 0x1cd75ba2ba2e18dbL, 0x7886c70238c13919L, 0xa375790fbc0d4075L},
            {0x881caa3913271394L, 0x914f7f9497f7ba94L, 0x76ac64366b6f5660L, 0x61f8efb455b1df84L, 0x1c7d6d187eaa165bL, 0xf4037a9c1b15a760L, 0xaf6952534fbb79b4L},
            {0x77d95a600f824230L, 0xe5a39f40ea41304fL, 0x7547f8e9b3d7d3bcL, 0x2ae7e1bb1086a0c9L, 0x9db29c86d65743b9L, 0x3499bfb7b01c552L, 0x44bc45426b974a12L},
            {0x1984adb7bcfec495L, 0xb07d8992ffb8817eL, 0xc4d5f5af08537d31L, 0xffb697b8cfa03374L, 0x416e215f645d8cbL, 0xf7a9783188157a80L, 0xfe0671c77addf1faL},
            {0x66f613698d2263a7L, 0x2d5e27113b032421L, 0x50723eb3c45bba59L, 0x19fa48781ce2b326L, 0xf34fc200e9ca457cL, 0xaa074b3b25a4f79L, 0xb030f76a048d3c4eL},
            {0x50cf2a1c284f5a5aL, 0x2caca2361a3ba2e0L, 0x3762f19bdf869c75L, 0xfc34a738dc6b2cfL, 0x1a72ebef9f3084a1L, 0xe80baa530c593464L, 0x745f6c0e1f21e62fL}
   };
}