package ru.yandex.webmaster3.internal.common.security;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.internal.common.security.dao.InternalApiClientsYDao;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;

/**
 * @author avhaliullin
 */
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Component
public class InternalACLService {
    private static final long CACHE_EXPIRATION_SECONDS = 10;

    private final InternalApiClientsYDao internalApiClientsYDao;
    private final LoadingCache<Integer, Optional<InternalClient>> cache = CacheBuilder.newBuilder()
            .expireAfterWrite(CACHE_EXPIRATION_SECONDS, TimeUnit.SECONDS)
            .build(CacheLoader.from(this::getClientInfoFromDB));

    private Optional<InternalClient> getClientInfoFromDB(int clientId) {
        try {
            return Optional.ofNullable(internalApiClientsYDao.getClient(clientId));
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to get client info for client " + clientId,
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    public Optional<InternalClient> getClientInfo(int clientId) {
        return cache.getUnchecked(clientId);
    }
}
