package ru.yandex.webmaster3.internal.feeds;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedInfo2;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedSccStatus;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.internal.common.InternalAction;
import ru.yandex.webmaster3.internal.common.InternalRequest;
import ru.yandex.webmaster3.internal.common.security.ActionInternalGrant;
import ru.yandex.webmaster3.internal.common.security.InternalGrant;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.feeds.FeedsService;

/**
 * Created by kravchenko99 on 01/10/2020.
 */
@Slf4j
@WriteAction
@Category("feeds")
@Description("Результат проверки скк для фидов Универсального Поиска")
@Component("/feeds/unisearch/sccResult/set")
@ActionInternalGrant(InternalGrant.FEEDS)
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class SetUnisearchFeedSccResultAction extends
        InternalAction<SetUnisearchFeedSccResultAction.Request, SetUnisearchFeedSccResultAction.Response> {

    private final FeedsNative2YDao feedsNative2YDao;
    private final FeedsService feedsService;
    private static final String OK = "ok";
    private static final String WARN = "warning";

    @Override
    public Response process(Request request) {
        NativeFeedInfo2 feed = feedsNative2YDao.get(request.domain, request.feedUrl);
        if (feed == null) {
            log.warn("Feed with key {} {} doesn't exists", request.domain, request.feedUrl);
            return new Response(WARN, "no feed with domain - " + request.domain + " , feed_url - " + request.feedUrl);
        }
        NativeFeedInfo2.NativeFeedInfo2Builder builder =
                feed.toBuilder()
                        .sccFinishTimestamp(DateTime.now());

        if (feed.getBusinessId() != null) {
            log.warn("Feed with key {} {} on mbi pipeline", request.domain, request.feedUrl);
            return new Response(WARN, "This feed on mbi pipeline: domain - " + request.domain + " , feed_url - " + request.feedUrl);
        }
        if (request.isCorrect() && request.isCorrectType()) {
            log.info("Feed with key {} {} is correct", request.domain, request.feedUrl);
            feedsNative2YDao.update(builder.statusScc(NativeFeedSccStatus.SUCCESS).build());
            return new Response(OK);
        }
        var comment = Optional.of(request).map(Request::getReasons).map(Reason::getComment).orElse("");
        if (request.isCorrect() && !request.isCorrectType()) {
            log.info("Feed with key {} {} has incorrectType with comment - {}", request.domain, request.feedUrl, comment);
            builder.statusScc(NativeFeedSccStatus.FAILED);
            builder.errorsScc(List.of("Неверный тип фида. " + comment));
            feedsNative2YDao.update(builder.build());
            if (feed.getStatusScc() != NativeFeedSccStatus.FAILED) {
                feedsService.sendFailedSccMessage(request.domain, request.feedUrl);
            }
            return new Response(OK);
        }
        if (request.getReasons() == null) {
            log.error("Feed with key {} {} is incorrect, but haven't reasons", request.domain, request.feedUrl);
            throw new IllegalStateException("Expected reasons");
        }

        if (request.reasons.noPlacement != null) {
            log.info("Feed with key {} {} was banned with status - {} and comment - {}", request.domain,
                    request.feedUrl, request.reasons.noPlacement, comment);
            builder.statusScc(NativeFeedSccStatus.BANNED);
            builder.errorsScc(List.of(comment));
            feedsNative2YDao.update(builder.build());
            if (feed.getStatusScc() != NativeFeedSccStatus.BANNED) {
                feedsService.sendBannedSccMessage(request.domain, request.feedUrl);
            }
            return new Response(OK);
        }

        // завалили весь фид
        if (request.reasons.isInvalid()) {
            log.info("Feed with key {} {} was failed with comment - {}", request.domain,
                    request.feedUrl, comment);
            builder.statusScc(NativeFeedSccStatus.FAILED);
            builder.errorsScc(List.of(comment));
            feedsNative2YDao.update(builder.build());

            if (feed.getStatusScc() != NativeFeedSccStatus.FAILED) {
                feedsService.sendFailedSccMessage(request.domain, request.feedUrl);
            }
            return new Response(OK);
        }

        // завалили оферы
        log.info("Feed with key {} {} has failed offers", request.domain, request.feedUrl);
        builder.statusScc(NativeFeedSccStatus.FAILED);
        List<String> comments = request.reasons.incorrectOffers.stream()
                .map(x -> "Офер с id - " + x.getOfferId() + " и url - " + x.getOfferUrl() + " не прошел проверку со " +
                        "статусом: " + x.getComment())
                .limit(5)
                .peek(log::debug)
                .collect(Collectors.toList());
        builder.errorsScc(comments);
        feedsNative2YDao.update(builder.build());

        if (feed.getStatusScc() != NativeFeedSccStatus.FAILED) {
            feedsService.sendFailedSccMessage(request.domain, request.feedUrl);
        }

        return new Response(OK);
    }

    @lombok.Value
    public static class Reason {
        String comment;
        @Description("Ошибки оферов")
        List<Offer> incorrectOffers;

        @Description("Фид является не валидным")
        boolean invalid;

        @Description("Статус бана")
        String noPlacement;


        @lombok.Value
        public static class Offer {
            @Description("Комент про проблему офера")
            String comment;
            String offerId;
            String offerUrl;
        }
    }

    @Getter
    @Setter(onMethod_ = @RequestQueryProperty)
    public static final class Request extends InternalRequest {
        @Description("Домен, то что передали из Вебмастера")
        @Setter(onMethod_ = @RequestQueryProperty(required = true))
        String domain;
        @Description("Урл фида")
        @Setter(onMethod_ = @RequestQueryProperty(required = true))
        String feedUrl;
        @Description("Есть ли ошибки")
        @Setter(onMethod_ = @RequestQueryProperty(required = true))
        boolean isCorrect = false;
        @Description("если feedType==realFeedType")
        boolean isCorrectType = true;
        @Description("Причины , только если not isCorrect || not isCorrectType")
        Reason reasons;
        @Description("Тип фида, который указал пользователь")
        String feedType;
        @Description("Тип фида настоящий")
        String realFeedType;
        @Description("Когда началась проверка из Вебмастера поле")
        DateTime created;
        @Description("Когда закончилась проверка")
        DateTime finished;
    }

    @Value
    public static class Response implements ActionResponse.NormalResponse {
        String status;
        String comment;

        public Response(String status) {
            this(status, null);
        }

        public Response(String status, String comment) {
            this.status = status;
            this.comment = comment;
        }
    }

}
