package ru.yandex.webmaster3.internal.user.domain;

import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.ToIntFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import lombok.Setter;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.host.verification.UserHostVerificationInfo;
import ru.yandex.webmaster3.core.host.verification.VerificationType;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.storage.host.service.MirrorService2;
import ru.yandex.webmaster3.storage.user.UserUnverifiedHost;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;

/**
 * @author avhaliullin
 */
@Setter
@Service
public class DomainVerificationHelperService {
    private MirrorService2 mirrorService2;
    private UserHostsService userHostsService;

    public DomainVerificationState getDomainVerificationState(long userId, String domain) {
        Set<WebmasterHostId> hostsForDomain = new HashSet<>(IdUtils.allHostsForDomain(domain));
        List<UserVerifiedHost> domainVerifiedHosts = userHostsService.getVerifiedHosts(new WebmasterUser(userId),
                hostsForDomain);
        Map<WebmasterHostId, UserHostVerificationInfo> verificationRecords = new HashMap<>();
        for (WebmasterHostId hostId : hostsForDomain) {
            UserHostVerificationInfo verificationRecord = userHostsService.getVerificationInfo(userId, hostId);
            if (verificationRecord != null) {
                verificationRecords.put(hostId, verificationRecord);
            }
        }

        Map<WebmasterHostId, UserVerifiedHost> domainVerifiedHostMap = domainVerifiedHosts.stream()
                .collect(Collectors.toMap(UserVerifiedHost::getWebmasterHostId, Function.identity()));
        Map<WebmasterHostId, UserUnverifiedHost> domainUnverifiedHostMap = hostsForDomain.stream()
                .flatMap(hostId -> {
                    if (!domainVerifiedHostMap.containsKey(hostId)) {
                        UserHostVerificationInfo verificationInfo = verificationRecords.get(hostId);
                        if (verificationInfo != null && verificationInfo.isAddedToList()) {
                            UserUnverifiedHost unverifiedHost = new UserUnverifiedHost(
                                    hostId, verificationInfo.getVerificationUin(), verificationInfo.getRecordId());
                            return Stream.of(Pair.of(hostId, unverifiedHost));
                        }
                    }
                    return Stream.empty();
                }).collect(W3Collectors.toHashMap());

        WebmasterHostId preferredHost = selectPreferredHost(
                hostsForDomain,
                domainVerifiedHostMap.keySet(),
                domainUnverifiedHostMap.keySet(),
                verificationRecords);

        return new DomainVerificationState(hostsForDomain, domainUnverifiedHostMap, domainVerifiedHostMap,
                verificationRecords, preferredHost);
    }

    private WebmasterHostId selectPreferredHost(
            Set<WebmasterHostId> hostsForDomain,
            Set<WebmasterHostId> verifiedHosts,
            Set<WebmasterHostId> unverifiedHosts,
            Map<WebmasterHostId, UserHostVerificationInfo> verificationRecords) {

        if (!verifiedHosts.isEmpty()) {
            if (verifiedHosts.size() == 1) {
                return verifiedHosts.iterator().next();
            }
            return verifiedHosts.stream().min(
                    Comparator.comparingInt(mirrorsScorer(verifiedHosts))
                            .thenComparingInt((WebmasterHostId h) -> h.getSchema() == WebmasterHostId.Schema.HTTPS ?
                                    0 : 1) // хост уже подтвержден. При этом https с большей вероятностью является
                            // основным
                            .thenComparingInt(DomainVerificationHelperService::scoreWww)
            ).get();
        }

        // Если есть PDD_EMU верификация в процессе, то берем preferred host оттуда
        var pddEmuRecordOpt = verificationRecords.values().stream()
                .filter(info -> info.getVerificationType() == VerificationType.PDD_EMU).findAny();
        if (pddEmuRecordOpt.isPresent()) {
            return pddEmuRecordOpt.get().getHostId();
        }

        return hostsForDomain.stream()
                .distinct()
                .min(
                        Comparator.comparingInt(mirrorsScorer(hostsForDomain))
                                .thenComparingInt(h -> unverifiedHosts.contains(h) ? 0 : 1)
                                .thenComparingInt((WebmasterHostId h) -> h.getSchema() == WebmasterHostId.Schema.HTTPS ? 1 : 0) // хост не подтвержден. http версия у него почти точно есть, хотя бы редиректящая, а https - может вообще не существовать
                                // приоритет у этой проверки ниже, чем у проверки http/https, потому что редирект
                                // с/на www настроить проще, чем https
                                .thenComparingInt(DomainVerificationHelperService::scoreWww)
                ).get();
    }

    private ToIntFunction<WebmasterHostId> mirrorsScorer(Collection<WebmasterHostId> hostIds) {
        Map<WebmasterHostId, Integer> scores = new HashMap<>();
        mirrorService2.getMainMirrorsFromAllMirrors(hostIds).values().forEach(host -> scores.put(host, scores.getOrDefault(host, 0) - 1));
        return host -> scores.getOrDefault(host, 0);
    }

    private static int scoreWww(WebmasterHostId hostId) {
        return hostId.getASCIIHostname().startsWith("www.") ? 1 : 0;
    }
}
