package ru.yandex.webmaster3.internal.user.domain.verification;

import com.datastax.driver.core.utils.UUIDs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.events2.HostEventId;
import ru.yandex.webmaster3.core.host.verification.UserHostVerificationInfo;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.http.request.RequestId;
import ru.yandex.webmaster3.internal.common.InternalAction;
import ru.yandex.webmaster3.internal.common.security.InternalGrant;
import ru.yandex.webmaster3.internal.user.domain.DomainVerificationState;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.verification.HostVerifierService;

/**
 * @author avhaliullin
 */
@Description("Добавить хост")
@WriteAction
public class AddDomainAction extends InternalAction<AddDomainRequest, AddDomainResponse> {
    private static final Logger log = LoggerFactory.getLogger(AddDomainAction.class);

    private final HostVerifierService hostVerifierService;

    @Autowired
    public AddDomainAction(HostVerifierService hostVerifierService) {
        super(InternalGrant.DOMAIN_VERIFICATION);
        this.hostVerifierService = hostVerifierService;
    }

    @Override
    public AddDomainResponse process(AddDomainRequest request) throws WebmasterException {
        WebmasterUser user = new WebmasterUser(request.getUserId());
        DomainVerificationState verificationState = request.getDomainVerificationState();
        WebmasterHostId preferredHost = verificationState.getPreferredHost();
        try {
            if (verificationState.isAdded(preferredHost)) {
                return new AddDomainResponse.DomainAlreadyAddedError(preferredHost, request.getDomain(), false);
            }

            UserHostVerificationInfo verificationInfo = verificationState.getVerificationRecords().get(preferredHost);
            HostEventId eventId = HostEventId.createFromUserRequest(
                    new RequestId.FakeRequestId(UUIDs.timeBased()),
                    request.getUserId()
            );
            boolean autoVerified = hostVerifierService.addHostUinInitRecord(user, preferredHost, verificationInfo, eventId);

            return new AddDomainResponse.NormalResponse(preferredHost, request.getDomain(), autoVerified);
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to add host",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }
}
